"""在虚拟指标中间插入一个 Unitary 和它的 Hermitian 不会改变全局张量"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../../4-2.ipynb.

# %% auto 0
__all__ = ['MPS']

# %% ../../4-2.ipynb 2
import torch
from typing import List, Tuple, Literal
from .functional import gen_random_mps_tensors, MPSType

# %% ../../4-2.ipynb 12
from tensor_network.mps.functional import (
    orthogonalize_arange,
    calc_global_tensor_by_tensordot,
    calculate_mps_norm_factors,
    calc_inner_product,
)
import sys
from einops import einsum


class MPS:
    def __init__(
        self,
        *,
        mps_tensors: List[torch.Tensor] | None = None,
        length: int | None = None,
        physical_dim: int | None = None,
        virtual_dim: int | None = None,
        mps_type: MPSType | None = None,
        dtype: torch.dtype | None = None,
        device: torch.device | None = None,
        requires_grad: bool | None = None,
    ) -> None:
        if mps_tensors is None:
            assert (
                length is not None
                and physical_dim is not None
                and virtual_dim is not None
                and mps_type is not None
                and dtype is not None
                and device is not None
                and requires_grad is not None
            ), (
                f"mps_tensors is None, so all arguments must be provided, but got {mps_tensors=}, {length=}, {physical_dim=}, {virtual_dim=}, {mps_type=}, {dtype=}, {device=}, {requires_grad=}"
            )
            mps_tensors = gen_random_mps_tensors(
                length, physical_dim, virtual_dim, mps_type, dtype, device
            )
            for i in range(len(mps_tensors)):
                mps_tensors[i].requires_grad = requires_grad
            self._length: int = length
            self._physical_dim: int = physical_dim
            self._virtual_dim: int = virtual_dim
            self._mps_type: MPSType = mps_type
            self._dtype: torch.dtype = dtype
            self._device: torch.device = device
        else:
            # TODO: checking whether the mps_tensors is valid, not emergent
            self._length: int = len(mps_tensors)
            self._physical_dim: int = mps_tensors[0].shape[1]
            self._virtual_dim: int = mps_tensors[0].shape[2]
            self._mps_type: MPSType = (
                MPSType.Open if mps_tensors[0].shape[0] == 1 else MPSType.Periodic
            )
            self._dtype: torch.dtype = mps_tensors[0].dtype
            self._device: torch.device = mps_tensors[0].device
            requires_grad = mps_tensors[0].requires_grad if requires_grad is None else requires_grad

        self._requires_grad: bool = requires_grad
        self._mps: List[torch.Tensor] = mps_tensors
        self._center: int | None = None

    def center_orthogonalization_(
        self, center: int, mode: Literal["svd", "qr"], truncate_dim: int | None = None
    ):
        """
        Perform center orthogonalization on the MPS. This is an in-place operation.

        Args:
            center: int, the center of the MPS.
            mode: Literal["svd", "qr"], the mode of orthogonalization.
            truncate_dim: int | None, the dimension to be truncated. If None, no truncation will be performed.
        """
        assert -self.length <= center < self.length, "center out of range"
        if center < 0:
            center = self.length + center
        if self._center is None:
            new_local_tensors = orthogonalize_arange(self._mps, 0, center, mode, truncate_dim)
            new_local_tensors = orthogonalize_arange(
                new_local_tensors, self.length - 1, center, mode, truncate_dim
            )
            for i in range(self.length):
                self._mps[i] = new_local_tensors[i]
        elif self.center != center:
            new_local_tensors, changed_indices = orthogonalize_arange(
                self._mps, self.center, center, mode, truncate_dim, return_changed=True
            )
            for changed_idx in changed_indices:
                self._mps[changed_idx] = new_local_tensors[changed_idx]
        else:
            # when self.center == center
            pass
        self._center = center

    def force_set_local_tensor_(self, i: int, value: torch.Tensor):
        """
        Force set the local tensor at index i to the given value with checking the shape and dtype.

        Args:
            i: int, the index of the local tensor to be set.
            value: torch.Tensor, the value to be set.
        """
        value = value.to(dtype=self._dtype, device=self._device)
        value.requires_grad = self._requires_grad
        self._mps[i] = value

    def __getitem__(self, i: int) -> torch.Tensor:
        return self._mps[i]

    def __setitem__(self, i: int, value: torch.Tensor):
        local_tensor_shape = self[i].shape
        local_tensor_dtype = self[i].dtype
        assert value.shape == local_tensor_shape, (
            f"value shape must match local tensor shape {local_tensor_shape}, but got {value.shape}"
        )
        assert value.dtype == local_tensor_dtype, (
            f"value dtype must match local tensor dtype {local_tensor_dtype}, but got {value.dtype}"
        )
        self.force_set_local_tensor_(i, value)

    def global_tensor(self) -> torch.Tensor:
        """
        Calculate the global tensor of the MPS.
        """
        # use tensordot to contract the mps tensors, because it's faster than calc_global_tensor_by_contract
        if self.length > 15:
            print(
                "Warning: Calculating global tensor of MPS with length > 15, this may up all the memory",
                file=sys.stderr,
            )
        return calc_global_tensor_by_tensordot(self._mps)

    def norm_factors(self) -> torch.Tensor:
        """
        Calculate the norm factors of the MPS.
        """
        return calculate_mps_norm_factors(self._mps).real

    def norm(self, *, _efficient_mode: bool = True) -> torch.Tensor:
        """
        Calculate the norm of the MPS.
        """
        if _efficient_mode and self.center is not None:
            return self._mps[self.center].norm()
        else:
            norm_factors = self.norm_factors()
            # use sqrt inside the product to avoid overflow
            return torch.prod(norm_factors.sqrt())

    def normalize_(self, *, _efficient_mode: bool = True):
        """
        Normalize the MPS in-place.
        """
        if _efficient_mode and self.center is not None:
            self._mps[self.center] /= self.norm()
        else:
            norm_factors = 1 / self.norm_factors().sqrt()
            for i in range(self.length):
                self._mps[i] *= norm_factors[i]

    def inner_product(self, other: "MPS", return_product_factors: bool = False) -> torch.Tensor:
        """
        Calculate the inner product of two MPS. These two MPS must have the same length.
        """
        assert isinstance(other, MPS), "other must be a MPS"
        assert self.length == other.length, "length of two MPS must be the same"
        product_factors = calc_inner_product(self._mps, other._mps)
        if return_product_factors:
            return product_factors
        else:
            return torch.prod(product_factors)

    def check_orthogonality(self, *, check_mode: Literal["print", "assert"] = "print"):
        """
        Check the orthogonality of the MPS.
        """
        assert check_mode.lower() in ["print", "assert"], (
            "check_mode must be either 'print' or 'assert'"
        )
        print_mode = check_mode.lower() == "print"
        if self.center is None:
            print("center is None, so no orthogonality check can be performed")
        else:
            identity = torch.eye(
                2, dtype=self._dtype, device=self._device
            )  # cache for the identity matrix
            for i in range(self.length):
                if i == self.center:
                    if print_mode:
                        print(f"Local Tensor {i}: Center")
                else:
                    local_tensor = self._mps[i]
                    if i < self.center:
                        product = torch.einsum("abc,abd->cd", local_tensor.conj(), local_tensor)
                    else:
                        product = torch.einsum("xab,yab->xy", local_tensor, local_tensor.conj())

                    assert product.shape[0] == product.shape[1]

                    if identity.shape[0] != product.shape[0]:
                        identity = torch.eye(
                            product.shape[0], dtype=self._dtype, device=self._device
                        )

                    diff_norm = (product - identity).norm(p=1).item()

                    if print_mode:
                        print(f"Local Tensor {i}: {diff_norm}")
                    else:
                        assert diff_norm < 1e-6, f"Local Tensor {i} is not orthogonal"

    def to_(self, dtype: torch.dtype | None = None, device: torch.device | None = None) -> "MPS":
        if dtype is not None and self._dtype != dtype:
            for i in range(self.length):
                self._mps[i] = self._mps[i].to(dtype=dtype)
            self._dtype = dtype
        if device is not None and self._device != device:
            for i in range(self.length):
                self._mps[i] = self._mps[i].to(device=device)
            self._device = device
        return self

    def one_body_reduced_density_matrix(
        self, *, idx: int, inplace_mutation: bool = False
    ) -> torch.Tensor:
        assert 0 <= idx < self.length, "idx must be in [0, length - 1]"
        if self.center is None:  # TODO: optimize this branch
            # maybe we can just use einsum here, need some benchmarking
            if inplace_mutation:
                self.center_orthogonalization_(idx, "qr")
                return self.one_body_reduced_density_matrix(idx=idx)
            else:
                # do center orthogonalization out of place
                local_tensors = self.local_tensors
                length = len(local_tensors)
                center = idx
                mode = "qr"
                local_tensors = orthogonalize_arange(local_tensors, 0, center, mode)
                local_tensors = orthogonalize_arange(local_tensors, length - 1, center, mode)
                center_tensor = local_tensors[center]
        else:
            if self.center == idx:
                center_tensor = self.center_tensor
            else:  # TODO: optimize this branch
                if inplace_mutation:
                    self.center_orthogonalization_(idx, "qr")
                    return self.one_body_reduced_density_matrix(idx=idx)
                else:
                    # moving center out of place
                    local_tensors = self.local_tensors
                    new_center = idx
                    local_tensors = orthogonalize_arange(
                        local_tensors, self.center, new_center, mode="qr"
                    )
                    center_tensor = local_tensors[new_center]

        return einsum(
            center_tensor,
            center_tensor.conj(),
            "left mid right, left mid_conj right -> mid mid_conj",
        )

    @property
    def center_tensor(self) -> torch.Tensor | None:
        if self.center is None:
            return None
        else:
            return self._mps[self.center]

    @property
    def local_tensors(self) -> List[torch.Tensor]:
        return [i for i in self._mps]

    @property
    def length(self) -> int:
        return self._length

    @property
    def physical_dim(self) -> int:
        return self._physical_dim

    @property
    def virtual_dim(self) -> int:
        return self._virtual_dim

    @property
    def mps_type(self) -> MPSType:
        return self._mps_type

    @property
    def center(self) -> int | None:
        return self._center
