# AUTOGENERATED! DO NOT EDIT! File to edit: ../3-5.ipynb.

# %% auto 0
__all__ = ['cossin_feature_map', 'feature_map_to_qubit_state', 'linear_mapping']

# %% ../3-5.ipynb 15
import torch

# %% ../3-5.ipynb 16
def cossin_feature_map(
    samples: torch.Tensor, theta: float = 1.0, check_range: bool = True
) -> torch.Tensor:
    """
    Apply cossin feature mapping for qubit systems (d=2).

    See: https://proceedings.neurips.cc/paper/2016/hash/5314b9674c86e3f9d1ba25ef9bb32895-Abstract.html

    And also see https://www.perplexity.ai/search/do-you-know-the-cossin-feature-PidGvE9TTZSx9NaxAi23Gg

    Args:
        samples: Input tensor of shape (batch_size, feature_num)
        theta: Angle parameter (default: 1.0)

    Returns:
        Transformed features of shape (batch_size, feature_num, 2)
    """
    # Ensure samples is a 2D tensor
    if samples.ndimension() == 1:
        samples = samples.unsqueeze(0)  # Add batch dimension if missing

    if check_range:
        assert torch.all((samples >= 0.0) & (samples <= 1.0)), (
            "Samples should be between 0 and 1. This is usually required. To override this, set check_range=False."
        )

    # Calculate the cosine and sine components
    angle = samples * (theta * torch.pi)
    cos_features = torch.cos(angle)
    sin_features = torch.sin(angle)

    # Stack the features along a new dimension
    # Shape becomes (batch_size, feature_num, 2)
    stacked_features = torch.stack([cos_features, sin_features], dim=-1)

    return stacked_features

# %% ../3-6.ipynb 4
from einops import einsum


def feature_map_to_qubit_state(features: torch.Tensor) -> torch.Tensor:
    """
    Convert a feature tensor of shape (batch_size, feature_dim, 2) to a qubit state tensor of shape (batch_size, 2, ..., 2)
    """
    assert features.ndim == 3 and features.shape[2] == 2, (
        f"feature must be a 3D tensor of shape (batch_size, feature_dim, 2), but got {features.shape}"
    )
    feature_num = features.shape[1]
    features = torch.split(features, 1, dim=1)  # tuple of tensors of shape (batch_size, 1, 2)
    features = [
        feature.squeeze(1) for feature in features
    ]  # list of tensors of shape (batch_size, 2)
    einsum_expression = "{features} -> {tensor}".format(
        features=",".join([f"batch f{i}" for i in range(feature_num)]),
        tensor=" ".join(["batch"] + [f"f{i}" for i in range(feature_num)]),
    )
    qubit_states = einsum(*features, einsum_expression)  # (batch_size, 2, ..., 2)
    return qubit_states

# %% ../4-4.ipynb 8
def linear_mapping(samples: torch.Tensor) -> torch.Tensor:
    """
    Apply linear feature mapping

    Args:
        samples: Input tensor of shape (batch_size, *)

    Returns:
        Output tensor of shape (batch_size, *, 2)
    """
    if samples.ndim == 1:
        samples = samples.reshape(1, -1)

    return torch.stack([samples, 1 - samples], dim=-1)