"""例子：考虑在黑板上进行书写：如果要求仅在黑板的中间某个子区域进行书写，那么对于所有样本，该子区域之外的像素值不会发生变化（黑色），这个区间被称为背景。有用的信息（文字）出现在中间进行书写的子区域，我们将其称之为信息区。文字笔画对应的位置，则被称为关键性少数"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../../4-10.ipynb.

# %% auto 0
__all__ = ['dyn_OEE_analyze', 'OEE_variation_one_qubit_measurement']

# %% ../../4-10.ipynb 2
import torch
from ..feature_mapping import cossin_feature_map, feature_map_to_qubit_state
import numpy as np

# %% ../../4-10.ipynb 5
from ..quantum_state.functional import calc_onsite_entanglement_entropy, project_state


def dyn_OEE_analyze(samples: torch.Tensor, nth_img: int):
    """
    dynamic onsite entanglement entropy analysis
    """
    assert samples.ndim == 2, "samples must be a 2D tensor"  # (num_samples, num_features)
    assert samples.shape[1] >= 3
    num_samples, num_features = samples.shape
    features = cossin_feature_map(samples, theta=0.5)  # (num_samples, num_features, 2)
    tensor_states = feature_map_to_qubit_state(features)  # (num_samples, [2] * num_features)
    tensor_state = tensor_states.sum(dim=0) / np.sqrt(num_samples)

    oees = calc_onsite_entanglement_entropy(tensor_state)
    print(f"初始OEE = {oees}")
    total_oee = oees.sum()

    for feature_idx in range(num_features):
        print(
            f"根据第{feature_idx}个样本的特征取值 (x={samples[nth_img, feature_idx]})，投影测量第{feature_idx}个量子位"
        )
        projected_state = project_state(
            tensor_state, features[nth_img, feature_idx, :], feature_idx
        )
        new_oees = calc_onsite_entanglement_entropy(projected_state)
        oee_change = new_oees.sum() - total_oee
        print(f"\t 测量后，其余量子位OEE = {new_oees}")
        print(f"\t 测量前后OEE变化量 = {oee_change.item()}")
    print()

# %% ../../4-10.ipynb 9
from ..mps.modules import MPS
from tqdm.auto import tqdm
from typing import Dict, Any


def _remove_at(tensor: torch.Tensor, idx: int) -> torch.Tensor:
    return torch.cat([tensor[:idx], tensor[idx + 1 :]])


def OEE_variation_one_qubit_measurement(
    mps: MPS,
    feature: torch.Tensor,
    oee_threshold: float | None = None,
    progress_bar_kwargs: Dict[str, Any] = {},
) -> torch.Tensor:
    assert feature.ndim == 2, "features must be a 2D tensor"
    feature_num = feature.shape[0]
    oees = mps.entanglement_entropy_onsite_()
    if oee_threshold is None:
        selected_pos = torch.arange(feature_num)
        select_all = True
    else:
        assert oee_threshold > 0.0, "oee_eps must be a positive number"
        lt_oee_eps = oees < oee_threshold
        oees[lt_oee_eps] = 0.0
        selected_pos = torch.nonzero(~lt_oee_eps, as_tuple=True)[0]
        select_all = False

    oee_sum = oees.sum()
    zero = torch.zeros_like(oee_sum)
    oee_changes = []

    for idx in tqdm(range(feature_num), **progress_bar_kwargs):
        if idx in selected_pos:
            mps.center_orthogonalization_(idx, mode="qr", normalize=False)
            # project mps at idx
            projected_mps = mps.project_one_qubit(idx, feature[idx])
            projected_mps._center = max(0, idx - 1)
            if select_all:
                new_oees = projected_mps.entanglement_entropy_onsite_()
            else:
                new_selected_pos = _remove_at(selected_pos, idx)
                new_selected_pos[new_selected_pos > idx] -= 1
                new_oees = projected_mps.entanglement_entropy_onsite_(
                    indices=new_selected_pos.tolist()
                )
            oee_changes.append(oee_sum - new_oees.sum())
        else:
            oee_changes.append(zero)

    oee_changes = torch.stack(oee_changes)
    return oee_changes
