"""References:"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../../5-2.ipynb.

# %% auto 0
__all__ = ['evolve_gate_2body', 'direction_to_next_center', 'calculate_mps_local_energies', 'tebd']

# %% ../../5-2.ipynb 2
import torch
from einops import einsum, rearrange

# %% ../../5-2.ipynb 9
from typing import List
from ..mps.modules import MPS
from ..utils.mapping import view_gate_matrix_as_tensor, view_gate_tensor_as_matrix
from ..utils.checking import check_quantum_gate


def evolve_gate_2body(
    mps_local_tensors: List[torch.Tensor], gl: torch.Tensor, gr: torch.Tensor, p0: int, p1: int
):
    # gl - (a, g, c)
    # gr - (b, g, d)
    assert p0 < p1
    assert gl.shape[1] == gr.shape[1]
    g_dim = gl.shape[1]
    local_tensors = mps_local_tensors

    local_tensor_left = local_tensors[p0]
    local_tensor_left = einsum(
        local_tensor_left,
        gl,
        "left physical_c right, new_physical g physical_c -> left new_physical g right",
    )
    local_tensors[p0] = rearrange(
        local_tensor_left, "left new_physical g right -> left new_physical (g right)"
    )

    local_tensor_right = local_tensors[p1]
    local_tensor_right = einsum(
        local_tensor_right,
        gr,
        "left physical right, new_physical g physical -> g left new_physical right",
    )
    local_tensors[p1] = rearrange(
        local_tensor_right, "g left new_physical right -> (g left) new_physical right"
    )

    I = torch.eye(g_dim, dtype=torch.int32, device=mps_local_tensors[0].device)

    for idx in range(p0 + 1, p1):
        local_tensor_i = local_tensors[idx]
        new_local_tensor_i = einsum(
            I, local_tensor_i, "g0 g1, left physical right -> g0 left physical g1 right"
        )
        local_tensors[idx] = rearrange(
            new_local_tensor_i, "g0 left physical g1 right -> (g0 left) physical (g1 right)"
        )

    return local_tensors

# %% ../../5-2.ipynb 15
from typing import Literal


def direction_to_next_center(
    l0: int, r0: int, l1: int, r1: int
) -> Literal["right-to-left", "left-to-right"]:
    l_min = min([abs(l0 - l1), abs(l0 - r1)])
    r_min = min([abs(r0 - l1), abs(r0 - r1)])
    # TODO: check if this is correct
    if l_min < r_min:
        return "right-to-left"
    else:
        return "left-to-right"


def calculate_mps_local_energies(
    mps: MPS, hamiltonians: List[torch.Tensor], positions: List[List[int]] | torch.Tensor
) -> torch.Tensor:
    if len(hamiltonians) == 1:
        hamiltonians = hamiltonians * len(positions)

    assert len(hamiltonians) == len(positions), (
        f"len(hamiltonians): {len(hamiltonians)}, len(positions): {len(positions)}"
    )
    local_energies = []
    for pos, hamiltonian in zip(positions, hamiltonians):
        assert len(pos) == 2, "Only support 2-body interaction for now"
        rdm = mps.two_body_reduced_density_matrix_(pos[0], pos[1], return_matrix=True)
        local_energies.append(einsum(view_gate_tensor_as_matrix(hamiltonian), rdm, "a b, b a ->"))

    return torch.stack(local_energies)

# %% ../../5-2.ipynb 17
from ..mps.functional import orthogonalize_arange
from typing import Tuple
from tqdm.auto import tqdm
import tensor_network.setup_ref_code_import as _
from Library.MatrixProductState import MPS_tebd


def tebd(
    hamiltonians: torch.Tensor | List[torch.Tensor],
    positions: torch.Tensor | List[List[int]],
    mps: MPS,
    tau: float,
    iterations: int,
    calc_observation_iters: int,
    e0_eps: float,
    tau_min: float,
    least_iters_for_tau: int,
    max_virtual_dim: int,
    progress_bar_kwargs: dict = {},
) -> Tuple[MPS, torch.Tensor]:
    device = mps.device
    dtype = mps.dtype

    if isinstance(positions, List):
        positions = torch.tensor(positions, device=device, dtype=torch.long)
    else:
        assert isinstance(positions, torch.Tensor)
    assert torch.all(positions[:, 0] < positions[:, 1])

    assert positions.ndim == 2  # (interaction_num, gate_apply_qubit_num)
    interaction_num, gate_apply_qubit_num = positions.shape
    assert gate_apply_qubit_num == 2, "Only support 2-body interaction for now"
    if isinstance(hamiltonians, torch.Tensor):
        if hamiltonians.ndim == 2 * gate_apply_qubit_num:
            hamiltonians = [hamiltonians]
        elif hamiltonians.ndim == 2 * gate_apply_qubit_num + 1:
            assert hamiltonians.shape[0] == interaction_num
            hamiltonians = [hamiltonians[i] for i in range(interaction_num)]
        else:
            raise ValueError(f"Invalid hamiltonians shape: {hamiltonians.shape}")
    elif isinstance(hamiltonians, List):
        assert len(hamiltonians) == interaction_num
        for h in hamiltonians:
            assert h.ndim == 2 * gate_apply_qubit_num
    else:
        raise ValueError(f"Invalid hamiltonians type: {type(hamiltonians)}")

    hamiltonians = [h.to(device=device) for h in hamiltonians]
    for h in hamiltonians:
        check_quantum_gate(h, num_qubits=gate_apply_qubit_num)

    assert 1.0 > tau >= 0.0 and 1.0 > tau_min >= 0.0
    assert iterations >= 0 and calc_observation_iters >= 0 and least_iters_for_tau >= 1
    assert max_virtual_dim >= 1

    mps.center_orthogonalization_(
        positions[0, -1].item(), mode="svd", truncate_dim=max_virtual_dim, normalize=False
    )
    mps.normalize_()

    gates = [
        view_gate_matrix_as_tensor(torch.matrix_exp(-tau * view_gate_tensor_as_matrix(h)))
        for h in hamiltonians
    ]

    I = torch.eye(mps.physical_dim, dtype=torch.int32, device=device)
    gr = einsum(I, I, "b0 b1, d0 d1 -> b0 b1 d0 d1")
    gr = rearrange(gr, "b0 b1 d0 d1 -> b0 (b1 d1) d0").to(dtype=dtype)  # (b, g, d)

    inversed_temperatur = 0.0
    iters_for_tau = 0
    local_energies = 0.0

    for t in tqdm(range(iterations), **progress_bar_kwargs):
        for p, pos in enumerate(positions):
            p_left, p_right = pos.tolist()

            if abs(mps.center - p_left) < abs(mps.center - p_right):
                mps.center_orthogonalization_(p_left, mode="qr")
            else:
                mps.center_orthogonalization_(p_right, mode="qr")

            gate = gates[p] if len(gates) > 1 else gates[0]
            gl = rearrange(gate, "a b c d -> a (b d) c")  # (a, g, c)
            mps_local_tensors = evolve_gate_2body(mps.local_tensors, gl, gr, p_left, p_right)

            if p == interaction_num - 1:
                pos_next = positions[0]
            else:
                pos_next = positions[p + 1]

            direction = direction_to_next_center(p_left, p_right, pos_next[0], pos_next[1])
            if direction == "right-to-left":
                mps_local_tensors = orthogonalize_arange(
                    mps_tensors=mps_local_tensors,
                    start_idx=p_right,
                    end_idx=p_left,
                    mode="qr",
                    normalize=False,
                )

                mps_local_tensors = orthogonalize_arange(
                    mps_tensors=mps_local_tensors,
                    start_idx=p_left,
                    end_idx=p_right,
                    mode="svd",
                    truncate_dim=max_virtual_dim,
                    normalize=False,
                )

                mps = MPS(mps_tensors=mps_local_tensors)
                mps._center = p_right
            else:
                mps_local_tensors = orthogonalize_arange(
                    mps_tensors=mps_local_tensors,
                    start_idx=p_left,
                    end_idx=p_right,
                    mode="qr",
                    normalize=False,
                )

                mps_local_tensors = orthogonalize_arange(
                    mps_tensors=mps_local_tensors,
                    start_idx=p_right,
                    end_idx=p_left,
                    mode="svd",
                    truncate_dim=max_virtual_dim,
                    normalize=False,
                )

                mps = MPS(mps_tensors=mps_local_tensors)
                mps._center = p_left

            mps.center_normalize_()

        inversed_temperatur += tau
        iters_for_tau += 1

        if (iters_for_tau > tau_min and (t + 1) % calc_observation_iters == 0) or (
            t == iterations - 1
        ):
            local_energies_new = calculate_mps_local_energies(mps, hamiltonians, positions)
            # FIXME: local_energies_new seems to be wrong, lower than avg_energy_ref
            ref_mps = MPS_tebd(
                tensors=mps.local_tensors, para={"dtype": mps.dtype, "device": mps.device}
            )
            ref_mps.center = mps.center
            ps = [p.tolist() for p in positions]
            hs = hamiltonians if len(hamiltonians) > 1 else hamiltonians * len(ps)
            local_energies_ref = ref_mps.calculate_local_energies(hs, ps)
            assert torch.allclose(local_energies_new, local_energies_ref)
            avg_diff_local_energies = (local_energies_new - local_energies).abs().mean()
            local_energies = local_energies_new
            if avg_diff_local_energies < e0_eps or t == iterations - 1:
                print(f"Iteration {t}")
                tau *= 0.5
                iters_for_tau = 0
                if tau < tau_min or t == iterations - 1:
                    print("  E converged. Break iteration.")
                    break

                gates = [
                    view_gate_matrix_as_tensor(
                        torch.matrix_exp(-tau * view_gate_tensor_as_matrix(h))
                    )
                    for h in hamiltonians
                ]
                print(f"  Reduce tau to {tau}")

    return mps, local_energies
