"""先对 $s_1s_2s_3$ 求和，可以得到 $\rho_{ss_0}$"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../../2-6.ipynb.

# %% auto 0
__all__ = ['calc_reduced_density_matrix', 'calc_observation', 'calc_onsite_entanglement_entropy', 'project_state', 'observe_bond_energies', 'bipartite_entanglement_entropy']

# %% ../../2-6.ipynb 3
from typing import List
import math
import torch
from ..utils.checking import check_quantum_gate

# %% ../../2-6.ipynb 4
def calc_reduced_density_matrix(state: torch.Tensor, qubit_idx: int | List[int]):
    """
    Calculate the reduced density matrix of a quantum state.
    Args:
        state (torch.Tensor): The quantum state to calculate the reduced density matrix of.
        qubit_idx (int | List[int]): The indices of the qubits to keep in reduction.
    Returns:
        torch.Tensor: The reduced density matrix.
    """
    assert isinstance(qubit_idx, (int, list)), "qubit_idx must be an integer or a list of integers"
    if isinstance(qubit_idx, int):
        qubit_idx = [qubit_idx]

    num_qubits = state.ndim
    for qi in qubit_idx:
        assert 0 <= qi < num_qubits, "qubit_idx must be in [0, num_qubits - 1]"

    qubit_indices = list(range(num_qubits))
    dims_to_reduce = [i for i in qubit_indices if i not in qubit_idx]
    dims_to_keep = qubit_idx
    state = state.permute(*(dims_to_keep + dims_to_reduce))  # (*dims_to_keep, *dims_to_reduce)
    shape_to_keep = math.prod([state.shape[i] for i in dims_to_keep])
    shape_to_reduce = math.prod([state.shape[i] for i in dims_to_reduce])
    state = state.reshape(shape_to_keep, shape_to_reduce)
    reduced_density_matrix = state @ state.conj().T
    return reduced_density_matrix


def calc_observation(
    state: torch.Tensor, operator: torch.Tensor, qubit_idx: int | List[int], _fast_mode: bool = True
):
    """
    Calculate the expectation value of an operator on a quantum state.
    Args:
        state (torch.Tensor): The quantum state to calculate the expectation value of.
        operator (torch.Tensor): The operator to calculate the expectation value of.
        qubit_idx (int | List[int]): The indices of the qubits to calculate the expectation value of.
    Returns:
        torch.Tensor: The expectation value of the operator on the state.
    """
    if isinstance(qubit_idx, int):
        length = 1
    else:
        length = len(qubit_idx)
    reduced_density_matrix = calc_reduced_density_matrix(state, qubit_idx)
    num_qubits_operator = check_quantum_gate(operator)
    assert num_qubits_operator == length, (
        "The number of qubits of the operator does not match the number of qubits of the state"
    )
    operator_mat = operator.reshape(2**num_qubits_operator, 2**num_qubits_operator)
    if _fast_mode:
        return (reduced_density_matrix * operator_mat.T).sum()
    else:
        return torch.trace(reduced_density_matrix @ operator_mat)

# %% ../../4-10.ipynb 4
from ..utils.checking import check_state_tensor
from .functional import calc_reduced_density_matrix


def calc_onsite_entanglement_entropy(
    state: torch.Tensor, qubit_idx: int | List[int] | None = None, eps: float = 1e-14
) -> torch.Tensor:
    """
    Calculate the onsite entanglement entropy of a quantum state.
    Args:
        state: The quantum state to calculate the onsite entanglement entropy of.
        qubit_idx: The indices of the qubits to calculate the onsite entanglement entropy of. If None, all qubits are considered.
        eps: The small number to avoid log(0).
    Returns:
        The onsite entanglement entropy of the quantum state.
    """
    check_state_tensor(state)
    n_qubits = state.ndim
    if qubit_idx is None:
        qubit_idx = list(range(n_qubits))
    elif isinstance(qubit_idx, int):
        qubit_idx = [qubit_idx]
    else:
        assert isinstance(qubit_idx, List), (
            f"qubit_idx must be an integer or a list of integers, got {type(qubit_idx)}"
        )
        assert len(qubit_idx) > 0, "qubit_idx must be a non-empty list"
    qubit_idx = torch.tensor(qubit_idx)
    assert torch.all(qubit_idx < n_qubits), "qubit_idx must be less than the number of qubits"
    assert torch.all(qubit_idx >= 0), "qubit_idx must be non-negative"
    oees = []
    for idx in qubit_idx:
        rdm = calc_reduced_density_matrix(state, idx.item())
        eigvals = torch.linalg.eigvalsh(rdm)
        oees.append(-eigvals.inner(torch.log(eigvals + eps)))

    oees = torch.stack(oees)
    return oees


def project_state(
    state: torch.Tensor, project_qubit_state: torch.Tensor, project_qubit_idx: int
) -> torch.Tensor:
    """
    Project a quantum state onto a specific qubit state.
    Args:
        state: The quantum state to project.
        project_qubit_state: The state to project onto.
        project_qubit_idx: The index of the qubit to project onto.
    Returns:
        The projected quantum state.
    """
    check_state_tensor(state)
    assert project_qubit_state.ndim == 1 and project_qubit_state.shape[0] == 2, (
        "project_qubit_state must be a 1D tensor with 2 elements"
    )
    new_state = torch.tensordot(state, project_qubit_state, dims=([project_qubit_idx], [0]))
    new_state = new_state / new_state.norm()
    return new_state

# %% ../../5-2.ipynb 29
from math import prod


def observe_bond_energies(
    quantum_state: torch.Tensor,
    hamiltonian: torch.Tensor | List[torch.Tensor],
    interaction_positions: List[List[int]],
):
    """
    Calculate the bond energies of a quantum state.

    Args:
        quantum_state: The quantum state to calculate the bond energies of.
        hamiltonian: The hamiltonian to calculate the bond energies of.
        interaction_positions: The positions of the interactions.

    Returns:
        The bond energies of the quantum state.
    """
    check_state_tensor(quantum_state)
    if isinstance(hamiltonian, torch.Tensor):
        check_quantum_gate(hamiltonian, assert_tensor_form=True)
        hamiltonian = [hamiltonian] * len(interaction_positions)
    elif isinstance(hamiltonian, List):
        assert len(hamiltonian) == len(interaction_positions), (
            "hamiltonian and interaction_positions must have the same length"
        )
        for t in hamiltonian:
            check_quantum_gate(t, assert_tensor_form=True)
    else:
        raise ValueError("hamiltonian must be a torch.Tensor or List[torch.Tensor]")

    bond_energies = []
    for h, pos in zip(hamiltonian, interaction_positions):
        bond_energies.append(calc_observation(quantum_state, h, pos))
    return torch.stack(bond_energies)


def bipartite_entanglement_entropy(quantum_state: torch.Tensor, qubit_idx: int | List[int] | None):
    """
    Calculate the bipartite entanglement entropy of a quantum state.

    Args:
        quantum_state: The quantum state to calculate the bipartite entanglement entropy of.
        qubit_idx: The index of the qubit to calculate the bipartite entanglement entropy of. If None, calculate the bipartite entanglement entropy of all qubits.

    Returns:
        The bipartite entanglement entropy of the quantum state.
    """
    EPS = 1e-14
    assert isinstance(qubit_idx, (int, List)) or qubit_idx is None
    check_state_tensor(quantum_state)
    num_qubits = quantum_state.ndim
    if qubit_idx is None:
        indices = range(1, num_qubits)
    elif isinstance(qubit_idx, int):
        indices = [qubit_idx]
    else:
        indices = qubit_idx
    assert all(1 <= i <= num_qubits for i in indices), (
        "qubit_idx must be in the range of [1, num_qubits)"
    )

    entropies = []
    for idx in indices:
        mat = quantum_state.reshape(prod(quantum_state.shape[:idx]), -1)
        eigenvalues = torch.linalg.svdvals(mat) ** 2
        entropies.append(-eigenvalues.inner(torch.log(eigenvalues + EPS)))

    return torch.stack(entropies)