# AUTOGENERATED! DO NOT EDIT! File to edit: ../../0-utils-data.ipynb.

# %% auto 0
__all__ = ['load_iris', 'get_mnist_datasets', 'load_mnist_images', 'get_fashion_mnist_datasets', 'split_classification_dataset']

# %% ../../0-utils-data.ipynb 0
from typing import Tuple, Iterable, Literal
import torch
from torch.utils import data
from torchvision import datasets, transforms
from functools import cache

# %% ../../0-utils-data.ipynb 2
@cache
def load_iris(*, force_single_precision=False) -> Tuple[torch.Tensor, torch.Tensor]:
    """
    Load the Iris dataset and return the data and targets.

    Args:
        force_single_precision: Whether to force the data to be single precision.
    Returns:
        Tuple[torch.Tensor, torch.Tensor]: The data and targets.
    """
    from sklearn import datasets

    iris = datasets.load_iris()
    data = torch.from_numpy(iris["data"])
    if force_single_precision:
        data = data.float()

    targets = torch.from_numpy(iris["target"]).to(torch.long)
    return data, targets

# %% ../../0-utils-data.ipynb 3
@cache
def get_mnist_datasets(
    cache_path: str, *, normalization: bool
) -> Tuple[data.Dataset, data.Dataset]:
    """
    Get the MNIST datasets.

    Args:
        cache_path: The path to the cache directory.
        normalization: Whether to normalize the images based on the mean and standard deviation of the MNIST dataset.
    Returns:
        Tuple[data.Dataset, data.Dataset]: The MNIST train and test datasets.
    """
    if normalization:
        transform = transforms.Compose(
            [
                transforms.ToTensor(),
                transforms.Normalize(
                    (0.1307,), (0.3081,)
                ),  # normalization, see https://discuss.pytorch.org/t/normalization-in-the-mnist-example/457
            ]
        )
    else:
        transform = transforms.Compose([transforms.ToTensor()])

    mnist_train_set = datasets.MNIST(
        root=cache_path, train=True, download=True, transform=transform
    )
    mnist_test_set = datasets.MNIST(
        root=cache_path, train=False, download=True, transform=transform
    )
    return mnist_train_set, mnist_test_set


def _calc_load_num(num: int | None, classes: int | Iterable[int] | None, dataset_len: int) -> int:
    if num is None:
        return dataset_len
    elif classes is None:
        # num not None, classes is None
        return num
    else:
        # num not None, classes is not None
        return dataset_len


def load_mnist_images(
    *,
    cache_path: str,
    num: int | None,
    from_subset: Literal["test", "train", "all"],
    shuffle: bool,
    normalization: bool,
    classes: int | Iterable[int] | None = None,
    return_labels: bool = False,
) -> torch.Tensor | Tuple[torch.Tensor, torch.Tensor]:
    """
    Load MNIST images from the cache directory.

    Args:
        cache_path: The path to the cache directory.
        num: The number of images to load.
        from_subset: The subset of the dataset to load.
        shuffle: Whether to shuffle the dataset.
        classes: The classes to load.
        normalization: Whether to normalize the images based on the mean and standard deviation of the MNIST dataset.
        return_labels: Whether to return the labels.
    Returns:
        torch.Tensor: The MNIST images. If return_labels is True, return a tuple of (images, labels).
    """
    assert num is None or num > 0
    assert classes is None or isinstance(classes, (int, Iterable)), (
        "classes must be an integer or an iterable of integers"
    )
    train_set, test_set = get_mnist_datasets(cache_path, normalization=normalization)
    if from_subset == "train":
        dataloader = data.DataLoader(
            train_set, _calc_load_num(num, classes, len(train_set)), shuffle=shuffle
        )
        images, labels = next(iter(dataloader))
    elif from_subset == "test":
        dataloader = data.DataLoader(
            test_set, _calc_load_num(num, classes, len(test_set)), shuffle=shuffle
        )
        images, labels = next(iter(dataloader))
    elif from_subset == "all":
        dataloader = data.DataLoader(
            train_set, _calc_load_num(num, classes, len(train_set)), shuffle=shuffle
        )
        train_data, train_labels = next(iter(dataloader))
        dataloader = data.DataLoader(
            test_set, _calc_load_num(num, classes, len(test_set)), shuffle=shuffle
        )
        test_data, test_labels = next(iter(dataloader))
        images = torch.cat([train_data, test_data], dim=0)
        labels = torch.cat([train_labels, test_labels], dim=0)
    else:
        raise ValueError(f"Invalid subset: {from_subset}")

    if classes is not None:
        if isinstance(classes, int):
            classes = [classes]
        else:
            classes = list(classes)

        mask = torch.isin(labels, torch.tensor(classes))
        images = images[mask]
        labels = labels[mask]

    if num is None:
        num = len(images)

    if return_labels:
        return images[:num], labels[:num]
    else:
        return images[:num]

# %% ../../0-utils-data.ipynb 4
@cache
def get_fashion_mnist_datasets(cache_path: str) -> Tuple[data.Dataset, data.Dataset]:
    """
    Get the Fashion MNIST datasets.

    Args:
        cache_path: The path to the cache directory.
    Returns:
        Tuple[data.Dataset, data.Dataset]: The Fashion MNIST train and test datasets.
    """

    fmnist_train_set = datasets.FashionMNIST(
        root=cache_path, train=True, download=True, transform=transforms.ToTensor()
    )
    fmnist_test_set = datasets.FashionMNIST(
        root=cache_path, train=False, download=True, transform=transforms.ToTensor()
    )
    return fmnist_train_set, fmnist_test_set

# %% ../../3-5.ipynb 20
def split_classification_dataset(
    data: torch.Tensor, targets: torch.Tensor, ratio: float, shuffle: bool = True
) -> Tuple[torch.Tensor, torch.Tensor, torch.Tensor, torch.Tensor]:
    """
    Split the dataset into training and testing sets.

    Args:
        data (torch.Tensor): The data tensor.
        target (torch.Tensor): The target tensor.
        ratio (float): The ratio of the training set.
        shuffle (bool): Whether to shuffle the data.

    Returns:
        Tuple[torch.Tensor, torch.Tensor, torch.Tensor, torch.Tensor]: The training samples, training labels, testing samples, testing labels.
    """
    assert targets.dtype in [torch.int, torch.long], "target must be an integer tensor"
    assert 1.0 > ratio > 0.0, "ratio must be between 0 and 1"
    num_classes = torch.max(targets) + 1
    num_samples = data.shape[0]
    assert num_samples == targets.shape[0], "data and target must have the same number of samples"
    train_samples = []
    train_labels = []
    test_samples = []
    test_labels = []
    for class_idx in range(num_classes):
        sample_class_i = data[targets == class_idx]
        num_samples_class_i = sample_class_i.shape[0]
        num_train = int(num_samples_class_i * (1 - ratio))
        if shuffle:
            shuffle_idx = torch.randperm(num_samples_class_i)
            shuffled_sample_class_i = sample_class_i[shuffle_idx]
            train_sample = shuffled_sample_class_i[:num_train]
            train_samples.append(train_sample)
            test_sample = shuffled_sample_class_i[num_train:]
            test_samples.append(test_sample)

            train_labels.append(
                torch.full((train_sample.shape[0],), class_idx, dtype=targets.dtype)
            )
            test_labels.append(torch.full((test_sample.shape[0],), class_idx, dtype=targets.dtype))
        else:
            train_sample = sample_class_i[:num_train]
            train_samples.append(train_sample)
            test_sample = sample_class_i[num_train:]
            test_samples.append(test_sample)
            train_labels.append(
                torch.full((train_sample.shape[0],), class_idx, dtype=targets.dtype)
            )
            test_labels.append(torch.full((test_sample.shape[0],), class_idx, dtype=targets.dtype))

    train_samples = torch.cat(train_samples, dim=0)
    train_labels = torch.cat(train_labels, dim=0)
    test_samples = torch.cat(test_samples, dim=0)
    test_labels = torch.cat(test_labels, dim=0)
    return train_samples, train_labels, test_samples, test_labels