import ast
import builtins
import linecache
import logging
import re
from collections import deque
from datetime import datetime
from functools import lru_cache
from json import loads
from typing import Any, Callable, Dict, List, Optional, Tuple, Union

from tornado import escape
from tornado.util import ObjectDict, exec_in, unicode_type

from tinybird.context import (
    ff_column_json_backticks_circuit_breaker,
    ff_split_to_array_escape,
)

from .datatypes import testers
from .tornado_template import VALID_CUSTOM_FUNCTION_NAMES, SecurityException, Template

TB_SECRET_IN_TEST_MODE = "tb_secret_dont_raise"
TB_SECRET_PREFIX = "tb_secret_"
CH_PARAM_PREFIX = "param_"
REQUIRED_PARAM_NOT_DEFINED = "Required parameter is not defined"

# Pre-compiled regex patterns for performance
_STRING_LINE_NUMBER_RE = re.compile(r"\<string\>:(\d*)")
_ARRAY_TYPE_RE = re.compile(r"Array\((\w+)\)")
_EMBEDDED_TEMPLATE_EXPRESSION_RE = re.compile(r"\{\{(.*?)\}\}")


def secret_template_key(secret_name: str) -> str:
    return f"{TB_SECRET_PREFIX}{secret_name}"


def is_secret_template_key(key: str) -> bool:
    return key.startswith(TB_SECRET_PREFIX)


class TemplateExecutionResults(dict):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.template_params = set()
        self.ch_params = set()

    def add_template_param(self, param: str):
        self.template_params.add(param)

    def add_ch_param(self, param: str):
        self.ch_params.add(param)

    def update_all(self, other: "TemplateExecutionResults"):
        self.update(other)
        self.ch_params.update(other.ch_params)
        self.template_params.update(other.template_params)


class SQLTemplateCustomError(Exception):
    def __init__(self, err, code=400):
        self.code = code
        self.err = err
        super().__init__(err)


class SQLTemplateException(ValueError):
    def __init__(self, err, documentation=None):
        self.documentation = documentation
        super().__init__(f"Template Syntax Error: {str(err)}")


# t = Template(""" SELECT * from test where lon between {{Float32(lon1, 0)}} and {{Float32(lon2, 0)}} """)
# names = get_var_names(t)
# print(generate(t, **{x: '' for x in names}))

# t = Template(""" SELECT * from test where lon between {{lon1}} and {{lon2}} """)
# names = get_var_names(t)
# replace_vars_smart(t)
# print(generate(t, **{x: '' for x in names}))

JOB_TIMESTAMP_PARAM = "job_timestamp"
DEFAULT_PARAM_NAMES = ["format", "q"]
RESERVED_PARAM_NAMES = [
    "__tb__semver",
    "debug_source_tables",
    "debug",
    "explain",
    "finalize_aggregations",
    "output_format_json_quote_64bit_integers",
    "output_format_json_quote_denormals",
    "output_format_parquet_string_as_string",
    "pipeline",
    "playground",
    "q",
    "query_id",
    "release_replacements",
    "tag",
    "template_parameters",
    "token",
    JOB_TIMESTAMP_PARAM,
]

parameter_types = [
    "String",
    "Boolean",
    "DateTime64",
    "DateTime",
    "Date",
    "Float32",
    "Float64",
    "Int8",
    "Int16",
    "Int32",
    "Int64",
    "Int128",
    "Int256",
    "UInt8",
    "UInt16",
    "UInt32",
    "UInt64",
    "UInt128",
    "UInt256",
    "Array",
    "JSON",
]


def transform_type(
    tester, transform, placeholder=None, required=None, description=None, enum=None, example=None, format=None
):
    def _f(x, default=None, defined=True, required=None, description=None, enum=None, example=None, format=None):
        if isinstance(x, Placeholder):
            if default:
                x = default
            else:
                x = placeholder
        elif x is None:
            x = default
            if x is None:
                if defined:
                    raise SQLTemplateException(REQUIRED_PARAM_NOT_DEFINED, documentation="/cli/advanced-templates.html")
                else:
                    return None
        if tester == "String":
            if x is not None:
                return transform(x)
        elif testers[tester](str(x)):
            return transform(x)
        raise SQLTemplateException(
            f"Error validating '{x}' to type {tester}", documentation="/cli/advanced-templates.html"
        )

    return _f


def _and(*args, **kwargs):
    operands = {"in": "in", "not_in": "not in", "gt": ">", "lt": "<", "gte": ">=", "lte": "<="}

    def _name(k):
        tk = k.rsplit("__", 1)
        return tk[0]

    def _op(k):
        tk = k.rsplit("__", 1)
        if len(tk) == 1:
            return "="
        else:
            if tk[1] in operands:
                return operands[tk[1]]
            raise SQLTemplateException(
                f"operand {tk[1]} not supported", documentation="/cli/advanced-templates.html#sql_and"
            )

    return Expression(
        " and ".join([f"{_name(k)} {_op(k)} {expression_wrapper(v, k)}" for k, v in kwargs.items() if v is not None])
    )


def error(s, code=400):
    raise ValueError(s)


def custom_error(s, code=400):
    raise SQLTemplateCustomError(s, code)


class Expression(str):
    pass


class Comment:
    def __init__(self, s):
        self.text = s

    def __str__(self):
        return self.text


class Placeholder:
    def __init__(self, name=None, line=None):
        self.name = name if name else "__no_value__"
        self.line = line or "unknown"

    def __str__(self):
        return "__no_value__"

    def __getitem__(self, i):
        if i > 2:
            raise IndexError()
        return Placeholder()

    def __add__(self, s):
        return Placeholder()

    def __call__(self, *args, **kwargs):
        raise SQLTemplateException(
            f"'{self.name}' is not a valid function, line {self.line}", documentation="/cli/advanced-templates.html"
        )

    def split(self, ch):
        return [Placeholder(), Placeholder()]

    def startswith(self, c):
        return False


class Symbol:
    def __init__(self, x):
        self.x = x

    def __str__(self):
        return self.x


class Integer(int):
    def __new__(self, value, type):
        return int.__new__(self, value)

    def __init__(self, value, type):
        int.__init__(value)
        self.type = type

    def __str__(self):
        return f"to{self.type}('{int(self)}')"


class Float(float):
    def __new__(self, value, type):
        return float.__new__(self, value)

    def __init__(self, value, type):
        float.__init__(value)
        self.type = type

    def __str__(self):
        return f"to{self.type}('{float(self)}')"


def columns(x, default=None, fn=None):
    if x is None or isinstance(x, Placeholder):
        if default is None:
            raise SQLTemplateException(
                "Missing columns() default value, use `columns(column_names, 'default_column_name')`",
                documentation="/cli/advanced-templates.html#columns",
            )
        x = default

    try:
        _columns = [c.strip() for c in x.split(",")]
    except AttributeError:
        raise SQLTemplateException(
            "The 'columns' function expects a String not an Array", documentation="/cli/advanced-templates.html#columns"
        )

    if fn:
        return Expression(",".join(f"{fn}({str(column(c, c))}) as {c}" for c in _columns))
    else:
        return Expression(",".join(str(column(c, c)) for c in _columns))


def column(x, default=None):
    bypass_colunn_json_backticks = ff_column_json_backticks_circuit_breaker.get(False)

    if x is None or isinstance(x, Placeholder):
        if default is None:
            raise SQLTemplateException(
                "Missing column() default value, use `column(column_name, 'default_column_name')`",
                documentation="/cli/advanced-templates.html#column",
            )
        x = default

    quote = "`"
    if bypass_colunn_json_backticks:
        return Symbol(quote + sqlescape(x) + quote)

    try:
        slices = x.split(".")
        escaped_slices = [quote + sqlescape(s) + quote for s in slices]
        escaped = ".".join(escaped_slices)
        return Symbol(escaped)
    except Exception:  # in case there's a problem with .split
        return Symbol(quote + sqlescape(x) + quote)


def symbol(x, quote="`"):
    if isinstance(x, Placeholder):
        return Symbol("`placeholder`")
    return Symbol(quote + sqlescape(x) + quote)


def table(x, quote="`"):
    if isinstance(x, Placeholder):
        return Symbol("placeholder")
    return Symbol(sqlescape(x))


# ClickHouse does not have a boolean type. Docs suggest to use 1/0:
#
#     https://clickhouse.com/docs/en/sql-reference/data-types/boolean/
#
def boolean(x, default=None):
    """
    >>> boolean(True)
    1
    >>> boolean(False)
    0
    >>> boolean('TRUE')
    1
    >>> boolean('FALSE')
    0
    >>> boolean('true')
    1
    >>> boolean('false')
    0
    >>> boolean(1)
    1
    >>> boolean(0)
    0
    >>> boolean('1')
    1
    >>> boolean('0')
    0
    >>> boolean(None)
    0
    >>> boolean(None, default=True)
    1
    >>> boolean(None, default=False)
    0
    >>> boolean(None, default='TRUE')
    1
    >>> boolean(None, default='FALSE')
    0
    >>> boolean(Placeholder())
    0
    >>> boolean(Placeholder(), default=True)
    1
    >>> boolean(Placeholder(), default=False)
    0
    >>> boolean(Placeholder(), default='TRUE')
    1
    >>> boolean(Placeholder(), default='FALSE')
    0
    """
    if x is None:
        if default is None:
            return 0
        return boolean(default)
    elif isinstance(x, Placeholder):
        return boolean(default)
    elif isinstance(x, str):
        if x == "0" or x.lower() == "false":
            return 0

    return int(bool(x))


def defined(x=None):
    return not (isinstance(x, Placeholder) or x is None)


def array_type(types):
    def _f(
        x, _type=None, default=None, defined=True, required=None, description=None, enum=None, example=None, format=None
    ):
        try:
            if isinstance(x, Placeholder):
                if default:
                    x = default
                elif _type and _type in types:
                    if _type == "String":
                        x = ""
                    else:
                        x = ",".join(map(str, [types[_type](x) for _ in range(2)]))
                else:
                    x = ""
            elif x is None:
                x = default
                if x is None:
                    if defined:
                        raise SQLTemplateException(
                            REQUIRED_PARAM_NOT_DEFINED, documentation="/cli/advanced-templates.html"
                        )
                    else:
                        return None
            values = []
            list_values = x if type(x) == list else x.split(",")  # noqa: E721
            for i, t in enumerate(list_values):
                if _type in testers:
                    if testers[_type](str(t)):
                        values.append(expression_wrapper(types[_type](t), str(t)))
                    else:
                        raise SQLTemplateException(
                            f"Error validating [{x}][{i}] ({'empty string' if t == '' else t}) to type {_type}",
                            documentation="/cli/advanced-templates.html",
                        )
                else:
                    values.append(expression_wrapper(types.get(_type, lambda x: x)(t), str(t)))
            return Expression(f"[{','.join(map(str, values))}]")
        except AttributeError as e:
            logging.warning(f"AttributeError on Array: {e}")
            raise SQLTemplateException(
                "transform type function Array is not well defined", documentation="/cli/advanced-templates.html"
            )

    return _f


def sql_unescape(x, what=""):
    """
    unescapes specific characters in a string. It allows to allow some
    special characters to be used, for example in like condictionals

    {{sql_unescape(String(like_filter), '%')}}


    >>> sql_unescape('testing%', '%')
    "'testing%'"
    >>> sql_unescape('testing%', '$')
    "'testing\\\\%'"
    >>> sql_unescape('testing"')
    '\\'testing"\\''
    """
    return Expression("'" + sqlescape_for_string_expression(x).replace(f"\\{what}", what) + "'")


def split_to_array(x, default="", separator: str = ","):
    try:
        if isinstance(x, Placeholder) or x is None:
            x = default
        return [s.strip() for s in x.split(separator)]
    except AttributeError as e:
        logging.warning(f"warning on split_to_array: {str(e)}")
        raise SQLTemplateException(
            "First argument of split_to_array has to be a value that can be split to a list of elements, but found a PlaceHolder with no value instead",
            documentation="/cli/advanced-templates.html#split_to_array",
        )


def enumerate_with_last(arr):
    """
    >>> enumerate_with_last([1, 2])
    [(False, 1), (True, 2)]
    >>> enumerate_with_last([1])
    [(True, 1)]
    """
    arr_len = len(arr)
    return [(arr_len == i + 1, x) for i, x in enumerate(arr)]


def string_type(x, default=None):
    if isinstance(x, Placeholder):
        if default:
            x = default
        else:
            x = "__no_value__"
    return x


def day_diff(d0, d1, default=None):
    """
    >>> day_diff('2019-01-01', '2019-01-01')
    0
    >>> day_diff('2019-01-01', '2019-01-02')
    1
    >>> day_diff('2019-01-02', '2019-01-01')
    1
    >>> day_diff('2019-01-02', '2019-02-01')
    30
    >>> day_diff('2019-02-01', '2019-01-02')
    30
    >>> day_diff(Placeholder(), Placeholder())
    0
    >>> day_diff(Placeholder(), '')
    0
    """
    try:
        return date_diff_in_days(d0, d1, date_format="%Y-%m-%d")
    except Exception:
        raise SQLTemplateException(
            "invalid date format in function `day_diff`, it must be ISO format date YYYY-MM-DD, e.g. 2018-09-26. For other fotmats, try `date_diff_in_days`",
            documentation="/cli/advanced-templates.html#date_diff_in_days",
        )


def date_diff_in_days(
    d0: Union[Placeholder, str],
    d1: Union[Placeholder, str],
    date_format: str = "%Y-%m-%d",
    default=None,
    backup_date_format=None,
    none_if_error=False,
):
    """
    >>> date_diff_in_days('2019-01-01', '2019-01-01')
    0
    >>> date_diff_in_days('2019-01-01', '2019-01-02')
    1
    >>> date_diff_in_days('2019-01-02', '2019-01-01')
    1
    >>> date_diff_in_days('2019-01-02', '2019-02-01')
    30
    >>> date_diff_in_days('2019-02-01 20:00:00', '2019-01-02 20:00:00', date_format="%Y-%m-%d %H:%M:%S")
    30
    >>> date_diff_in_days('2019-02-01', '2019-01-02')
    30
    >>> date_diff_in_days(Placeholder(), Placeholder())
    0
    >>> date_diff_in_days(Placeholder(), '')
    0
    >>> date_diff_in_days('2019-01-01', '2019/01/01', backup_date_format='%Y/%m/%d')
    0
    >>> date_diff_in_days('2019-01-01', '2019/01/04', backup_date_format='%Y/%m/%d')
    3
    >>> date_diff_in_days('2019/01/04', '2019-01-01', backup_date_format='%Y/%m/%d')
    3
    >>> date_diff_in_days('2019-02-01T20:00:00z', '2019-02-15', '%Y-%m-%dT%H:%M:%Sz', backup_date_format='%Y-%m-%d')
    13
    >>> date_diff_in_days('2019-02-01 20:00:00', '2019-02-15', '%Y-%m-%dT%H:%M:%Sz', backup_date_format='%Y-%m-%d', none_if_error=True) is None
    True
    >>> date_diff_in_days('2019-01-01', '2019-00-02', none_if_error=True) is None
    True
    >>> date_diff_in_days('2019-01-01 00:00:00', '2019-01-02 00:00:00', none_if_error=True) is None
    True
    """
    if isinstance(d0, Placeholder) or isinstance(d1, Placeholder):
        if default:
            return default
        return 0
    try:
        return __date_diff(d0, d1, date_format, backup_date_format, "days", none_if_error)
    except Exception:
        raise SQLTemplateException(
            "invalid date format in function `date_diff_in_days`, it must be ISO format date YYYY-MM-DD, e.g. 2018-09-26",
            documentation="/cli/advanced-templates.html#date_diff_in_days",
        )


def date_diff_in_hours(
    d0: Union[Placeholder, str],
    d1: Union[Placeholder, str],
    date_format: str = "%Y-%m-%d %H:%M:%S",
    default=None,
    backup_date_format=None,
    none_if_error=False,
):
    """
    >>> date_diff_in_hours('2022-12-19T18:42:23.521Z', '2022-12-19T18:42:23.521Z', date_format='%Y-%m-%dT%H:%M:%S.%fz')
    0
    >>> date_diff_in_hours('2022-12-19T20:43:22Z', '2022-12-19T18:42:23Z','%Y-%m-%dT%H:%M:%Sz')
    2
    >>> date_diff_in_hours('2022-12-14 18:42:22', '2022-12-19 18:42:22')
    120
    >>> date_diff_in_hours('2022-12-19 18:42:23.521', '2022-12-19 18:42:24.521','%Y-%m-%d %H:%M:%S.%f')
    0
    >>> date_diff_in_hours(Placeholder(), Placeholder())
    0
    >>> date_diff_in_hours(Placeholder(), '')
    0
    >>> date_diff_in_hours('2022-12-19T03:22:12.102Z', '2022-12-19', date_format='%Y-%m-%dT%H:%M:%S.%fz', backup_date_format='%Y-%m-%d')
    3
    >>> date_diff_in_hours('2022-12-19', '2022-12-19', '%Y-%m-%dT%H:%M:%Sz', backup_date_format='%Y-%m-%d')
    0
    >>> date_diff_in_hours('2022-12-19', '2022-12-18', '%Y-%m-%dT%H:%M:%Sz', backup_date_format='%Y-%m-%d')
    24
    >>> date_diff_in_hours('2022-12-19', '2022-12-19 02:01:00', backup_date_format='%Y-%m-%d')
    2
    >>> date_diff_in_hours('2022-25-19T00:00:03.521Z', '2022-12-19', date_format='%Y-%m-%dT%H:%M:%S.%fz', backup_date_format='%Y-%m-%d', none_if_error=True) is None
    True
    >>> date_diff_in_hours('2022-25-19 00:00:03', '2022-12-19', date_format='%Y-%m-%dT%H:%M:%S.%fz', backup_date_format='%Y-%m-%d', none_if_error=True) is None
    True
    >>> date_diff_in_hours('2022-12-19', '2022-25-19', '%Y-%m-%dT%H:%M:%Sz', backup_date_format='%Y-%m-%d', none_if_error=True) is None
    True
    >>> date_diff_in_hours('2022-12-19', '2022-25-19 00:01:00', backup_date_format='%Y-%m-%d', none_if_error=True) is None
    True
    >>> date_diff_in_hours('2022-12-32 18:42:22', '2022-12-19 18:42:22', none_if_error=True) is None
    True
    >>> date_diff_in_hours('2022-12-18T18:42:22Z', '2022-12-19T18:42:22Z', none_if_error=True) is None
    True
    """
    if isinstance(d0, Placeholder) or isinstance(d1, Placeholder):
        if default:
            return default
        return 0
    try:
        return __date_diff(d0, d1, date_format, backup_date_format, "hours", none_if_error)
    except Exception:
        raise SQLTemplateException(
            "invalid date_format in function `date_diff_in_hours`, defaults to YYYY-MM-DD hh:mm:ss. Or %Y-%m-%d %H:%M:%S [.ssssss]Z, e.g. ms: 2022-12-19T18:42:22.591Z s:2022-12-19T18:42:22Z",
            documentation="/cli/advanced-templates.html#date_diff_in_hours",
        )


def date_diff_in_minutes(
    d0: Union[Placeholder, str],
    d1: Union[Placeholder, str],
    date_format: str = "%Y-%m-%d %H:%M:%S",
    default=None,
    backup_date_format=None,
    none_if_error=False,
):
    """
    >>> date_diff_in_minutes('2022-12-19T18:42:23.521Z', '2022-12-19T18:42:23.521Z', date_format='%Y-%m-%dT%H:%M:%S.%fz')
    0
    >>> date_diff_in_minutes('2022-12-19T18:43:22Z', '2022-12-19T18:42:23Z','%Y-%m-%dT%H:%M:%Sz')
    0
    >>> date_diff_in_minutes('2022-12-14 18:42:22', '2022-12-19 18:42:22')
    7200
    >>> date_diff_in_minutes('2022-12-19 18:42:23.521', '2022-12-19 18:42:24.521','%Y-%m-%d %H:%M:%S.%f')
    0
    >>> date_diff_in_minutes(Placeholder(), Placeholder())
    0
    >>> date_diff_in_minutes(Placeholder(), '')
    0
    >>> date_diff_in_minutes('2022-12-19T03:22:12.102Z', '2022-12-19', date_format='%Y-%m-%dT%H:%M:%S.%fz', backup_date_format='%Y-%m-%d')
    202
    >>> date_diff_in_minutes('2022-12-19', '2022-12-19', '%Y-%m-%dT%H:%M:%Sz', backup_date_format='%Y-%m-%d')
    0
    >>> date_diff_in_minutes('2022-12-19', '2022-12-18', '%Y-%m-%dT%H:%M:%Sz', backup_date_format='%Y-%m-%d')
    1440
    >>> date_diff_in_minutes('2022-12-19', '2022-12-19 00:01:00', backup_date_format='%Y-%m-%d')
    1
    >>> date_diff_in_minutes('2022-25-19T00:00:03.521Z', '2022-12-19', date_format='%Y-%m-%dT%H:%M:%S.%fz', backup_date_format='%Y-%m-%d', none_if_error=True) is None
    True
    >>> date_diff_in_minutes('2022-12-19', '2022-25-19', '%Y-%m-%dT%H:%M:%Sz', backup_date_format='%Y-%m-%d', none_if_error=True) is None
    True
    >>> date_diff_in_minutes('2022-12-19', '2022-25-19 00:01:00', backup_date_format='%Y-%m-%d', none_if_error=True) is None
    True
    >>> date_diff_in_minutes('2022-25-19T00:00:03.521Z', '2022-12-19 00:23:12', date_format='%Y-%m-%dT%H:%M:%S.%fz', backup_date_format='%Y-%m-%d', none_if_error=True) is None
    True
    >>> date_diff_in_minutes('2022-12-14 18:42:22', '2022/12/19 18:42:22', none_if_error=True) is None
    True
    >>> date_diff_in_minutes('2022-12-14 18:42:22', '2022/12/19 18:42:22', date_format='%Y/%m/%dT%H:%M:%S.%fz', none_if_error=True) is None
    True
    """
    if isinstance(d0, Placeholder) or isinstance(d1, Placeholder):
        if default:
            return default
        return 0
    try:
        return __date_diff(d0, d1, date_format, backup_date_format, "minutes", none_if_error)
    except Exception:
        raise SQLTemplateException(
            "invalid date_format in function `date_diff_in_seconds`, defaults to YYYY-MM-DD hh:mm:ss. Or %Y-%m-%d %H:%M:%S [.ssssss]Z, e.g. ms: 2022-12-19T18:42:22.591Z s:2022-12-19T18:42:22Z",
            documentation="/cli/advanced-templates.html#date_diff_in_minutes",
        )


def date_diff_in_seconds(
    d0: Union[Placeholder, str],
    d1: Union[Placeholder, str],
    date_format: str = "%Y-%m-%d %H:%M:%S",
    default=None,
    backup_date_format=None,
    none_if_error=False,
):
    """
    >>> date_diff_in_seconds('2022-12-19T18:42:23.521Z', '2022-12-19T18:42:23.521Z', date_format='%Y-%m-%dT%H:%M:%S.%fz')
    0
    >>> date_diff_in_seconds('2022-12-19T18:42:22Z', '2022-12-19T18:42:23Z','%Y-%m-%dT%H:%M:%Sz')
    1
    >>> date_diff_in_seconds('2022-12-19 18:42:22', '2022-12-19 18:43:22')
    60
    >>> date_diff_in_seconds('2022-12-14 18:42:22', '2022-12-19 18:42:22')
    432000
    >>> date_diff_in_seconds('2022-12-19T18:42:23.521Z', '2022-12-19T18:42:23.531Z','%Y-%m-%dT%H:%M:%S.%fz')
    0
    >>> date_diff_in_seconds('2022-12-19 18:42:23.521', '2022-12-19 18:42:24.521','%Y-%m-%d %H:%M:%S.%f')
    1
    >>> date_diff_in_seconds('2022-12-19T18:42:23.521Z', '2022-12-19T18:44:23.531Z','%Y-%m-%dT%H:%M:%S.%fz')
    120
    >>> date_diff_in_seconds(Placeholder(), Placeholder())
    0
    >>> date_diff_in_seconds(Placeholder(), '')
    0
    >>> date_diff_in_seconds('2022-12-19T00:00:03.521Z', '2022-12-19', date_format='%Y-%m-%dT%H:%M:%S.%fz', backup_date_format='%Y-%m-%d')
    3
    >>> date_diff_in_seconds('2022-12-19', '2022-12-19', '%Y-%m-%dT%H:%M:%Sz', backup_date_format='%Y-%m-%d')
    0
    >>> date_diff_in_seconds('2022-12-19', '2022-12-19 00:01:00', backup_date_format='%Y-%m-%d')
    60
    >>> date_diff_in_seconds('2022-25-19T00:00:03.521Z', '2022-12-19', date_format='%Y-%m-%dT%H:%M:%S.%fz', backup_date_format='%Y-%m-%d', none_if_error=True) is None
    True
    >>> date_diff_in_seconds('2022-12-19', '2022-25-19', '%Y-%m-%dT%H:%M:%Sz', backup_date_format='%Y-%m-%d', none_if_error=True) is None
    True
    >>> date_diff_in_seconds('2022-12-19', '2022-25-19 00:01:00', backup_date_format='%Y-%m-%d', none_if_error=True) is None
    True
    >>> date_diff_in_seconds('2022-10-19T00:00:03.521Z', '2022/12/19', date_format='%Y-%m-%dT%H:%M:%S.%fz', backup_date_format='%Y-%m-%d', none_if_error=True) is None
    True
    >>> date_diff_in_seconds('2022-10-19 00:00:03', '2022-10-19 00:05:03', date_format='%Y-%m-%dT%H:%M:%S.%fz', none_if_error=True) is None
    True
    >>> date_diff_in_seconds('2022/12/19 00:00:03', '2022-10-19 00:05:03', none_if_error=True) is None
    True
    >>> date_diff_in_seconds('2022-25-19 00:00:03', '2022-10-19 00:05:03', none_if_error=True) is None
    True
    """
    if isinstance(d0, Placeholder) or isinstance(d1, Placeholder):
        if default:
            return default
        return 0
    try:
        return __date_diff(d0, d1, date_format, backup_date_format, "seconds", none_if_error)
    except Exception:
        raise SQLTemplateException(
            "invalid date_format in function `date_diff_in_seconds`, defaults to YYYY-MM-DD hh:mm:ss. Or %Y-%m-%d %H:%M:%S [.ssssss]Z, e.g. ms: 2022-12-19T18:42:22.591Z s:2022-12-19T18:42:22Z",
            documentation="/cli/advanced-templates.html#date_diff_in_seconds",
        )


def __date_diff(
    d0: Union[Placeholder, str],
    d1: Union[Placeholder, str],
    date_format: str = "%Y-%m-%d %H:%M:%S",
    backup_date_format=None,
    unit: str = "seconds",
    none_if_error=False,
):
    try:
        formatted_d0 = _parse_datetime(d0, date_format, backup_date_format)
        formatted_d1 = _parse_datetime(d1, date_format, backup_date_format)
        diff = abs(formatted_d1 - formatted_d0).total_seconds()

        if unit == "days":
            return int(diff / 86400)
        elif unit == "hours":
            return int(diff / 3600)
        elif unit == "minutes":
            return int(diff / 60)
        else:
            return int(diff)
    except Exception:
        if none_if_error:
            return None

        raise SQLTemplateException(
            "invalid date_format in function date_diff_* function", documentation="/cli/advanced-templates.html"
        )


def _parse_datetime(date_string, date_format, backup_date_format=None):
    formats = [date_format]
    if backup_date_format:
        formats.append(backup_date_format)

    for fmt in formats:
        try:
            return datetime.strptime(date_string, fmt)
        except ValueError:
            continue

    raise SQLTemplateException(
        "invalid date_format in function date_diff_* function", documentation="/cli/advanced-templates.html"
    )


def json_type(x, default=None):
    """
    >>> json_type(None, '[]')
    []
    >>> json_type(None)
    {}
    >>> json_type('{"a": 1}')
    {'a': 1}
    >>> json_type('[{"a": 1}]')
    [{'a': 1}]
    >>> json_type({"a": 1})
    {'a': 1}
    >>> json_type([{"a": 1}])
    [{'a': 1}]
    """
    if isinstance(x, Placeholder):
        if default:
            x = default
        else:
            x = "__no_value__"

    try:
        if x is None:
            if isinstance(default, str):
                x = default
            else:
                x = "{}"

        value = ""  # used for exception message
        if isinstance(x, (str, bytes, bytearray)):
            if len(x) > 16:
                value = x[:16] + "..."
            else:
                value = x

            parsed = loads(x)
            x = parsed
    except Exception as e:
        msg = f"Error parsing JSON: '{value}' - {str(e)}"
        raise SQLTemplateException(msg)

    return x


function_list = {
    "columns": columns,
    "table": table,
    "TABLE": table,
    "error": error,
    "custom_error": custom_error,
    "sql_and": _and,
    "defined": defined,
    "column": column,
    "enumerate_with_last": enumerate_with_last,
    "split_to_array": split_to_array,
    "day_diff": day_diff,
    "date_diff_in_days": date_diff_in_days,
    "date_diff_in_hours": date_diff_in_hours,
    "date_diff_in_minutes": date_diff_in_minutes,
    "date_diff_in_seconds": date_diff_in_seconds,
    "sql_unescape": sql_unescape,
    "JSON": json_type,
    # 'enumerate': enumerate
}


def get_transform_types(placeholders=None):
    if placeholders is None:
        placeholders = {}
    types = {
        "bool": boolean,
        "Boolean": boolean,
        "DateTime": transform_type(
            "DateTime",
            str,
            placeholders.get("DateTime", None),
            required=None,
            description=None,
            enum=None,
            example=None,
            format=None,
        ),
        "DateTime64": transform_type(
            "DateTime64",
            str,
            placeholders.get("DateTime64", None),
            required=None,
            description=None,
            enum=None,
            example=None,
            format=None,
        ),
        "Date": transform_type(
            "Date",
            str,
            placeholders.get("Date", None),
            required=None,
            description=None,
            enum=None,
            example=None,
            format=None,
        ),
        "Float32": transform_type(
            "Float32",
            lambda x: Float(x, "Float32"),
            placeholders.get("Float32", None),
            required=None,
            description=None,
            enum=None,
            example=None,
            format=None,
        ),
        "Float64": transform_type(
            "Float64",
            lambda x: Float(x, "Float64"),
            placeholders.get("Float64", None),
            required=None,
            description=None,
            enum=None,
            example=None,
            format=None,
        ),
        "Int": transform_type(
            "Int32",
            int,
            placeholders.get("Int", None),
            required=None,
            description=None,
            enum=None,
            example=None,
            format=None,
        ),
        "Integer": transform_type(
            "Int32",
            int,
            placeholders.get("Int32", None),
            required=None,
            description=None,
            enum=None,
            example=None,
            format=None,
        ),
        "Int8": transform_type(
            "Int8",
            lambda x: Integer(x, "Int8"),
            placeholders.get("Int8", None),
            required=None,
            description=None,
            enum=None,
            example=None,
            format=None,
        ),
        "Int16": transform_type(
            "Int16",
            lambda x: Integer(x, "Int16"),
            placeholders.get("Int16", None),
            required=None,
            description=None,
            enum=None,
            example=None,
            format=None,
        ),
        "Int32": transform_type(
            "Int32",
            lambda x: Integer(x, "Int32"),
            placeholders.get("Int32", None),
            required=None,
            description=None,
            enum=None,
            example=None,
            format=None,
        ),
        "Int64": transform_type(
            "Int64",
            lambda x: Integer(x, "Int64"),
            placeholders.get("Int64", None),
            required=None,
            description=None,
            enum=None,
            example=None,
            format=None,
        ),
        "Int128": transform_type(
            "Int128",
            lambda x: Integer(x, "Int128"),
            placeholders.get("Int128", None),
            required=None,
            description=None,
            enum=None,
            example=None,
            format=None,
        ),
        "Int256": transform_type(
            "Int256",
            lambda x: Integer(x, "Int256"),
            placeholders.get("Int256", None),
            required=None,
            description=None,
            enum=None,
            example=None,
            format=None,
        ),
        "UInt8": transform_type(
            "UInt8",
            lambda x: Integer(x, "UInt8"),
            placeholders.get("UInt8", None),
            required=None,
            description=None,
            enum=None,
            example=None,
            format=None,
        ),
        "UInt16": transform_type(
            "UInt16",
            lambda x: Integer(x, "UInt16"),
            placeholders.get("UInt16", None),
            required=None,
            description=None,
            enum=None,
            example=None,
            format=None,
        ),
        "UInt32": transform_type(
            "UInt32",
            lambda x: Integer(x, "UInt32"),
            placeholders.get("UInt32", None),
            required=None,
            description=None,
            enum=None,
            example=None,
            format=None,
        ),
        "UInt64": transform_type(
            "UInt64",
            lambda x: Integer(x, "UInt64"),
            placeholders.get("UInt64", None),
            required=None,
            description=None,
            enum=None,
            example=None,
            format=None,
        ),
        "UInt128": transform_type(
            "UInt128",
            lambda x: Integer(x, "UInt128"),
            placeholders.get("UInt128", None),
            required=None,
            description=None,
            enum=None,
            example=None,
            format=None,
        ),
        "UInt256": transform_type(
            "UInt256",
            lambda x: Integer(x, "UInt256"),
            placeholders.get("UInt256", None),
            required=None,
            description=None,
            enum=None,
            example=None,
            format=None,
        ),
        "Symbol": symbol,
        "Column": symbol,
        "String": transform_type(
            "String",
            str,
            placeholder="__no_value__",
            required=None,
            description=None,
            enum=None,
            example=None,
            format=None,
        ),
        "JSON": json_type,
    }

    types["Array"] = array_type(
        {
            "bool": boolean,
            "Boolean": boolean,
            "DateTime": transform_type(
                "DateTime",
                str,
                placeholders.get("DateTime", None),
                required=None,
                description=None,
                enum=None,
                example=None,
                format=None,
            ),
            "DateTime64": transform_type(
                "DateTime64",
                str,
                placeholders.get("DateTime64", None),
                required=None,
                description=None,
                enum=None,
                example=None,
                format=None,
            ),
            "Date": transform_type(
                "Date",
                str,
                placeholders.get("Date", None),
                required=None,
                description=None,
                enum=None,
                example=None,
                format=None,
            ),
            "Float32": transform_type(
                "Float32",
                float,
                placeholders.get("Float32", None),
                required=None,
                description=None,
                enum=None,
                example=None,
                format=None,
            ),
            "Float64": transform_type(
                "Float64",
                float,
                placeholders.get("Float64", None),
                required=None,
                description=None,
                enum=None,
                example=None,
                format=None,
            ),
            "Int": transform_type(
                "Int32",
                int,
                placeholders.get("Int", None),
                required=None,
                description=None,
                enum=None,
                example=None,
                format=None,
            ),
            "Integer": transform_type(
                "Int32",
                int,
                placeholders.get("Int32", None),
                required=None,
                description=None,
                enum=None,
                example=None,
                format=None,
            ),
            "Int8": transform_type(
                "Int8",
                int,
                placeholders.get("Int8", None),
                required=None,
                description=None,
                enum=None,
                example=None,
                format=None,
            ),
            "Int16": transform_type(
                "Int16",
                int,
                placeholders.get("Int16", None),
                required=None,
                description=None,
                enum=None,
                example=None,
                format=None,
            ),
            "Int32": transform_type(
                "Int32",
                int,
                placeholders.get("Int32", None),
                required=None,
                description=None,
                enum=None,
                example=None,
                format=None,
            ),
            "Int64": transform_type(
                "Int64",
                int,
                placeholders.get("Int64", None),
                required=None,
                description=None,
                enum=None,
                example=None,
                format=None,
            ),
            "Int128": transform_type(
                "Int128",
                int,
                placeholders.get("Int128", None),
                required=None,
                description=None,
                enum=None,
                example=None,
                format=None,
            ),
            "Int256": transform_type(
                "Int256",
                int,
                placeholders.get("Int256", None),
                required=None,
                description=None,
                enum=None,
                example=None,
                format=None,
            ),
            "UInt8": transform_type(
                "UInt8",
                int,
                placeholders.get("UInt8", None),
                required=None,
                description=None,
                enum=None,
                example=None,
                format=None,
            ),
            "UInt16": transform_type(
                "UInt16",
                int,
                placeholders.get("UInt16", None),
                required=None,
                description=None,
                enum=None,
                example=None,
                format=None,
            ),
            "UInt32": transform_type(
                "UInt32",
                int,
                placeholders.get("UInt32", None),
                required=None,
                description=None,
                enum=None,
                example=None,
                format=None,
            ),
            "UInt64": transform_type(
                "UInt64",
                int,
                placeholders.get("UInt64", None),
                required=None,
                description=None,
                enum=None,
                example=None,
                format=None,
            ),
            "UInt128": transform_type(
                "UInt128",
                int,
                placeholders.get("UInt128", None),
                required=None,
                description=None,
                enum=None,
                example=None,
                format=None,
            ),
            "UInt256": transform_type(
                "UInt256",
                int,
                placeholders.get("UInt256", None),
                required=None,
                description=None,
                enum=None,
                example=None,
                format=None,
            ),
            "Symbol": symbol,
            "Column": symbol,
            "String": transform_type(
                "String",
                str,
                placeholder="__no_value__",
                required=None,
                description=None,
                enum=None,
                example=None,
                format=None,
            ),
        }
    )

    types.update(function_list)
    return types


type_fns = get_transform_types()
type_fns_check = get_transform_types(
    {
        "DateTime64": "2019-01-01 00:00:00.000",
        "DateTime": "2019-01-01 00:00:00",
        "Date": "2019-01-01",
        "Float32": 0.0,
        "Float64": 0.0,
        "Int": 0,
        "Integer": 0,
        "UInt8": 0,
        "UInt16": 0,
        "UInt32": 0,
        "UInt64": 0,
        "UInt128": 0,
        "UInt256": 0,
        "Int8": 0,
        "Int16": 0,
        "Int32": 0,
        "Int64": 0,
        "Int128": 0,
        "Int256": 0,
        "Symbol": "symbol",
        "JSON": "{}",
    }
)


# from https://github.com/elouajib/sqlescapy/
# MIT license
def sqlescape_generator(translations):
    def sqlscape(str):
        return str.translate(str.maketrans(translations))

    return sqlscape


sqlescape = sqlescape_generator(
    {
        "\0": "\\0",
        "\r": "\\r",
        "\x08": "\\b",
        "\x09": "\\t",
        "\x1a": "\\z",
        "\n": "\\n",
        '"': "",
        "'": "\\'",
        "\\": "\\\\",
        "%": "\\%",
        "`": "\\`",
    }
)

# sqlescape_for_string_expression is only meant to be used when escaping
# string expressions (column=<string expression>) within SQL templates.
# This version includes a specific translation on top of the ones in the
# sqlescape above to escape double quotes (" will be translated into \")
# instead of removing them.
# It'll allow users to use parameter values with strings including double quotes.
sqlescape_for_string_expression = sqlescape_generator(
    {
        "\0": "\\0",
        "\r": "\\r",
        "\x08": "\\b",
        "\x09": "\\t",
        "\x1a": "\\z",
        "\n": "\\n",
        '"': '\\"',
        "'": "\\'",
        "\\": "\\\\",
        "%": "\\%",
        "`": "\\`",
    }
)


def escape_single_quote_str(s):
    return "'" + s.replace("'", "''") + "'"


def expression_wrapper(x, name, escape_arrays: bool = False):
    if type(x) in (unicode_type, bytes, str):
        return "'" + sqlescape_for_string_expression(x) + "'"
    elif isinstance(x, Placeholder):
        return "'__no_value__'"
    elif isinstance(x, Comment):
        return "-- {x} \n"
    if x is None:
        truncated_name = name[:20] + "..." if len(name) > 20 else name
        raise SQLTemplateException(
            f'expression "{truncated_name}" evaluated to null', documentation="/cli/advanced-templates.html"
        )
    if isinstance(x, list) and escape_arrays:
        logging.warning(f"expression_wrapper -> list :{x}:")

        try:
            result = (
                f"[{','.join(escape_single_quote_str(item) if isinstance(item, str) else str(item) for item in x)}]"
            )
            return result
        except Exception as e:
            logging.error(f"Error escaping array: {e}")
    return x


_namespace = {
    "column": column,
    "symbol": symbol,
    "error": error,
    "custom_error": custom_error,
    "_tt_utf8": escape.utf8,  # for internal use
    "_tt_string_types": (unicode_type, bytes),
    "xhtml_escape": lambda x: x,
    "expression_wrapper": expression_wrapper,
    # disable __builtins__ and some other functions
    # they raise a pretty non understandable error but if someone
    # is using them they know what they are trying to do
    # read https://anee.me/escaping-python-jails-849c65cf306e on how to escape from python jails
    "__buildins__": {},
    "__import__": {},
    "__debug__": {},
    "__doc__": {},
    "__name__": {},
    "__package__": {},
    "open": None,
    "close": None,
    "print": None,
    "input": None,
}


reserved_vars = set(["_tt_tmp", "_tt_append", "isinstance", "str", "error", "custom_error", *list(vars(builtins))])
for p in DEFAULT_PARAM_NAMES:  # we handle these in an specific manner
    reserved_vars.discard(p)  # `format` is part of builtins
# Allow 'id' to be used as a template parameter - https://gitlab.com/tinybird/analytics/-/issues/19119
reserved_vars.discard("id")
error_vars = ["error", "custom_error"]


def generate(self, **kwargs) -> Tuple[str, TemplateExecutionResults]:
    """Generate this template with the given arguments."""
    namespace = {}
    template_execution_results = TemplateExecutionResults()
    for key in kwargs.get("tb_secrets", []):
        # Avoid double-prefixing if the key already has the tb_secret_ prefix
        if is_secret_template_key(key):
            template_execution_results.add_template_param(key)
        else:
            template_execution_results.add_template_param(secret_template_key(key))

    if TB_SECRET_IN_TEST_MODE in kwargs:
        template_execution_results[TB_SECRET_IN_TEST_MODE] = None

    def set_tb_secret(x, default=None):
        try:
            key = secret_template_key(x)
            if key in template_execution_results.template_params:
                # secret available: Always use workspace secret regardless of test mode
                template_execution_results.add_ch_param(x)
                return Symbol("{" + sqlescape(x) + ": String}")
            else:
                # secret not available: Check test mode and defaults
                is_test_mode = TB_SECRET_IN_TEST_MODE in template_execution_results
                if default is not None:
                    # Use provided default value
                    return default
                elif is_test_mode:
                    # In test mode without default - return placeholder
                    return Symbol("{" + sqlescape(x) + ": String}")
                else:
                    # Not in test mode, no secret, no default - raise error
                    raise SQLTemplateException(
                        f"Cannot access secret '{x}'. Check the secret exists in the Workspace and the token has the required scope."
                    )
        except Exception:
            raise SQLTemplateException(
                f"Cannot access secret '{x}'. Check the secret exists in the Workspace and the token has the required scope."
            )

    def set_max_threads(x):
        try:
            template_execution_results["max_threads"] = int(x)
            return Expression(f"-- max_threads {x}\n")
        except Exception:
            return Expression(f"-- max_threads: wrong argument {x}\n")

    def set_backend_hint(hint):
        template_execution_results["backend_hint"] = str(hint)
        if hint is None or hint is False:
            template_execution_results["backend_hint"] = None
        return Expression(f"-- backend_hint {hint}\n")

    def set_cache_ttl(ttl_expression):
        valid_ttl_expressions = ("5s", "1m", "5m", "30m", "1h")
        if ttl_expression not in valid_ttl_expressions:
            raise SQLTemplateException(f"Invalid TTL cache expression, valid expressions are {valid_ttl_expressions}")
        template_execution_results["cache_ttl"] = ttl_expression
        return Expression(f"-- cache_ttl {ttl_expression}\n")

    def set_activate(feature):
        valid_features = ("analyzer", "parallel_replicas")
        if feature not in valid_features:
            raise SQLTemplateException(f"'{feature}' is not a valid 'activate' argument")
        template_execution_results["activate"] = feature
        return Expression(f"-- activate {feature}\n")

    def set_disable_feature(feature):
        valid_features = "analyzer"
        if feature not in valid_features:
            raise SQLTemplateException(f"'{feature}' is not a valid 'disable' argument")
        template_execution_results["disable"] = feature
        return Expression(f"-- disable {feature}\n")

    namespace.update(_namespace)
    # This is to fix the issue https://gitlab.com/tinybird/analytics/-/issues/19119
    # We need to do the override here because if we modify the _namespace, we would filter out parameters from the users
    namespace.update({"id": None})
    namespace.update(kwargs)
    namespace.update(
        {
            # __name__ and __loader__ allow the traceback mechanism to find
            # the generated source code.
            "__name__": self.name.replace(".", "_"),
            "__loader__": ObjectDict(get_source=lambda name: self.code),
            "max_threads": set_max_threads,
            "tb_secret": set_tb_secret,
            "tb_var": set_tb_secret,
            "backend_hint": set_backend_hint,
            "cache_ttl": set_cache_ttl,
            "activate": set_activate,
            "disable": set_disable_feature,
        }
    )

    exec_in(self.compiled, namespace)
    execute = namespace["_tt_execute"]
    # Clear the traceback module's cache of source data now that
    # we've generated a new template (mainly for this module's
    # unittests, where different tests reuse the same name).
    linecache.clearcache()

    try:
        return execute().decode(), template_execution_results
    except SQLTemplateCustomError as e:
        raise e
    except UnboundLocalError as e:
        try:
            message = getattr(e, "msg", str(e)).split("(<string>.generated.py")[0].strip()
            text = getattr(e, "text", message)
            line = None
            try:
                line = _STRING_LINE_NUMBER_RE.findall(text)
                message = _STRING_LINE_NUMBER_RE.sub("", message)
            except TypeError:
                pass

            if line:
                raise SQLTemplateException(f"{message.strip()} line {line[0]}")
            else:
                raise SQLTemplateException(f"{message.strip()}")
        except Exception as e:
            if isinstance(e, SQLTemplateException):
                raise e
            else:
                logging.exception(f"Error on unbound local error: {e}")
                raise ValueError(str(e))
    except TypeError as e:
        error = str(e)
        if "not supported between instances of 'Placeholder' and " in str(e):
            error = f"{str(e)}. If you are using a dynamic parameter, you need to wrap it around a valid Data Type (e.g. Int8(placeholder))"
        raise ValueError(error)
    except Exception as e:
        if "x" in namespace and namespace["x"] and hasattr(namespace["x"], "line") and namespace["x"].line:
            line = namespace["x"].line
            raise ValueError(f"{e}, line {line}")
        raise e


class CodeWriter:
    def __init__(self, file, template):
        self.file = file
        self.current_template = template
        self.apply_counter = 0
        self._indent = 0

    def indent_size(self):
        return self._indent

    def indent(self):
        class Indenter:
            def __enter__(_):
                self._indent += 1
                return self

            def __exit__(_, *args):
                assert self._indent > 0
                self._indent -= 1

        return Indenter()

    def write_line(self, line, line_number, indent=None):
        if indent is None:
            indent = self._indent
        line_comment = "  # %s:%d" % ("<generated>", line_number)
        print("    " * indent + line + line_comment, file=self.file)


def get_var_names(t: Template):
    """
    Extract variable names from a template.

    === BASIC EXPRESSIONS ===

    Simple variable reference:
    >>> get_var_names(Template("SELECT * FROM test WHERE id = {{my_var}}"))
    [{'line': 1, 'name': 'my_var'}]

    Multiple variables:
    >>> get_var_names(Template("SELECT {{a}}, {{b}} FROM test"))
    [{'line': 1, 'name': 'a'}, {'line': 1, 'name': 'b'}]

    No variables (static SQL):
    >>> get_var_names(Template("SELECT * FROM test"))
    []

    === TYPE CASTING FUNCTIONS ===

    Integer types:
    >>> [v['name'] for v in get_var_names(Template("{{Int8(a, 0)}} {{Int16(b, 0)}} {{Int32(c, 0)}} {{Int64(d, 0)}}"))]
    ['Int8', 'a', 'Int16', 'b', 'Int32', 'c', 'Int64', 'd']

    Unsigned integer types:
    >>> [v['name'] for v in get_var_names(Template("{{UInt8(a, 0)}} {{UInt32(b, 0)}} {{UInt64(c, 0)}}"))]
    ['UInt8', 'a', 'UInt32', 'b', 'UInt64', 'c']

    Float types:
    >>> [v['name'] for v in get_var_names(Template("{{Float32(price, 0.0)}} {{Float64(amount, 0.0)}}"))]
    ['Float32', 'price', 'Float64', 'amount']

    String type:
    >>> [v['name'] for v in get_var_names(Template("{{String(name, 'default')}}"))]
    ['String', 'name']

    Boolean type (False is in reserved_vars):
    >>> [v['name'] for v in get_var_names(Template("{{Boolean(flag, False)}}"))]
    ['Boolean', 'flag']

    Date/DateTime types:
    >>> [v['name'] for v in get_var_names(Template("{{Date(d)}} {{DateTime(dt)}} {{DateTime64(dt64)}}"))]
    ['Date', 'd', 'DateTime', 'dt', 'DateTime64', 'dt64']

    Array type:
    >>> [v['name'] for v in get_var_names(Template("{{Array(ids, 'Int32')}}"))]
    ['Array', 'ids']

    JSON type:
    >>> [v['name'] for v in get_var_names(Template("{{JSON(data, '{}')}}"))]
    ['JSON', 'data']

    === SQL SAFETY FUNCTIONS ===

    Column function (column is in _namespace, filtered out):
    >>> [v['name'] for v in get_var_names(Template("SELECT {{column(col_name, 'id')}} FROM t"))]
    ['col_name']

    Columns function:
    >>> [v['name'] for v in get_var_names(Template("SELECT {{columns(col_list, 'a,b')}} FROM t"))]
    ['columns', 'col_list']

    Symbol function (symbol is in _namespace, filtered out):
    >>> [v['name'] for v in get_var_names(Template("SELECT * FROM {{symbol(table_name)}}"))]
    ['table_name']

    Table function:
    >>> [v['name'] for v in get_var_names(Template("SELECT * FROM {{table(tbl)}}"))]
    ['table', 'tbl']

    === CONTROL FLOW - IF/ELIF/ELSE ===

    Simple if:
    >>> get_var_names(Template("{% if condition %}{{value}}{% end %}"))
    [{'line': 1, 'name': 'condition'}, {'line': 1, 'name': 'value'}]

    If with defined():
    >>> [v['name'] for v in get_var_names(Template("{% if defined(flag) %}WHERE x = 1{% end %}"))]
    ['defined', 'flag']

    If/else:
    >>> [v['name'] for v in get_var_names(Template("{% if cond %}{{a}}{% else %}{{b}}{% end %}"))]
    ['cond', 'a', 'b']

    If/elif/else chain:
    >>> [v['name'] for v in get_var_names(Template("{% if a %}1{% elif b %}2{% elif c %}3{% else %}4{% end %}"))]
    ['a', 'b', 'c']

    Nested if:
    >>> [v['name'] for v in get_var_names(Template("{% if outer %}{% if inner %}{{val}}{% end %}{% end %}"))]
    ['outer', 'inner', 'val']

    If with complex condition (co_names deduplicates names):
    >>> [v['name'] for v in get_var_names(Template("{% if defined(a) and defined(b) %}{{c}}{% end %}"))]
    ['defined', 'a', 'b', 'c']

    If with or condition:
    >>> [v['name'] for v in get_var_names(Template("{% if x or y %}{{z}}{% end %}"))]
    ['x', 'y', 'z']

    If with comparison:
    >>> [v['name'] for v in get_var_names(Template("{% if count > 0 %}{{result}}{% end %}"))]
    ['count', 'result']

    === CONTROL FLOW - FOR LOOPS ===

    Simple for loop:
    >>> [v['name'] for v in get_var_names(Template("{% for item in items %}{{item}}{% end %}"))]
    ['items', 'item', 'item']

    For with index (enumerate is in reserved_vars):
    >>> [v['name'] for v in get_var_names(Template("{% for i, item in enumerate(items) %}{{i}}:{{item}}{% end %}"))]
    ['items', 'i', 'item', 'i', 'item']

    Nested for loops:
    >>> [v['name'] for v in get_var_names(Template("{% for row in rows %}{% for col in cols %}{{row}}.{{col}}{% end %}{% end %}"))]
    ['rows', 'row', 'cols', 'col', 'row', 'col']

    For with split_to_array:
    >>> [v['name'] for v in get_var_names(Template("{% for x in split_to_array(csv_data) %}{{x}}{% end %}"))]
    ['split_to_array', 'csv_data', 'x', 'x']

    For with enumerate_with_last:
    >>> [v['name'] for v in get_var_names(Template("{% for is_last, item in enumerate_with_last(items) %}{{item}}{% end %}"))]
    ['enumerate_with_last', 'items', 'is_last', 'item', 'item']

    === CONTROL FLOW - WHILE ===

    Simple while:
    >>> [v['name'] for v in get_var_names(Template("{% while running %}{{counter}}{% end %}"))]
    ['running', 'counter']

    While with condition:
    >>> [v['name'] for v in get_var_names(Template("{% while count < max_count %}{{count}}{% end %}"))]
    ['count', 'max_count', 'count']

    === CONTROL FLOW - BREAK/CONTINUE ===

    Break inside for loop:
    >>> [v['name'] for v in get_var_names(Template("{% for x in items %}{% if x > limit %}{% break %}{% end %}{{x}}{% end %}"))]
    ['items', 'x', 'x', 'limit', 'x']

    Continue inside for loop:
    >>> [v['name'] for v in get_var_names(Template("{% for x in items %}{% if x < 0 %}{% continue %}{% end %}{{x}}{% end %}"))]
    ['items', 'x', 'x', 'x']

    Break inside while loop:
    >>> [v['name'] for v in get_var_names(Template("{% while running %}{% if done %}{% break %}{% end %}{{counter}}{% end %}"))]
    ['running', 'done', 'counter']

    === CONTROL FLOW - TRY/EXCEPT/FINALLY ===

    Simple try/except:
    >>> [v['name'] for v in get_var_names(Template("{% try %}{{risky}}{% except %}{{fallback}}{% end %}"))]
    ['risky', 'fallback']

    Try/except/finally:
    >>> [v['name'] for v in get_var_names(Template("{% try %}{{a}}{% except %}{{b}}{% finally %}{{c}}{% end %}"))]
    ['a', 'b', 'c']

    Except with type:
    >>> [v['name'] for v in get_var_names(Template("{% try %}{{a}}{% except MyError as e %}{{e}}{% end %}"))]
    ['a', 'MyError', 'e', 'e']

    === SET STATEMENTS ===

    Simple set:
    >>> [v['name'] for v in get_var_names(Template("{% set x = myvar + 1 %}{{x}}"))]
    ['myvar', 'x', 'x']

    Set with expression:
    >>> [v['name'] for v in get_var_names(Template("{% set total = a + b * c %}{{total}}"))]
    ['a', 'b', 'c', 'total', 'total']

    Set with function call:
    >>> [v['name'] for v in get_var_names(Template("{% set items = list_data %}{{items}}"))]
    ['list_data', 'items', 'items']

    Set with template expression (skipped - contains {{}}):
    >>> [v['name'] for v in get_var_names(Template("{% set x = {{String(y)}} %}{{x}}"))]
    ['x']

    === UTILITY FUNCTIONS ===

    Defined function:
    >>> [v['name'] for v in get_var_names(Template("{% if defined(param) %}yes{% end %}"))]
    ['defined', 'param']

    Error function (error is in _namespace, filtered out):
    >>> [v['name'] for v in get_var_names(Template("{% if not valid %}{{error('Invalid input')}}{% end %}"))]
    ['valid']

    Custom error (custom_error is in _namespace, filtered out):
    >>> get_var_names(Template("{{custom_error('Not found', 404)}}"))
    []

    === DATE FUNCTIONS ===

    Day diff:
    >>> [v['name'] for v in get_var_names(Template("{{day_diff(start_date, end_date)}}"))]
    ['day_diff', 'start_date', 'end_date']

    Date diff in days:
    >>> [v['name'] for v in get_var_names(Template("{{date_diff_in_days(d1, d2)}}"))]
    ['date_diff_in_days', 'd1', 'd2']

    Date diff in hours:
    >>> [v['name'] for v in get_var_names(Template("{{date_diff_in_hours(t1, t2)}}"))]
    ['date_diff_in_hours', 't1', 't2']

    === RUNTIME CONFIGURATION ===

    Max threads:
    >>> [v['name'] for v in get_var_names(Template("{{max_threads(num_threads)}}"))]
    ['max_threads', 'num_threads']

    TB secret:
    >>> [v['name'] for v in get_var_names(Template("{{tb_secret(secret_name)}}"))]
    ['tb_secret', 'secret_name']

    Cache TTL:
    >>> [v['name'] for v in get_var_names(Template("{{cache_ttl(ttl_value)}}"))]
    ['cache_ttl', 'ttl_value']

    === COMPLEX NESTED COMBINATIONS ===

    If inside for:
    >>> [v['name'] for v in get_var_names(Template("{% for item in items %}{% if defined(item) %}{{item}}{% end %}{% end %}"))]
    ['items', 'item', 'defined', 'item', 'item']

    For inside if:
    >>> [v['name'] for v in get_var_names(Template("{% if show_list %}{% for x in data %}{{x}}{% end %}{% end %}"))]
    ['show_list', 'data', 'x', 'x']

    Set + if + for (max, range are in reserved_vars):
    >>> [v['name'] for v in get_var_names(Template("{% set limit = max_val %}{% if limit > 0 %}{% for i in my_range(limit) %}{{i}}{% end %}{% end %}"))]
    ['max_val', 'limit', 'limit', 'my_range', 'limit', 'i', 'i']

    Multiple control blocks:
    >>> t = Template("{% set limit = max_rows %}{% if flag %}{% for c in cols %}{{c}}{% end %}{% elif other %}{{x}}{% else %}{{y}}{% end %}{% while more %}{{b}}{% end %}{% try %}{{q}}{% except E as e %}{{e}}{% finally %}{{done}}{% end %}")
    >>> [v['name'] for v in get_var_names(t)]
    ['max_rows', 'limit', 'flag', 'cols', 'c', 'c', 'other', 'x', 'y', 'more', 'b', 'q', 'E', 'e', 'e', 'done']

    Type casting inside control block:
    >>> [v['name'] for v in get_var_names(Template("{% if defined(id) %}WHERE id = {{Int32(id, 0)}}{% end %}"))]
    ['defined', 'id', 'Int32', 'id']

    Column + Array combination (column is in _namespace):
    >>> [v['name'] for v in get_var_names(Template("SELECT {{column(col)}} FROM t WHERE id IN {{Array(ids, 'Int32')}}"))]
    ['col', 'Array', 'ids']

    Deeply nested structure:
    >>> t = Template("{% if a %}{% for x in items %}{% if defined(x) %}{% try %}{{Int32(x, 0)}}{% except %}{{default}}{% end %}{% end %}{% end %}{% end %}")
    >>> [v['name'] for v in get_var_names(t)]
    ['a', 'items', 'x', 'defined', 'x', 'Int32', 'x', 'default']

    === FUNCTIONS WITH CONTROL FLOW ===

    Date function inside if:
    >>> [v['name'] for v in get_var_names(Template("{% if defined(start) %}{{date_diff_in_days(start, end)}}{% end %}"))]
    ['defined', 'start', 'date_diff_in_days', 'start', 'end']

    Date function inside for:
    >>> [v['name'] for v in get_var_names(Template("{% for d in dates %}{{day_diff(d, today)}}{% end %}"))]
    ['dates', 'd', 'day_diff', 'd', 'today']

    Date function with elif:
    >>> t = Template("{% if mode == 'days' %}{{date_diff_in_days(t1, t2)}}{% elif mode == 'hours' %}{{date_diff_in_hours(t1, t2)}}{% end %}")
    >>> [v['name'] for v in get_var_names(t)]
    ['mode', 'date_diff_in_days', 't1', 't2', 'mode', 'date_diff_in_hours', 't1', 't2']

    Set with date function:
    >>> [v['name'] for v in get_var_names(Template("{% set diff = day_diff(start_date, end_date) %}{{diff}}"))]
    ['day_diff', 'start_date', 'end_date', 'diff', 'diff']

    Max threads inside if:
    >>> [v['name'] for v in get_var_names(Template("{% if parallel %}{{max_threads(thread_count)}}{% end %}"))]
    ['parallel', 'max_threads', 'thread_count']

    Cache TTL with condition:
    >>> [v['name'] for v in get_var_names(Template("{% if use_cache %}{{cache_ttl(ttl)}}{% else %}{{cache_ttl(0)}}{% end %}"))]
    ['use_cache', 'cache_ttl', 'ttl', 'cache_ttl']

    TB secret inside for:
    >>> [v['name'] for v in get_var_names(Template("{% for name in secret_names %}{{tb_secret(name)}}{% end %}"))]
    ['secret_names', 'name', 'tb_secret', 'name']

    Type casting with date function in elif chain (type is in reserved_vars):
    >>> t = Template("{% if kind == 'int' %}{{Int32(val, 0)}}{% elif kind == 'date' %}{{DateTime(val)}}{% elif kind == 'diff' %}{{date_diff_in_days(val, now)}}{% end %}")
    >>> [v['name'] for v in get_var_names(t)]
    ['kind', 'Int32', 'val', 'kind', 'DateTime', 'val', 'kind', 'date_diff_in_days', 'val', 'now']

    While with date function:
    >>> [v['name'] for v in get_var_names(Template("{% while day_diff(current, target) > 0 %}{{current}}{% end %}"))]
    ['day_diff', 'current', 'target', 'current']

    Try/except with date function:
    >>> [v['name'] for v in get_var_names(Template("{% try %}{{date_diff_in_hours(t1, t2)}}{% except %}{{default_hours}}{% end %}"))]
    ['date_diff_in_hours', 't1', 't2', 'default_hours']

    Complex: set + for + if + multiple functions:
    >>> t = Template("{% set threshold = Int32(max_days, 30) %}{% for d in dates %}{% if day_diff(d, now) < threshold %}{{String(d)}}{% end %}{% end %}")
    >>> [v['name'] for v in get_var_names(t)]
    ['Int32', 'max_days', 'threshold', 'dates', 'd', 'day_diff', 'd', 'now', 'threshold', 'String', 'd']

    Nested control blocks with config functions:
    >>> t = Template("{% if enabled %}{% set threads = max_threads(n) %}{% for i in tasks %}{{cache_ttl(ttl)}}{% end %}{% end %}")
    >>> [v['name'] for v in get_var_names(t)]
    ['enabled', 'max_threads', 'n', 'threads', 'tasks', 'i', 'cache_ttl', 'ttl']

    === MULTILINE TEMPLATES ===

    Multiline with mixed constructs (column is in _namespace, filter is in reserved_vars):
    >>> t = Template('''
    ... SELECT
    ...     {{column(col1)}},
    ...     {{column(col2)}}
    ... FROM {{table(tbl)}}
    ... {% if defined(flag) %}
    ... WHERE {{column(filter_col)}} = {{String(value)}}
    ... {% end %}
    ... ''')
    >>> sorted(set(v['name'] for v in get_var_names(t)))
    ['String', 'col1', 'col2', 'defined', 'filter_col', 'flag', 'table', 'tbl', 'value']
    """
    try:
        # Recursive helper that traverses the template's parsed chunks and collects variable names.
        # The template is parsed into a tree of chunks: _ChunkList (container), _Expression ({{...}}),
        # and _ControlBlock ({% if %}, {% for %}, etc.).
        #
        # We use compile() to extract variable names because Python's compiler automatically
        # collects all referenced names into the code object's co_names attribute. This avoids
        # manually parsing Python expressions with the ast module. For example:
        #   compile("Int32(num_val, 0)", ...).co_names → ('Int32', 'num_val')
        def _n(chunks, v):
            for x in chunks:
                line_number = x.line

                if type(x).__name__ == "_ChunkList":
                    # Container node: recurse into its children
                    _n(x.chunks, v)

                elif type(x).__name__ == "_Expression":
                    # Simple template expression like {{my_var}} or {{Int32(num_val, 0)}}
                    # Compile the expression to bytecode and extract all referenced names
                    # from co_names (the tuple of names used by the bytecode)
                    c = compile(x.expression, "<string>", "exec", dont_inherit=True)
                    # Filter out internal namespace functions and reserved variable names
                    variable_names = [x for x in c.co_names if x not in _namespace and x not in reserved_vars]
                    v += list(map(lambda variable: {"line": line_number, "name": variable}, variable_names))

                elif type(x).__name__ == "_ControlBlock":
                    # Control structure like {% if cond %}, {% for item in items %}, {% try %}, etc.
                    # Compile only the statement (condition/iterator), not the full generated code.
                    # This avoids compiling large SQL literals in the body, which is expensive
                    # and unnecessary since we only need variable names from the condition.
                    # Note: "try" has no condition/expression, so skip compilation for it.
                    if x.statement != "try":
                        statement_code = x.statement + ": pass"
                        c = compile(statement_code, "<string>", "exec", dont_inherit=True)
                        variable_names = [x for x in c.co_names if x not in _namespace and x not in reserved_vars]
                        v += list(map(lambda variable: {"line": line_number, "name": variable}, variable_names))

                    # Recurse into the body of the control block to find nested expressions
                    _n(x.body.chunks, v)

                elif type(x).__name__ == "_IntermediateControlBlock":
                    # Intermediate control structure like {% elif cond %}, {% else %}, {% except %}, etc.
                    # These appear inside _ControlBlock bodies.
                    # For "else"/"finally", there's no condition to extract.
                    # For "elif cond", we need to extract variables from the condition.
                    # For "except Type as e", we need to extract the exception type and alias.
                    # Note: "elif"/"except" aren't valid Python on their own, so we wrap them.
                    if x.statement.startswith("elif "):
                        statement_code = "if False: pass\n" + x.statement + ": pass"
                        c = compile(statement_code, "<string>", "exec", dont_inherit=True)
                        variable_names = [x for x in c.co_names if x not in _namespace and x not in reserved_vars]
                        v += list(map(lambda variable: {"line": line_number, "name": variable}, variable_names))

                    elif x.statement.startswith("except "):
                        # "except Exception" or "except Exception as e"
                        statement_code = "try: pass\n" + x.statement + ": pass"
                        c = compile(statement_code, "<string>", "exec", dont_inherit=True)
                        variable_names = [x for x in c.co_names if x not in _namespace and x not in reserved_vars]
                        v += list(map(lambda variable: {"line": line_number, "name": variable}, variable_names))

                elif (
                    type(x).__name__ == "_Statement"
                    and x.statement not in ("break", "continue")  # No variables; fail compile() outside loop
                    and "{{" not in x.statement  # Skip template expressions which aren't valid Python
                ):
                    # Statement like {% set x = expr %}
                    c = compile(x.statement, "<string>", "exec", dont_inherit=True)
                    variable_names = [x for x in c.co_names if x not in _namespace and x not in reserved_vars]
                    v += list(map(lambda variable: {"line": line_number, "name": variable}, variable_names))

        var: list[dict[str, Any]] = []
        # Start traversal from the root of the parsed template
        _n(t.file.body.chunks, var)
        return var
    except SecurityException as e:
        raise SQLTemplateException(e)


def get_var_data(content, node_id=None):
    """Extract variable data from a template expression.

    Optimized to use a single AST traversal instead of two separate walks.
    """

    def node_to_value(x):
        if type(x) in (ast.Bytes, ast.Str):
            return x.s
        elif type(x) == ast.Num:  # noqa: E721
            return x.n
        elif type(x) == ast.NameConstant:  # noqa: E721
            return x.value
        elif type(x) == ast.Name:  # noqa: E721
            return x.id
        elif type(x) == ast.List:  # noqa: E721
            # List can hold different types
            return _get_list_var_data(x)
        elif type(x) == ast.BinOp:  # noqa: E721
            # in this case there could be several variables
            # if that's the case the left one is the main
            r = node_to_value(x.left)
            if not r:
                r = node_to_value(x.right)
            return r
        elif type(x) == ast.Constant:  # noqa: E721
            return x.value
        elif type(x) == ast.UnaryOp and type(x.operand) == ast.Constant:  # noqa: E721
            if type(x.op) == ast.USub:  # noqa: E721
                return x.operand.value * -1
            else:
                return x.operand.value
        else:
            try:
                return x.id
            except Exception:
                # don't let this ruin the parsing
                pass
        return None

    def _get_list_var_data(x):
        if not x.elts:
            return []

        first_elem = x.elts[0]
        if type(first_elem) in (ast.Bytes, ast.Str):
            return [elem.s for elem in x.elts]
        elif type(first_elem) == ast.Num:  # noqa: E721
            return [elem.n for elem in x.elts]
        elif type(first_elem) == ast.NameConstant or type(first_elem) == ast.Constant:  # noqa: E721
            return [elem.value for elem in x.elts]
        elif type(first_elem) == ast.Name:  # noqa: E721
            return [elem.id for elem in x.elts]

        return []

    def check_default_value(value):
        if isinstance(value, int):
            MAX_SAFE_INTEGER = 9007199254740991
            if value > MAX_SAFE_INTEGER:
                return str(value)
        return value

    # TODO: Remove this retry logic. It was added in commit 1314a3b120 as a workaround for
    # a Python 3.11 bug (https://github.com/python/cpython/issues/106905) where AST recursion
    # depth tracking was broken. However, retrying doesn't actually help since the corrupted
    # state persists. The bug was fixed in Python 3.11.5+ and 3.12+, so this can be simplified
    # to just `ast.parse(content)` once we confirm all environments use patched Python versions.
    def parse_content(content, retries=0):
        try:
            parsed = ast.parse(content)
            return parsed
        except Exception as e:
            if "AST constructor recursion depth mismatch" not in str(e):
                raise e
            retries += 1
            if retries > 3:
                raise e
            return parse_content(content, retries)

    parsed = parse_content(content)
    vars = {}

    # Single pass: traverse AST while tracking parent references
    # Use FIFO order so children are visited in source order (matching ast.walk)
    # because first-seen wins for duplicates like defined(a) and defined(a, ...) later.
    queue = deque([(parsed, None)])  # (node, parent)
    while queue:
        node, parent = queue.popleft()

        if isinstance(node, ast.Call) and isinstance(node.func, ast.Name):
            try:
                func = node.func.id
                # parse function args
                args = []
                for x in node.args:
                    if type(x) == ast.Call:  # noqa: E721
                        # Nested calls are traversed via `ast.iter_child_nodes` below.
                        continue
                    else:
                        args.append(node_to_value(x))

                kwargs = {}
                for x in node.keywords:
                    value = node_to_value(x.value)
                    kwargs[x.arg] = value
                    if x.arg == "default":
                        kwargs["default"] = check_default_value(value)
                if func in VALID_CUSTOM_FUNCTION_NAMES:
                    # Type definition here is set to 'String' because it comes from a
                    # `defined(variable)` expression that does not contain any type hint.
                    # It will be overriden in later definitions or left as is otherwise.
                    # args[0] check is used to avoid adding unnamed parameters found in
                    # templates like: `split_to_array('')`
                    if args and isinstance(args[0], list):
                        raise ValueError(f'"{args[0]}" can not be used as a variable name')
                    if len(args) > 0 and args[0] not in vars and args[0]:
                        vars[args[0]] = {
                            "type": "String",
                            "default": None,
                            "used_in": "function_call",
                        }
                elif func == "Array":
                    if "default" not in kwargs:
                        default = kwargs.get("default", args[2] if len(args) > 2 and args[2] else None)
                        kwargs["default"] = check_default_value(default)
                    if args:
                        if isinstance(args[0], list):
                            raise ValueError(f'"{args[0]}" can not be used as a variable name')
                        vars[args[0]] = {
                            "type": f"Array({args[1]})" if len(args) > 1 else "Array(String)",
                            **kwargs,
                        }
                elif func in parameter_types:
                    # avoid variable names to be None
                    if args and args[0] is not None:
                        # if this is a cast use the function name to get the type
                        if "default" not in kwargs:
                            default = kwargs.get("default", args[1] if len(args) > 1 else None)
                            kwargs["default"] = check_default_value(default)
                        try:
                            if isinstance(args[0], list):
                                raise ValueError(f'"{args[0]}" can not be used as a variable name')
                            vars[args[0]] = {"type": func, **kwargs}
                            if "default" in kwargs:
                                kwargs["default"] = check_default_value(kwargs["default"])
                        except TypeError as e:
                            logging.exception(f"pipe parsing problem {content} (node '{node_id}'):  {e}")
            except ValueError:
                raise
            except Exception as e:
                # if we find a problem parsing, let the parsing continue
                logging.exception(f"pipe parsing problem {content} (node: '{node_id}'):  {e}")
        elif isinstance(node, ast.Name):
            # when parent node is a call it means it's managed by the Call workflow (see above)
            is_cast = (
                isinstance(parent, ast.Call) and isinstance(parent.func, ast.Name) and parent.func.id in parameter_types
            )
            is_reserved_name = node.id in reserved_vars or node.id in function_list or node.id in _namespace
            if (not isinstance(parent, ast.Call) and not is_cast) and not is_reserved_name:
                vars[node.id] = {"type": "String", "default": None}

        # Add children preserving source order so downstream precedence matches templates
        for child in ast.iter_child_nodes(node):
            queue.append((child, node))

    return [dict(name=k, **v) for k, v in vars.items()]


def get_var_names_and_types(t, node_id=None):
    """
    >>> get_var_names_and_types(Template("SELECT * FROM filter_value WHERE description = {{Float32(with_value, 0.0)}}"))
    [{'name': 'with_value', 'type': 'Float32', 'default': 0.0}]
    >>> get_var_names_and_types(Template("SELECT * FROM filter_value WHERE description = {{Float32(with_value, -0.0)}}"))
    [{'name': 'with_value', 'type': 'Float32', 'default': -0.0}]
    >>> get_var_names_and_types(Template("SELECT * FROM filter_value WHERE description = {{Int32(with_value, 0)}}"))
    [{'name': 'with_value', 'type': 'Int32', 'default': 0}]
    >>> get_var_names_and_types(Template("SELECT * FROM filter_value WHERE description = {{Int32(with_value, -0)}}"))
    [{'name': 'with_value', 'type': 'Int32', 'default': 0}]
    >>> get_var_names_and_types(Template("SELECT * FROM filter_value WHERE description = {{Float32(with_value, -0.1)}}"))
    [{'name': 'with_value', 'type': 'Float32', 'default': -0.1}]
    >>> get_var_names_and_types(Template("SELECT * FROM filter_value WHERE description = {{Float32(with_value, 0.1)}}"))
    [{'name': 'with_value', 'type': 'Float32', 'default': 0.1}]
    >>> get_var_names_and_types(Template("SELECT * FROM filter_value WHERE description = {{String(d, 'test_1')}} AND value = {{Int8(v, 3)}}"))
    [{'name': 'd', 'type': 'String', 'default': 'test_1'}, {'name': 'v', 'type': 'Int8', 'default': 3}]
    >>> get_var_names_and_types(Template("select * from test {% if defined(number_variable) %} where number_variable = {{UInt64(number_variable)}} {% end %}"))
    [{'name': 'number_variable', 'type': 'UInt64', 'default': None}]
    >>> get_var_names_and_types(Template("select * from test {% if defined({{UInt64(number_variable)}}) %} where 1 {% end %}"))
    [{'name': 'number_variable', 'type': 'UInt64', 'default': None}]
    >>> get_var_names_and_types(Template("select * from test {% if defined(testing) and defined(testing2) %} where 1 {%end %}"))
    [{'name': 'testing', 'type': 'String', 'default': None, 'used_in': 'function_call'}, {'name': 'testing2', 'type': 'String', 'default': None, 'used_in': 'function_call'}]
    >>> get_var_names_and_types(Template("select * from test {% if defined({{UInt64(number_variable)}}) %} where 1 {% end %}"))
    [{'name': 'number_variable', 'type': 'UInt64', 'default': None}]
    >>> get_var_names_and_types(Template("select * from test {% if defined({{UInt64(x)}}) and defined(y) %} where 1 {% end %}"))
    [{'name': 'x', 'type': 'UInt64', 'default': None}, {'name': 'y', 'type': 'String', 'default': None, 'used_in': 'function_call'}]
    >>> get_var_names_and_types(Template("select * from test {% if defined(y) and x > 0 %} where x = {{UInt64(x)}} {% end %}"))
    [{'name': 'x', 'type': 'UInt64', 'default': None}, {'name': 'y', 'type': 'String', 'default': None, 'used_in': 'function_call'}]
    >>> get_var_names_and_types(Template("select * from test {% if defined(y) and defined(z) and x > 0 %} {{UInt64(x)}} {% end %}"))
    [{'name': 'x', 'type': 'UInt64', 'default': None}, {'name': 'y', 'type': 'String', 'default': None, 'used_in': 'function_call'}, {'name': 'z', 'type': 'String', 'default': None, 'used_in': 'function_call'}]
    >>> get_var_names_and_types(Template("{% if '{{' in marker %}{{UInt64(x)}}{% end %}"))
    [{'name': 'x', 'type': 'UInt64', 'default': None}, {'name': 'marker', 'type': 'String', 'default': None}]
    >>> get_var_names_and_types(Template("{% if a %}1{% elif defined(y) %}{{UInt64(x)}}{% end %}"))
    [{'name': 'y', 'type': 'String', 'default': None, 'used_in': 'function_call'}, {'name': 'x', 'type': 'UInt64', 'default': None}, {'name': 'a', 'type': 'String', 'default': None}]
    >>> get_var_names_and_types(Template("{% for x in items %}{{Int32(x, 0)}}{% end %}"))
    [{'name': 'x', 'type': 'Int32', 'default': 0}, {'name': 'items', 'type': 'String', 'default': None}]
    >>> get_var_names_and_types(Template("{% while more %}{{UInt64(x)}}{% end %}"))
    [{'name': 'x', 'type': 'UInt64', 'default': None}, {'name': 'more', 'type': 'String', 'default': None}]
    >>> get_var_names_and_types(Template("{% set a = Int32(x, 0) %}"))
    [{'name': 'a', 'type': 'String', 'default': None}, {'name': 'x', 'type': 'Int32', 'default': 0}]
    >>> get_var_names_and_types(Template("{% try %}{{UInt64(x)}}{% except E as e %}{{e}}{% end %}"))
    [{'name': 'x', 'type': 'UInt64', 'default': None}, {'name': 'E', 'type': 'String', 'default': None}, {'name': 'e', 'type': 'String', 'default': None}]
    >>> get_var_names_and_types(Template("select {{Array(cod_stock_source_type,'Int16', defined=False)}}"))
    [{'name': 'cod_stock_source_type', 'type': 'Array(Int16)', 'defined': False, 'default': None}]
    >>> get_var_names_and_types(Template("select {{Array(cod_stock_source_type, defined=False)}}"))
    [{'name': 'cod_stock_source_type', 'type': 'Array(String)', 'defined': False, 'default': None}]
    >>> get_var_names_and_types(Template("select {{cod_stock_source_type}}"))
    [{'name': 'cod_stock_source_type', 'type': 'String', 'default': None}]
    >>> get_var_names_and_types(Template("SELECT {{len([1] * 10**7)}}"))
    Traceback (most recent call last):
    ...
    tinybird.tornado_template.SecurityException: Invalid BinOp: Pow()
    >>> get_var_names_and_types(Template("select {{String(cod_stock_source_type, 'test')}}"))
    [{'name': 'cod_stock_source_type', 'type': 'String', 'default': 'test'}]
    >>> get_var_names_and_types(Template("select {{split_to_array(test)}}"))
    [{'name': 'test', 'type': 'String', 'default': None, 'used_in': 'function_call'}]
    >>> get_var_names_and_types(Template("select {{String(test + 'abcd', 'default_value')}}"))
    [{'name': 'test', 'type': 'String', 'default': None}]
    >>> get_var_names_and_types(Template("SELECT * FROM filter_value WHERE description = {{String(d, 'test_1', description='test', required=True)}} AND value = {{Int8(v, 3, format='number', example='1')}}"))
    [{'name': 'd', 'type': 'String', 'description': 'test', 'required': True, 'default': 'test_1'}, {'name': 'v', 'type': 'Int8', 'format': 'number', 'example': '1', 'default': 3}]
    >>> get_var_names_and_types(Template("SELECT * FROM filter_value WHERE description = {{String(d, default='test_1', description='test')}}"))
    [{'name': 'd', 'type': 'String', 'default': 'test_1', 'description': 'test'}]
    >>> get_var_names_and_types(Template("select {{Array(cod_stock_source_type, 'Int16', default='1', defined=False)}}"))
    [{'name': 'cod_stock_source_type', 'type': 'Array(Int16)', 'default': '1', 'defined': False}]
    >>> get_var_names_and_types(Template('select {{symbol(split_to_array(attr, "amount_net")[0] + "_intermediate" )}}'))
    [{'name': 'attr', 'type': 'String', 'default': None, 'used_in': 'function_call'}]
    >>> get_var_names_and_types(Template("SELECT * FROM filter_value WHERE description = {{Float32(with_value, 0.1)}} AND description = {{Float32(zero, 0)}} AND value = {{Float32(no_default)}}"))
    [{'name': 'with_value', 'type': 'Float32', 'default': 0.1}, {'name': 'zero', 'type': 'Float32', 'default': 0}, {'name': 'no_default', 'type': 'Float32', 'default': None}]
    >>> get_var_names_and_types(Template("SELECT * FROM filter_value WHERE description = {{Float32(with_value, -0.1)}} AND description = {{Float32(zero, 0)}} AND value = {{Float32(no_default)}}"))
    [{'name': 'with_value', 'type': 'Float32', 'default': -0.1}, {'name': 'zero', 'type': 'Float32', 'default': 0}, {'name': 'no_default', 'type': 'Float32', 'default': None}]
    >>> get_var_names_and_types(Template('''SELECT * FROM abcd WHERE hotel_id <> 0 {% if defined(date_from) %} AND script_created_at > {{DateTime(date_from, '2020-09-09 10:10:10', description="This is a description", required=True)(date_from, '2020-09-09', description="Filter script alert creation date", required=False)}} {% end %}'''))
    [{'name': 'date_from', 'type': 'DateTime', 'description': 'This is a description', 'required': True, 'default': '2020-09-09 10:10:10'}]
    >>> get_var_names_and_types(Template("SELECT * FROM filter_value WHERE symbol = {{Int128(symbol_id, 11111, description='Symbol Id', required=True)}} AND user = {{Int256(user_id, 3555, description='User Id')}}"))
    [{'name': 'symbol_id', 'type': 'Int128', 'description': 'Symbol Id', 'required': True, 'default': 11111}, {'name': 'user_id', 'type': 'Int256', 'description': 'User Id', 'default': 3555}]
    >>> get_var_names_and_types(Template("SELECT now() > {{DateTime64(timestamp, '2020-09-09 10:10:10.000')}}"))
    [{'name': 'timestamp', 'type': 'DateTime64', 'default': '2020-09-09 10:10:10.000'}]
    >>> get_var_names_and_types(Template("select {{Int32(dup, 1)}} {{String(dup, 'last')}}"))
    [{'name': 'dup', 'type': 'Int32', 'default': 1}, {'name': 'dup', 'type': 'String', 'default': 'last'}]
    >>> get_var_names_and_types(Template("SELECT * FROM filter_value WHERE symbol = {{Int64(symbol_id, 9223372036854775807)}}"))
    [{'name': 'symbol_id', 'type': 'Int64', 'default': '9223372036854775807'}]
    """
    try:
        # Recursive helper that traverses the template's parsed chunks and collects
        # variable data including types and defaults.
        #
        # Optimization: Instead of calling x.generate(writer) which generates full
        # Python code (expensive for large templates), we parse just the statement
        # and recurse into the body separately.
        #
        # Backward compatibility: The original implementation called get_var_data on
        # full generated code, where type functions (Array, Int32, etc.) would overwrite
        # variables from defined(). To maintain this behavior, we process body expressions
        # FIRST to get types, then only add variables from control statements if they
        # weren't already found in body expressions.

        # Track variable names seen from EXPRESSIONS (not control statements)
        # Used to prevent control statements from adding variables with wrong types
        typed_names: set[str] = set()

        statement_wraps: dict[str, Callable[[str], str]] = {
            "control": lambda statement: f"{statement}: pass",
            "elif": lambda statement: "if False: pass\n" + statement + ": pass",
            "except": lambda statement: "try: pass\n" + statement + ": pass",
        }

        def parse_statement_code_if_new_vars(statement_code: str, *, skip_names: set[str]) -> list[dict[str, Any]]:
            """Parse statement code and return variables not already typed by expressions."""
            try:
                # `{{...}}` inside `{% ... %}` is not template syntax in Tornado templates.
                # It becomes Python braces (e.g. set literals), which can hide unrelated vars
                # in the statement. Strip them so we still parse the rest of the statement.
                if "{{" in statement_code and "}}" in statement_code:
                    statement_code = _EMBEDDED_TEMPLATE_EXPRESSION_RE.sub("None", statement_code)
                var_data = get_var_data(statement_code, node_id=node_id)
            except Exception:
                return []

            if not var_data:
                return []

            return [vd for vd in var_data if vd["name"] not in skip_names]

        def parse_statement(statement: str, *, wrap: Callable[[str], str] | None) -> list[dict[str, Any]]:
            """Parse a statement and return variables not already typed by expressions."""
            statement_expr_names: set[str] = set()
            vars_out: list[dict[str, Any]] = []

            if "{{" in statement and "}}" in statement:
                for match in _EMBEDDED_TEMPLATE_EXPRESSION_RE.finditer(statement):
                    expr = match.group(1).strip()
                    if not expr:
                        continue
                    try:
                        var_data = get_var_data(expr, node_id=node_id)
                    except Exception:
                        continue

                    if not var_data:
                        continue

                    for vd in var_data:
                        # Body expressions win for types; do not add statement-derived typed vars.
                        if vd["name"] in typed_names:
                            continue
                        statement_expr_names.add(vd["name"])
                        vars_out.append(vd)

            statement_code = wrap(statement) if wrap else statement
            vars_out.extend(
                parse_statement_code_if_new_vars(statement_code, skip_names=typed_names | statement_expr_names)
            )
            return vars_out

        def _n(chunks: list, vars_out: list[dict[str, Any]]) -> None:
            for x in chunks:
                kind = type(x).__name__
                match kind:
                    case "_ChunkList":
                        _n(x.chunks, vars_out)

                    case "_Expression":
                        # Template expression like {{Int32(num_val, 0)}} - extract type info
                        var_data = get_var_data(x.expression, node_id=node_id)
                        if var_data:
                            typed_names.update(vd["name"] for vd in var_data)
                            vars_out.extend(var_data)

                    case "_ControlBlock":
                        # Process body FIRST to get type functions
                        _n(x.body.chunks, vars_out)
                        # Then parse statement, but only add vars NOT already typed by expressions
                        if x.statement != "try":
                            vars_out.extend(parse_statement(x.statement, wrap=statement_wraps["control"]))

                    case "_IntermediateControlBlock":
                        # elif/else/except/finally - only add vars not typed by expressions
                        if x.statement.startswith("elif "):
                            vars_out.extend(parse_statement(x.statement, wrap=statement_wraps["elif"]))
                        elif x.statement.startswith("except "):
                            vars_out.extend(parse_statement(x.statement, wrap=statement_wraps["except"]))

                    case "_Statement":
                        # {% set x = ... %}, {% break %}, etc.
                        if x.statement not in ("break", "continue") and "{{" not in x.statement:
                            vars_out.extend(parse_statement(x.statement, wrap=None))

        vars_out: list[dict[str, Any]] = []
        _n(t.file.body.chunks, vars_out)
        return vars_out
    except SecurityException as e:
        raise SQLTemplateException(e)


@lru_cache(maxsize=2**10)
def get_var_names_and_types_cached(t: Template):
    return get_var_names_and_types(t)


def wrap_vars(t, escape_arrays: bool = False):
    def _n(chunks, v):
        for x in chunks:
            if type(x).__name__ == "_ChunkList":
                _n(x.chunks, v)
            elif type(x).__name__ == "_Expression":
                x.expression = (
                    "expression_wrapper("
                    + x.expression
                    + ',"""'
                    + x.expression.replace('"', '\\"')
                    + '""",escape_arrays='
                    + str(escape_arrays)
                    + ")"
                )
            elif type(x).__name__ == "_ControlBlock":
                _n(x.body.chunks, v)

    var: List[Any] = []
    _n(t.file.body.chunks, var)
    t.code = t._generate_python(t.loader)
    try:
        t.compiled = compile(
            escape.to_unicode(t.code), "%s.generated.py" % t.name.replace(".", "_"), "exec", dont_inherit=True
        )
    except Exception:
        # formatted_code = _format_code(t.code).rstrip()
        # app_log.error("%s code:\n%s", t.name, formatted_code)
        raise

    return var


def get_used_tables_in_template(sql):
    """
    >>> get_used_tables_in_template("select * from {{table('test')}}")
    ['test']
    >>> get_used_tables_in_template("select * from {%if x %}{{table('test')}}{%else%}{{table('test2')}}{%end%}")
    ['test', 'test2']
    >>> get_used_tables_in_template("select * from {{table('my.test')}}")
    ['my.test']
    >>> get_used_tables_in_template("select * from {{table('my.test')}}, another_table")
    ['my.test']
    >>> get_used_tables_in_template("select * from another_table")
    []
    >>> get_used_tables_in_template("select * from {{table('my.test')}}, {{table('another.one')}}")
    ['my.test', 'another.one']
    """
    try:
        t = Template(sql)

        def _n(chunks, tables):
            for x in chunks:
                if type(x).__name__ == "_Expression":
                    c = compile(x.expression, "<string>", "exec", dont_inherit=True)
                    v = [x.lower() for x in c.co_names if x not in _namespace and x not in reserved_vars]
                    if "table" in v:

                        def _t(*args, **kwargs):
                            return str(args[0])

                        n = {"table": _t, "TABLE": _t}
                        e = "_tt_tmp = %s" % x.expression
                        exec_in(e, n)
                        tables += [n["_tt_tmp"]]
                elif type(x).__name__ == "_ControlBlock":
                    _n(x.body.chunks, tables)

        tables = []
        _n(t.file.body.chunks, tables)
        return tables
    except SecurityException as e:
        raise SQLTemplateException(e)


@lru_cache(maxsize=2**13)
def get_template_and_variables(sql: str, name: Optional[str], escape_arrays: bool = False):
    """
    Generates a Template and does all the processes necessary. As the object and template variables are cached
    it is important to NOT MODIFY THESE OBJECTS.
    Neither render_sql_template() or generate() modify them, so neither should you
    """
    variable_warnings = []

    try:
        t = Template(sql, name)
        template_variables = get_var_names(t)

        for variable in template_variables:
            if variable["name"] in DEFAULT_PARAM_NAMES:
                name = variable["name"]
                line = variable["line"]
                raise ValueError(f'"{name}" can not be used as a variable name, line {line}')
            if variable["name"] in RESERVED_PARAM_NAMES:
                variable_warnings.append(variable["name"])

        wrap_vars(t, escape_arrays=escape_arrays)

        return t, template_variables, variable_warnings
    except SecurityException as e:
        raise SQLTemplateException(e)


def preprocess_variables(variables: dict, template_variables_with_types: List[dict]):
    """
    >>> preprocess_variables({"test": '24'}, [{"name": "test", "type": "Int32", "default": None}])
    {}
    >>> preprocess_variables({"test": "1,2"}, [{"name": "test", "type": "Array(String)", "default": None}])
    {'test': ['1', '2']}
    >>> preprocess_variables({"test": ['1', '2']}, [{"name": "test", "type": "Array(String)", "default": None}])
    {'test': ['1', '2']}
    >>> preprocess_variables({"test": [1,2]}, [{"name": "test", "type": "Array(String)", "default": None}])
    {'test': ['1', '2']}
    >>> preprocess_variables({"test": "1,2,3"}, [{"name": "test", "type": "Array(Int32)", "default": None}])
    {'test': [1, 2, 3]}
    >>> preprocess_variables({"test": "1,2,msg"}, [{"name": "test", "type": "Array(Int32)", "default": None}])
    {}
    """
    processed_variables = {}
    for variable, value in variables.items():
        try:
            template_vars = [t_var for t_var in template_variables_with_types if t_var["name"] == variable] or None
            if template_vars is None or value is None:
                continue

            t_var = template_vars[0]
            var_type = t_var.get("type")
            if var_type is None:
                continue

            # For now, we only preprocess Array types
            match = _ARRAY_TYPE_RE.match(var_type)
            if match is None:
                continue

            array_type = match.group(1)
            array_fn = type_fns.get("Array")
            parsed_exp = array_fn(value, array_type)
            processed_variables[variable] = ast.literal_eval(parsed_exp)
        except Exception:
            continue

    return processed_variables


def format_SQLTemplateException_message(e: SQLTemplateException, vars_and_types: Optional[dict] = None):
    def join_with_different_last_separator(items, separator=", ", last_separator=" and "):
        if not items:
            return ""
        if len(items) == 1:
            return items[0]

        result = separator.join(items[:-1])
        return result + last_separator + items[-1]

    message = str(e)
    var_names = ""

    try:
        if REQUIRED_PARAM_NOT_DEFINED in message and vars_and_types:
            vars_with_default_none = []
            for item in vars_and_types:
                if (
                    item.get("default") is None
                    and item.get("used_in", None) is None
                    and item.get("name") not in vars_with_default_none
                    and item.get("name") is not JOB_TIMESTAMP_PARAM
                ):
                    vars_with_default_none.append(item["name"])

            var_names = join_with_different_last_separator(vars_with_default_none)
    except Exception:
        pass

    if var_names:
        raise SQLTemplateException(
            f"{REQUIRED_PARAM_NOT_DEFINED}. Check the parameters {join_with_different_last_separator(vars_with_default_none)}. Please provide a value or set a default value in the pipe code.",
            e.documentation,
        )
    else:
        raise e


def render_sql_template(
    sql: str,
    variables: Optional[dict] = None,
    secrets: Optional[List[str]] = None,
    test_mode: bool = False,
    name: Optional[str] = None,
    local_variables: Optional[dict] = None,
    secrets_in_test_mode: Optional[bool] = True,
) -> Tuple[str, TemplateExecutionResults, list]:
    """
    >>> render_sql_template("select * from table where f = {{Float32(foo)}}", { 'foo': -1 })
    ("select * from table where f = toFloat32('-1.0')", {}, [])
    >>> render_sql_template("{% if defined(open) %}ERROR{% else %}YEAH!{% end %}")
    ('YEAH!', {}, [])
    >>> render_sql_template("{% if defined(close) %}ERROR{% else %}YEAH!{% end %}")
    ('YEAH!', {}, [])
    >>> render_sql_template("{% if defined(input) %}ERROR{% else %}YEAH!{% end %}")
    ('YEAH!', {}, [])
    >>> render_sql_template("{% if defined(print) %}ERROR{% else %}YEAH!{% end %}")
    ('YEAH!', {}, [])
    >>> render_sql_template("select * from table where str = {{foo}}", { 'foo': 'test' })
    ("select * from table where str = 'test'", {}, [])
    >>> render_sql_template("select * from table where f = {{foo}}", { 'foo': 1.0 })
    ('select * from table where f = 1.0', {}, [])
    >>> render_sql_template("select {{Boolean(foo)}} from table", { 'foo': True })
    ('select 1 from table', {}, [])
    >>> render_sql_template("select {{Boolean(foo)}} from table", { 'foo': False })
    ('select 0 from table', {}, [])
    >>> render_sql_template("select * from table where f = {{Float32(foo)}}", { 'foo': 1 })
    ("select * from table where f = toFloat32('1.0')", {}, [])
    >>> render_sql_template("select * from table where f = {{foo}}", { 'foo': "';drop table users;" })
    ("select * from table where f = '\\\\';drop table users;'", {}, [])
    >>> render_sql_template("select * from {{symbol(foo)}}", { 'foo': 'table-name' })
    ('select * from `table-name`', {}, [])
    >>> render_sql_template("select * from {{symbol(foo)}}", { 'foo': '"table-name"' })
    ('select * from `table-name`', {}, [])
    >>> render_sql_template("select * from {{table(foo)}}", { 'foo': '"table-name"' })
    ('select * from table-name', {}, [])
    >>> render_sql_template("select * from {{Int32(foo)}}", { 'foo': 'non_int' })
    Traceback (most recent call last):
    ...
    tinybird.sql_template.SQLTemplateException: Template Syntax Error: Error validating 'non_int' to type Int32
    >>> render_sql_template("select * from table where f = {{Float32(foo)}}", test_mode=True)
    ("select * from table where f = toFloat32('0.0')", {}, [])
    >>> render_sql_template("SELECT * FROM query_log__dev where a = {{test}}", test_mode=True)
    ("SELECT * FROM query_log__dev where a = '__no_value__'", {}, [])
    >>> render_sql_template("SELECT {{test}}", {'token':'testing'})
    Traceback (most recent call last):
    ...
    tinybird.sql_template.SQLTemplateException: Template Syntax Error: expression "test" evaluated to null
    >>> render_sql_template("SELECT {{testisasuperlongthingandwedontwanttoreturnthefullthing}}", {'token':'testing'})
    Traceback (most recent call last):
    ...
    tinybird.sql_template.SQLTemplateException: Template Syntax Error: expression "testisasuperlongthin..." evaluated to null
    >>> render_sql_template("SELECT {{ Array(embedding, 'Float32') }}", {'token':'testing', 'embedding': '1,2,3,4, null'})
    Traceback (most recent call last):
    ...
    tinybird.sql_template.SQLTemplateException: Template Syntax Error: Error validating [1,2,3,4, null][4] ( null) to type Float32
    >>> render_sql_template("SELECT {{ Array(embedding, 'Int32', '') }}", {'token':'testing', 'embedding': ''})
    Traceback (most recent call last):
    ...
    tinybird.sql_template.SQLTemplateException: Template Syntax Error: Error validating [][0] (empty string) to type Int32
    >>> render_sql_template('{% if test %}SELECT 1{% else %} select 2 {% end %}')
    (' select 2 ', {}, [])
    >>> render_sql_template('{% if Int32(test, 1) %}SELECT 1{% else %} select 2 {% end %}')
    ('SELECT 1', {}, [])
    >>> render_sql_template('{% for v in test %}SELECT {{v}} {% end %}',test_mode=True)
    ("SELECT '__no_value__' SELECT '__no_value__' SELECT '__no_value__' ", {}, [])
    >>> render_sql_template("select {{Int32(foo, 1)}}", test_mode=True)
    ("select toInt32('1')", {}, [])
    >>> render_sql_template("SELECT count() c FROM test_table where a > {{Float32(myvar)}} {% if defined(my_condition) %} and c = Int32({{my_condition}}){% end %}", {'myvar': 1.0})
    ("SELECT count() c FROM test_table where a > toFloat32('1.0') ", {}, [])
    >>> render_sql_template("SELECT count() c FROM where {{sql_and(a=a, b=b)}}", {'a': '1', 'b': '2'})
    ("SELECT count() c FROM where a = '1' and b = '2'", {}, [])
    >>> render_sql_template("SELECT count() c FROM where {{sql_and(a=a, b=b)}}", {'b': '2'})
    ("SELECT count() c FROM where b = '2'", {}, [])
    >>> render_sql_template("SELECT count() c FROM where {{sql_and(a=Int(a, defined=False), b=Int(b, defined=False))}}", {'b': '2'})
    ('SELECT count() c FROM where b = 2', {}, [])
    >>> render_sql_template("SELECT count() c FROM where {{sql_and(a__in=Array(a), b=b)}}", {'a': 'a,b,c','b': '2'})
    ("SELECT count() c FROM where a in ['a','b','c'] and b = '2'", {}, [])
    >>> render_sql_template("SELECT count() c FROM where {{sql_and(a__not_in=Array(a), b=b)}}", {'a': 'a,b,c','b': '2'})
    ("SELECT count() c FROM where a not in ['a','b','c'] and b = '2'", {}, [])
    >>> render_sql_template("SELECT c FROM where a > {{Date(start)}}", test_mode=True)
    ("SELECT c FROM where a > '2019-01-01'", {}, [])
    >>> render_sql_template("SELECT c FROM where a > {{DateTime(start)}}", test_mode=True)
    ("SELECT c FROM where a > '2019-01-01 00:00:00'", {}, [])
    >>> render_sql_template("SELECT c FROM where a > {{DateTime(start)}}", {'start': '2018-09-07 23:55:00'})
    ("SELECT c FROM where a > '2018-09-07 23:55:00'", {}, [])
    >>> render_sql_template('SELECT * FROM tracker {% if defined(start) %} {{DateTime(start)}} and {{DateTime(end)}} {% end %}', {'start': '2019-08-01 00:00:00', 'end': '2019-08-02 00:00:00'})
    ("SELECT * FROM tracker  '2019-08-01 00:00:00' and '2019-08-02 00:00:00' ", {}, [])
    >>> render_sql_template('SELECT * from test limit {{Int(limit)}}', test_mode=True)
    ('SELECT * from test limit 0', {}, [])
    >>> render_sql_template('SELECT {{symbol(attr)}} from test', test_mode=True)
    ('SELECT `placeholder` from test', {}, [])
    >>> render_sql_template('SELECT {{Array(foo)}}', {'foo': 'a,b,c,d'})
    ("SELECT ['a','b','c','d']", {}, [])
    >>> render_sql_template("SELECT {{Array(foo, 'Int32')}}", {'foo': '1,2,3,4'})
    ('SELECT [1,2,3,4]', {}, [])
    >>> render_sql_template("SELECT {{Array(foo, 'Int32')}}", test_mode=True)
    ('SELECT [0,0]', {}, [])
    >>> render_sql_template("SELECT {{Array(foo)}}", test_mode=True)
    ("SELECT ['']", {}, [])
    >>> render_sql_template("{{max_threads(2)}} SELECT 1")
    ('-- max_threads 2\\n SELECT 1', {'max_threads': 2}, [])
    >>> render_sql_template("SELECT {{String(foo)}}", test_mode=True)
    ("SELECT '__no_value__'", {}, [])
    >>> render_sql_template("SELECT {{String(foo, 'test')}}", test_mode=True)
    ("SELECT 'test'", {}, [])
    >>> render_sql_template("SELECT {{String(foo, 'test')}}", {'foo': 'tt'})
    ("SELECT 'tt'", {}, [])
    >>> render_sql_template("SELECT {{String(format, 'test')}}", {'format': 'tt'})
    Traceback (most recent call last):
    ...
    ValueError: "format" can not be used as a variable name, line 1
    >>> render_sql_template("SELECT {{format}}", {'format': 'tt'})
    Traceback (most recent call last):
    ...
    ValueError: "format" can not be used as a variable name, line 1
    >>> render_sql_template("SELECT {{String(q, 'test')}}", {'q': 'tt'})
    Traceback (most recent call last):
    ...
    ValueError: "q" can not be used as a variable name, line 1
    >>> render_sql_template("SELECT {{column(agg)}}", {})
    Traceback (most recent call last):
    ...
    tinybird.sql_template.SQLTemplateException: Template Syntax Error: Missing column() default value, use `column(column_name, 'default_column_name')`
    >>> render_sql_template("SELECT {{column(agg)}}", {'agg': 'foo'})
    ('SELECT `foo`', {}, [])
    >>> render_sql_template("SELECT {{column(agg)}}", {'agg': '"foo"'})
    ('SELECT `foo`', {}, [])
    >>> render_sql_template("SELECT {{column(agg)}}", {'agg': 'json.a'})
    ('SELECT `json`.`a`', {}, [])
    >>> render_sql_template('{% if not defined(test) %}error("This is an error"){% end %}', {})
    ('error("This is an error")', {}, [])
    >>> render_sql_template('{% if not defined(test) %}custom_error({error: "This is an error"}){% end %}', {})
    ('custom_error({error: "This is an error"})', {}, [])
    >>> render_sql_template("SELECT {{String(foo + 'abcd')}}", test_mode=True)
    ("SELECT '__no_value__'", {}, [])
    >>> render_sql_template("SELECT {{columns(agg)}}", {})
    Traceback (most recent call last):
    ...
    tinybird.sql_template.SQLTemplateException: Template Syntax Error: Missing columns() default value, use `columns(column_names, 'default_column_name')`
    >>> render_sql_template("SELECT {{columns(agg, 'a,b,c')}} FROM table", {})
    ('SELECT `a`,`b`,`c` FROM table', {}, [])
    >>> render_sql_template("SELECT {{columns(agg, 'a,b,c')}} FROM table", {'agg': 'foo'})
    ('SELECT `foo` FROM table', {}, [])
    >>> render_sql_template("SELECT {{columns('a,b,c')}} FROM table", {})
    ('SELECT `a`,`b`,`c` FROM table', {}, [])
    >>> render_sql_template("% {% if whatever(passenger_count) %}{% end %}", test_mode=True)
    Traceback (most recent call last):
    ...
    tinybird.sql_template.SQLTemplateException: Template Syntax Error: 'whatever' is not a valid function, line 1
    >>> render_sql_template("% {% if defined((passenger_count) %}{% end %}", test_mode=True)
    Traceback (most recent call last):
    ...
    SyntaxError: invalid syntax
    >>> render_sql_template("SELECT * FROM dim_fecha_evento where foo like {{sql_unescape(String(pepe), '%')}}", {"pepe": 'raul_el_bueno_is_the_best_%'})
    ("SELECT * FROM dim_fecha_evento where foo like 'raul_el_bueno_is_the_best_%'", {}, [])
    >>> render_sql_template("SELECT * FROM table WHERE field={{String(field_filter)}}", {"field_filter": 'action."test run"'})
    ('SELECT * FROM table WHERE field=\\'action.\\\\"test run\\\\"\\'', {}, [])
    >>> render_sql_template("SELECT {{Int128(foo)}} as x, {{Int128(bar)}} as y", {'foo': -170141183460469231731687303715884105728, 'bar': 170141183460469231731687303715884105727})
    ("SELECT toInt128('-170141183460469231731687303715884105728') as x, toInt128('170141183460469231731687303715884105727') as y", {}, [])
    >>> render_sql_template("SELECT {{Int256(foo)}} as x, {{Int256(bar)}} as y", {'foo': -57896044618658097711785492504343953926634992332820282019728792003956564819968, 'bar': 57896044618658097711785492504343953926634992332820282019728792003956564819967})
    ("SELECT toInt256('-57896044618658097711785492504343953926634992332820282019728792003956564819968') as x, toInt256('57896044618658097711785492504343953926634992332820282019728792003956564819967') as y", {}, [])
    >>> render_sql_template('% SELECT * FROM {% import os %}{{ os.popen("whoami").read() }}')
    Traceback (most recent call last):
    ...
    tinybird.tornado_template.ParseError: import is forbidden at <string>:1
    >>> render_sql_template('% SELECT * FROM {% import os %}{{ os.popen("ls").read() }}')
    Traceback (most recent call last):
    ...
    tinybird.tornado_template.ParseError: import is forbidden at <string>:1
    >>> render_sql_template('% SELECT * FROM {% import os %}{{ os.popen("cat etc/passwd").read() }}')
    Traceback (most recent call last):
    ...
    tinybird.tornado_template.ParseError: import is forbidden at <string>:1
    >>> render_sql_template('% SELECT * FROM {% from os import popen %}{{ popen("cat etc/passwd").read() }}')
    Traceback (most recent call last):
    ...
    tinybird.tornado_template.ParseError: import is forbidden at <string>:1
    >>> render_sql_template('% SELECT {{len([1] * 10**7)}}')
    Traceback (most recent call last):
    ...
    tinybird.sql_template.SQLTemplateException: Template Syntax Error: Invalid BinOp: Pow()
    >>> render_sql_template("% SELECT {{Array(click_selector, 'String', 'pre,pro')}}")
    ("% SELECT ['pre','pro']", {}, [])
    >>> render_sql_template("% SELECT {{Array(click_selector, 'String', 'pre,pro')}}", {'click_selector': 'hi,hello'})
    ("% SELECT ['hi','hello']", {}, [])
    >>> render_sql_template("% SELECT {{Array(click_selector, 'String', '')}}")
    ("% SELECT ['']", {}, [])
    >>> render_sql_template("SELECT {{ Array(embedding, 'Int32', '') }}")
    Traceback (most recent call last):
    ...
    tinybird.sql_template.SQLTemplateException: Template Syntax Error: Error validating [][0] (empty string) to type Int32
    >>> render_sql_template("% SELECT {{Array(click_selector, 'String', '')}}", test_mode=True)
    ("% SELECT ['']", {}, [])
    >>> render_sql_template("% SELECT now() > {{DateTime64(variable, '2020-09-09 10:10:10.000')}}", {})
    ("% SELECT now() > '2020-09-09 10:10:10.000'", {}, [])
    >>> render_sql_template("% SELECT {% if defined(x) %} x, 1", {})
    Traceback (most recent call last):
    ...
    tinybird.tornado_template.UnClosedIfError: Missing {% end %} block for if at line 1
    >>> render_sql_template("% SELECT * FROM employees WHERE 0 {% for kv in JSON(payload) %} OR department = {{kv['dp']}} {% end %}")
    ('% SELECT * FROM employees WHERE 0 ', {}, [])
    >>> render_sql_template("% SELECT * FROM employees WHERE 0 {% for kv in JSON(payload, '[{\\"dp\\":\\"Sales\\"}]') %} OR department = {{kv['dp']}} {% end %}")
    ("% SELECT * FROM employees WHERE 0  OR department = 'Sales' ", {}, [])
    >>> render_sql_template("% SELECT * FROM employees WHERE 0 {% for kv in JSON(payload) %} OR department = {{kv['dp']}} {% end %}", { 'payload': '[{"dp":"Design"},{"dp":"Marketing"}]'})
    ("% SELECT * FROM employees WHERE 0  OR department = 'Design'  OR department = 'Marketing' ", {}, [])
    >>> render_sql_template("% {% for kv in JSON(payload) %} department = {{kv['dp']}} {% end %}", test_mode=True)
    Traceback (most recent call last):
    ...
    tinybird.sql_template.SQLTemplateException: Template Syntax Error: Error parsing JSON: '__no_value__' - Expecting value: line 1 column 1 (char 0)
    >>> render_sql_template("% {% for kv in JSON(payload, '') %} department = {{kv['dp']}} {% end %}")
    Traceback (most recent call last):
    ...
    tinybird.sql_template.SQLTemplateException: Template Syntax Error: Error parsing JSON: '' - Expecting value: line 1 column 1 (char 0)
    >>> render_sql_template("% {% if defined(test) %}{% set _groupByCSV = ','.join(test) %} SELECT test as aa, {{Array(test, 'String')}} as test, {{_groupByCSV}} as a {% end %}", {"test": "1,2"})
    ("%  SELECT test as aa, ['1','2'] as test, '1,2' as a ", {}, [])
    >>> render_sql_template("% {% if defined(test) %}{% set _groupByCSV = ','.join(test) %} SELECT test as aa, {{Array(test, 'String')}} as test, {{_groupByCSV}} as a {% end %}", {"test": ["1","2"]})
    ("%  SELECT test as aa, ['1','2'] as test, '1,2' as a ", {}, [])
    >>> render_sql_template("% {% if defined(test) %}{% set _total = sum(test) %} SELECT test as aa, {{Array(test, 'Int32')}} as test, {{_total}} as a {% end %}", {"test": "1,2"})
    ('%  SELECT test as aa, [1,2] as test, 3 as a ', {}, [])
    >>> render_sql_template("% {% if defined(test) %}{% set _groupByCSV = ','.join(test) %} SELECT test as aa, {{Array(test, 'String')}} as test, {{_groupByCSV}} as a {% end %}", {"test": ["1","2"]})
    ("%  SELECT test as aa, ['1','2'] as test, '1,2' as a ", {}, [])
    >>> render_sql_template("% SELECT {% if defined(x) %} x, 1")
    Traceback (most recent call last):
    ...
    tinybird.tornado_template.UnClosedIfError: Missing {% end %} block for if at line 1
    >>> render_sql_template("select * from table where str = {{pipeline}}", { 'pipeline': 'test' })
    ("select * from table where str = 'test'", {}, ['pipeline'])
    >>> render_sql_template("select * from table where str = {{tb_secret('test')}}", secrets = [ 'tb_secret_test' ])
    ('select * from table where str = {test: String}', {}, [])
    >>> render_sql_template("select * from table where str = {{tb_var('test')}}", secrets = [ 'tb_secret_test' ])
    ('select * from table where str = {test: String}', {}, [])
    >>> render_sql_template("select * from table where str = {{tb_secret('test')}}", variables = { 'test': '1234' })
    Traceback (most recent call last):
    ...
    tinybird.sql_template.SQLTemplateException: Template Syntax Error: Cannot access secret 'test'. Check the secret exists in the Workspace and the token has the required scope.
    >>> render_sql_template("select * from table where str = {{tb_secret('test')}}", test_mode=True)
    ('select * from table where str = {test: String}', {}, [])
    >>> render_sql_template("select * from table where str = {{tb_secret('test')}}", secrets = [ 'tb_secret_test' ], test_mode=True)
    ('select * from table where str = {test: String}', {}, [])
    >>> render_sql_template("select * from table where str = {{tb_secret('test')}}", secrets = [ 'tb_secret_test2' ])
    Traceback (most recent call last):
    ...
    tinybird.sql_template.SQLTemplateException: Template Syntax Error: Cannot access secret 'test'. Check the secret exists in the Workspace and the token has the required scope.
    >>> render_sql_template("select * from table where str = {{tb_secret('test', 'default_value')}}", secrets = [])
    ("select * from table where str = 'default_value'", {}, [])
    >>> render_sql_template("select * from table where str = {{tb_secret('test', 'default_value')}}", secrets = [ 'tb_secret_test' ])
    ('select * from table where str = {test: String}', {}, [])
    >>> render_sql_template("select * from table where str = {{tb_secret('test', '')}}")
    ("select * from table where str = ''", {}, [])
    >>> render_sql_template("select * from table where str = {{tb_secret('test', 'default_value')}}", test_mode=True)
    ("select * from table where str = 'default_value'", {}, [])
    >>> render_sql_template("select * from table where str = {{tb_secret('test', '')}}", test_mode=True)
    ("select * from table where str = ''", {}, [])
    >>> render_sql_template("select * from table where str = {{tb_secret('test', 'default_value')}}", secrets = [ 'tb_secret_test' ], test_mode=True)
    ('select * from table where str = {test: String}', {}, [])
    >>> render_sql_template("select * from table where str = {{String(test)}} and category = {{String(category, 'shirts')}} and color = {{ Int32(color)}}", test_mode=False)
    Traceback (most recent call last):
    ...
    tinybird.sql_template.SQLTemplateException: Template Syntax Error: Required parameter is not defined. Check the parameters test and color. Please provide a value or set a default value in the pipe code.
    >>> render_sql_template("select columns(cols, 'salary') from table where str = {{String(test)}}", test_mode=False)
    Traceback (most recent call last):
    ...
    tinybird.sql_template.SQLTemplateException: Template Syntax Error: Required parameter is not defined. Check the parameters test. Please provide a value or set a default value in the pipe code.
    >>> render_sql_template("SELECT * FROM test WHERE {% for item in JSON(filters, '[{}]') %} {{item.get('operand')}} {% end %}")
    Traceback (most recent call last):
        ...
    tinybird.sql_template.SQLTemplateException: Template Syntax Error: expression "item.get('operand')" evaluated to null
    >>> render_sql_template("SELECT * FROM test WHERE {% for item in JSON(filters, '[{\\\"operand\\\":\\\"test\\\"}]') %} {{item.get('operand')}} {% end %}")
    ("SELECT * FROM test WHERE  'test' ", {}, [])
    >>> render_sql_template("SELECT * FROM test WHERE {% for item in JSON(filters, '[\\\"test\\\"]') %} {{item.get('operator')}} {% end %}")
    Traceback (most recent call last):
        ...
    tinybird.sql_template.SQLTemplateException: Template Syntax Error: 'str' object has no attribute 'get'. Make sure you're using an object/dictionary where trying to use .get()
    """
    escape_split_to_array = ff_split_to_array_escape.get(False)

    t, template_variables, variable_warnings = get_template_and_variables(
        sql, name, escape_arrays=escape_split_to_array
    )

    ## TODO: Could we skip running this unless we need it for some variable preprocessing?
    template_variables_with_types = get_var_names_and_types_cached(t)

    if variables is not None:
        processed_variables = preprocess_variables(variables, template_variables_with_types)
        variables.update(processed_variables)

    # Handle job_timestamp special case providing the default value if not provided
    if any(var["name"] == JOB_TIMESTAMP_PARAM for var in template_variables_with_types):
        variables = variables or {}
        variables.setdefault(JOB_TIMESTAMP_PARAM, datetime.now().strftime("%Y-%m-%d %H:%M:%S"))

    if test_mode:

        def dummy(*args, **kwargs):
            return Comment("error launched")

        v: dict = {x["name"]: Placeholder(x["name"], x["line"]) for x in template_variables}
        is_tb_secret = any([s for s in template_variables if s["name"] == "tb_secret" or s["name"] == "tb_var"])

        if variables:
            v.update(variables)

        if secrets:
            v.update({"tb_secrets": secrets})

        if is_tb_secret and secrets_in_test_mode:
            v.update({TB_SECRET_IN_TEST_MODE: None})

        v.update(type_fns_check)
        v.update(
            {
                # disable error throws on check
                "error": dummy,
                "custom_error": dummy,
            }
        )

        if local_variables:
            v.update(local_variables)

    else:
        v = {x["name"]: None for x in template_variables}
        if variables:
            v.update(variables)

        if secrets:
            v.update({"tb_secrets": secrets})

        v.update(type_fns)

        if local_variables:
            v.update(local_variables)

    try:
        sql, template_execution_results = generate(t, **v)
        try:
            if TB_SECRET_IN_TEST_MODE in template_execution_results:
                del template_execution_results[TB_SECRET_IN_TEST_MODE]
        except Exception:
            pass
        return sql, template_execution_results, variable_warnings
    except NameError as e:
        raise SQLTemplateException(e, documentation="/cli/advanced-templates.html#defined")
    except SQLTemplateException as e:
        format_SQLTemplateException_message(e, vars_and_types=template_variables_with_types)
        raise
    except AttributeError as e:
        # This happens when trying to use `get` on a string or when the object is None
        if "'str' object has no attribute 'get'" in str(e):
            raise SQLTemplateException(
                "'str' object has no attribute 'get'. Make sure you're using an object/dictionary where trying to use .get()",
                documentation="/cli/advanced-templates.html",
            )
        raise SQLTemplateException(str(e), documentation="/cli/advanced-templates.html")
    except IndexError as e:
        # This happens when trying to access string indices on empty strings
        if "string index out of range" in str(e):
            raise SQLTemplateException(
                "String index out of range. Check that string parameters have values before accessing specific characters (e.g., param[0]). Provide default values or add length checks in your template.",
                documentation="/cli/advanced-templates.html",
            )
        raise SQLTemplateException(str(e), documentation="/cli/advanced-templates.html")
    except Exception as e:
        # errors might vary here, we need to support as much as possible
        # https://gitlab.com/tinybird/analytics/-/issues/943
        if "length" in v and not v["length"]:
            raise SQLTemplateException("length cannot be used as a variable name or as a function inside of a template")
        elif "missing 1 required positional argument" in str(e):
            raise SQLTemplateException(
                "one of the transform type functions is missing an argument",
                documentation="/cli/advanced-templates.html#transform-types-functions",
            )
        elif "not callable" in str(e) or "unhashable type" in str(e):
            raise SQLTemplateException(
                "wrong syntax, you might be using a not valid function inside a control block",
                documentation="/cli/advanced-templates.html",
            )
        raise e


def extract_variables_from_sql(sql: str, params: List[Dict[str, Any]]) -> Dict[str, Any]:
    sql = sql[1:] if sql[0] == "%" else sql
    defaults = {}
    mock_data = {}
    try:
        for param in params:
            mock_data[param["name"]] = "__NO__VALUE__DEFINED__"
        # Initialize a dictionary to track variables
        variable_tracker = {}

        # Wrapper function to capture variable assignments
        def capture_variable(name, value):
            variable_tracker[name] = value
            return value

        # Modify the template by adding capture hooks
        tracked_template_string = sql
        for var_name in mock_data.keys():
            tracked_template_string += f"{{% set __ = capture_variable('{var_name}', {var_name}) %}}"

        # Define the modified template with tracking
        template = Template(tracked_template_string)
        type_fns = get_transform_types()
        template.generate(**mock_data, **type_fns, capture_variable=capture_variable)
        for var_name, value in variable_tracker.items():
            if value != "__NO__VALUE__DEFINED__":
                defaults[var_name] = value
    except Exception as e:
        logging.error(f"Error extracting variables from sql: {e}")
        return {}

    return defaults


def render_template_with_secrets(name: str, content: str, secrets: Optional[Dict[str, str]] = None) -> str:
    """Renders a template with secrets, allowing for default values.

    Args:
        name: The name of the template
        content: The template content
        secrets: A dictionary mapping secret names to their values

    Returns:
        The rendered template

    Examples:
        >>> render_template_with_secrets(
        ...     "my_kafka_connection",
        ...     "KAFKA_BOOTSTRAP_SERVERS {{ tb_secret('PRODUCTION_KAFKA_SERVERS', 'localhost:9092') }}",
        ...     secrets = {'PRODUCTION_KAFKA_SERVERS': 'server1:9092,server2:9092'}
        ... )
        'KAFKA_BOOTSTRAP_SERVERS server1:9092,server2:9092'

        >>> render_template_with_secrets(
        ...     "my_kafka_connection",
        ...     "KAFKA_BOOTSTRAP_SERVERS {{ tb_secret('MISSING_SECRET', 'localhost:9092') }}",
        ...     secrets = {}
        ... )
        'KAFKA_BOOTSTRAP_SERVERS localhost:9092'

        >>> render_template_with_secrets(
        ...     "my_kafka_connection",
        ...     "KAFKA_BOOTSTRAP_SERVERS {{ tb_secret('MISSING_SECRET', '') }}",
        ...     secrets = {}
        ... )
        'KAFKA_BOOTSTRAP_SERVERS ""'

        >>> render_template_with_secrets(
        ...     "my_kafka_connection",
        ...     "KAFKA_BOOTSTRAP_SERVERS {{ tb_secret('MISSING_SECRET', 0) }}",
        ...     secrets = {}
        ... )
        'KAFKA_BOOTSTRAP_SERVERS 0'

        >>> render_template_with_secrets(
        ...     "my_kafka_connection",
        ...     "KAFKA_BOOTSTRAP_SERVERS {{ tb_secret('MISSING_SECRET') }}",
        ...     secrets = {}
        ... )
        Traceback (most recent call last):
        ...
        tinybird.sql_template.SQLTemplateException: Template Syntax Error: Cannot access secret 'MISSING_SECRET'. Check the secret exists in the Workspace and the token has the required scope.
    """
    if not secrets:
        secrets = {}

    def tb_secret(secret_name: str, default: Optional[str] = None) -> str:
        """Get a secret value with an optional default.

        Args:
            secret_name: The name of the secret to retrieve
            default: The default value to use if the secret is not found

        Returns:
            The secret value or default

        Raises:
            SQLTemplateException: If the secret is not found and no default is provided
        """
        if secret_name in secrets:
            value = secrets[secret_name]
            if isinstance(value, str) and len(value) == 0:
                return '""'
            return value
        elif default is not None:
            if isinstance(default, str) and len(default) == 0:
                return '""'
            return default
        else:
            raise SQLTemplateException(
                f"Cannot access secret '{secret_name}'. Check the secret exists in the Workspace and the token has the required scope."
            )

    # Create the template
    t = Template(content, name=name, autoescape=None)

    try:
        # Create namespace with our tb_secret function
        namespace = {"tb_secret": tb_secret}

        # Generate the template without all the extra processing
        # This directly uses the underlying _generate method of the Template class
        result = t.generate(**namespace)

        # Convert the result to string
        if isinstance(result, bytes):
            return result.decode("utf-8")

        return str(result)
    except SQLTemplateCustomError as e:
        raise e
    except SQLTemplateException as e:
        raise e
    except Exception as e:
        raise SQLTemplateException(f"Error rendering template with secrets: {str(e)}")
