# Contributing to TOON Python

Thank you for your interest in contributing to TOON Python! This document provides guidelines and information for contributors.

## Table of Contents

- [Development Setup](#development-setup)
- [Code Style](#code-style)
- [Testing](#testing)
- [Pull Request Process](#pull-request-process)
- [Issue Reporting](#issue-reporting)
- [Development Guidelines](#development-guidelines)
- [Release Process](#release-process)

## Development Setup

### Prerequisites

- Python 3.10 or higher
- Git
- [uv](https://docs.astral.sh/uv/) (recommended) or pip

### Setup Steps

1. **Fork and clone the repository**
   ```bash
   git clone https://github.com/your-username/toon-python.git
   cd toon-python
   ```

2. **Create a virtual environment**
   ```bash
   # Using uv (recommended)
   uv venv
   source .venv/bin/activate  # On Windows: .venv\Scripts\activate
   
   # Or using Python's venv
   python -m venv .venv
   source .venv/bin/activate  # On Windows: .venv\Scripts\activate
   ```

3. **Install dependencies**
   ```bash
   # Using uv (recommended)
   uv pip install -e ".[dev]"
   
   # Or using pip
   pip install -e ".[dev]"
   ```

4. **Verify setup**
   ```bash
   pytest --version
   ruff --version
   mypy --version
   ```

## Code Style

This project uses several tools to maintain code quality and consistency:

### Formatting

We use [Black](https://black.readthedocs.io/) for code formatting:

```bash
# Format all code
black src/ tests/

# Check formatting without making changes
black --check src/ tests/
```

### Linting

We use [Ruff](https://docs.astral.sh/ruff/) for linting:

```bash
# Check for linting issues
ruff check src/ tests/

# Auto-fix fixable issues
ruff check --fix src/ tests/
```

### Type Checking

We use [MyPy](https://mypy.readthedocs.io/) for static type checking:

```bash
# Run type checking
mypy src/

# Check specific file
mypy src/toon_python/encoder.py
```

### Pre-commit Hooks (Optional)

To automatically run these tools before commits:

```bash
# Install pre-commit hooks
pre-commit install

# Run hooks manually on all files
pre-commit run --all-files
```

## Testing

### Running Tests

```bash
# Run all tests
pytest

# Run with coverage
pytest --cov=src/toon_python

# Run specific test file
pytest tests/test_encoder.py

# Run specific test
pytest tests/test_encoder.py::test_encode_simple_object

# Run with verbose output
pytest -v
```

### Test Structure

- **Unit tests**: Test individual functions and classes (`test_*.py`)
- **Integration tests**: Test end-to-end functionality (`test_integration.py`)
- **Property-based tests**: Use Hypothesis for comprehensive testing

### Writing Tests

1. **Follow naming conventions**:
   - Test files: `test_*.py`
   - Test functions: `test_*`
   - Test classes: `Test*`

2. **Use descriptive test names**:
   ```python
   def test_encode_object_with_nested_structure():
       # Good
       pass
   
   def test_encode():
       # Too generic
       pass
   ```

3. **Use fixtures for common setup**:
   ```python
   def test_encode_with_custom_options(sample_data):
       options = EncodeOptions(indent=4)
       result = encode(sample_data, options)
       assert result is not None
   ```

4. **Include property-based tests for complex logic**:
   ```python
   @given(st.lists(st.integers()))
   def test_encode_roundtrip_integers(int_list):
       encoded = encode({"numbers": int_list})
       assert "numbers" in encoded
       assert str(len(int_list)) in encoded
   ```

### Coverage

We aim for high test coverage. Check coverage with:

```bash
pytest --cov=src/toon_python --cov-report=html
```

View the HTML report in `htmlcov/index.html`.

## Pull Request Process

### Before Submitting

1. **Create a feature branch**:
   ```bash
   git checkout -b feature/your-feature-name
   ```

2. **Make your changes** following the code style guidelines

3. **Run all quality checks**:
   ```bash
   # Format code
   black src/ tests/
   
   # Lint code
   ruff check src/ tests/
   
   # Type check
   mypy src/
   
   # Run tests
   pytest
   ```

4. **Update documentation** if needed:
   - README.md for user-facing changes
   - Docstrings for API changes
   - Add examples for new features

### Submitting a Pull Request

1. **Push your branch**:
   ```bash
   git push origin feature/your-feature-name
   ```

2. **Create a pull request** with:
   - Clear title describing the change
   - Detailed description of what was changed and why
   - Reference any related issues
   - Screenshots or examples if applicable

3. **Fill out the PR template** (if available)

4. **Wait for review** and address feedback

### PR Review Process

- All PRs require at least one review
- Maintainers may request changes or additional tests
- CI must pass before merge
- Follow semantic versioning for changes

## Issue Reporting

### Bug Reports

When reporting bugs, include:

1. **Environment information**:
   - Python version
   - OS version
   - TOON Python version

2. **Minimal reproduction example**:
   ```python
   from toon_python import encode
   
   data = {"example": "data"}
   result = encode(data)
   # Expected: X, Got: Y
   ```

3. **Error traceback** (if applicable)

4. **Expected vs actual behavior**

### Feature Requests

When requesting features:

1. **Describe the use case** clearly
2. **Explain why the feature is needed**
3. **Provide examples** of how it would work
4. **Consider alternative approaches**

## Development Guidelines

### Code Organization

- **Core logic**: `src/toon_python/`
- **Tests**: `tests/`
- **Documentation**: `README.md`, inline docstrings

### API Design

- Follow Python conventions (PEP 8, PEP 484)
- Use type hints everywhere
- Provide clear docstrings with examples
- Consider backward compatibility

### Performance

- Profile before optimizing
- Consider token efficiency (core project goal)
- Avoid unnecessary complexity
- Use appropriate data structures

### Documentation

- **Docstrings**: Use Google style or NumPy style
- **Examples**: Include in docstrings and README
- **Comments**: Explain "why", not "what"

### Commit Messages

Follow [Conventional Commits](https://www.conventionalcommits.org/):

```
feat: add new encoding option for custom delimiters
fix: resolve circular reference detection issue
docs: update README with installation instructions
test: add property-based tests for array encoding
refactor: simplify encoder logic for better maintainability
```

## Release Process

Releases are managed by maintainers:

1. **Version bump** in `pyproject.toml`
2. **Update CHANGELOG.md** with changes
3. **Create git tag**:
   ```bash
   git tag v0.1.2
   git push origin v0.1.2
   ```
4. **Build and publish** to PyPI (automated via CI)

## Getting Help

- **Issues**: Use GitHub Issues for bugs and feature requests
- **Discussions**: Use GitHub Discussions for questions and ideas
- **Documentation**: Check README.md and docstrings
- **Examples**: Look at test files for usage patterns

## Code of Conduct

This project follows the [Python Community Code of Conduct](https://www.python.org/psf/conduct/). Please be respectful and inclusive in all interactions.

---

Thank you for contributing to TOON Python! Your contributions help make this project better for everyone.