# API Contract: TOON Python Library

## Public API

### Primary Function

```python
def encode(data: Any, options: Optional[EncodeOptions] = None) -> str:
    """
    Convert any JSON-serializable value to TOON format.
    
    Args:
        data: Any Python data structure to encode
        options: Optional encoding configuration
        
    Returns:
        TOON format string
        
    Raises:
        ToonEncodingError: If data cannot be encoded
        CircularReferenceError: If circular references detected
        DatasetTooLargeError: If data exceeds 10MB limit
    """
```

### Configuration Class

```python
@dataclass
class EncodeOptions:
    """
    Configuration options for TOON encoding.
    
    Attributes:
        indent: Number of spaces per nesting level (default: 2)
        delimiter: Character for separating array values (default: comma)
        length_marker: Optional prefix for array counts (default: None)
    """
    indent: int = 2
    delimiter: Delimiter = Delimiter.COMMA
    length_marker: Optional[str] = None
```

### Delimiter Enum

```python
class Delimiter(Enum):
    """
    Supported delimiters for array value separation.
    
    Values:
        COMMA: Use comma as delimiter (default)
        TAB: Use tab as delimiter
        PIPE: Use pipe as delimiter
    """
    COMMA = ','
    TAB = '\t'
    PIPE = '|'
```

## Error Types

```python
class ToonEncodingError(Exception):
    """Base exception for TOON encoding errors."""
    pass

class CircularReferenceError(ToonEncodingError):
    """Raised when circular references are detected in input data."""
    pass

class DatasetTooLargeError(ToonEncodingError):
    """Raised when input data exceeds 10MB size limit."""
    pass

class NonSerializableError(ToonEncodingError):
    """Raised when input contains non-serializable objects."""
    pass
```

## Usage Examples

### Basic Encoding

```python
import toon_python

# Simple dictionary
data = {"name": "Alice", "age": 30, "active": True}
result = toon_python.encode(data)
# Output: name: Alice
#         age: 30
#         active: true
```

### With Custom Options

```python
from toon_python import encode, EncodeOptions, Delimiter

options = EncodeOptions(
    indent=4,
    delimiter=Delimiter.PIPE,
    length_marker="#"
)
result = encode(data, options)
```

### Array Encoding

```python
# Primitive array
tags = ["admin", "ops", "dev"]
result = encode({"tags": tags})
# Output: tags[3]: admin,ops,dev

# Tabular array
items = [
    {"sku": "A1", "qty": 2, "price": 9.99},
    {"sku": "B2", "qty": 1, "price": 14.5}
]
result = encode({"items": items})
# Output: items[2]{sku,qty,price}:
#         A1,2,9.99
#         B2,1,14.5
```

## Performance Characteristics

### Token Reduction
- Target: 30-60% fewer tokens than equivalent JSON
- Measured against common LLM tokenizers
- Optimized for structured data patterns

### Memory Usage
- Streaming output prevents memory buildup
- Linear scaling with data size
- <10MB dataset constraint

### Error Handling
- Strict mode: fail fast with clear exceptions
- Detailed error messages for debugging
- Circular reference detection with path information

## Type Support

### Supported Types
- **Primitives**: str, int, float, bool, None
- **Collections**: dict, list, tuple, set, frozenset
- **Special**: datetime, date, Decimal, UUID, bytes

### Type Conversions
| Python Type | TOON Representation | Notes |
|-------------|---------------------|--------|
| str | string | Preserved as-is |
| int | number | Preserved, including large integers |
| float | number | -0 → 0, NaN/±inf → null |
| bool | boolean | true/false literals |
| None | null | null literal |
| datetime | string | ISO format |
| date | string | ISO format |
| Decimal | string | String representation |
| UUID | string | String representation |
| bytes | string | Base64 encoded |
| set/frozenset | array | Converted to list |
| Other | null | Non-serializable objects |

## Integration Points

### Import Statement
```python
import toon_python
# or
from toon_python import encode, EncodeOptions, Delimiter
```

### Package Structure
- Public API available in `toon_python.__init__.py`
- Internal modules not exposed (encoder, normalizer, etc.)
- Type definitions available for IDE support

### Version Compatibility
- Python 3.10+ required
- No runtime dependencies
- Type hints available for static analysis