# TOON Python Library Quick Start

## Installation

```bash
pip install toon-python
```

## Basic Usage

### Simple Encoding

```python
import toon_python

# Basic dictionary
data = {
    "name": "Alice",
    "age": 30,
    "active": True,
    "tags": ["admin", "user"]
}

result = toon_python.encode(data)
print(result)
```

**Output**:
```
name: Alice
age: 30
active: true
tags[2]: admin,user
```

### Custom Formatting

```python
from toon_python import encode, EncodeOptions, Delimiter

# Custom formatting options
options = EncodeOptions(
    indent=4,                    # 4 spaces instead of 2
    delimiter=Delimiter.PIPE,       # Use | instead of ,
    length_marker="#"               # Add # prefix to array counts
)

result = encode(data, options)
print(result)
```

**Output**:
```
name: Alice
age: 30
active: true
tags[#2]: admin|user
```

## Array Encoding Examples

### Primitive Arrays (Inline)

```python
data = {"colors": ["red", "green", "blue"]}
result = encode(data)
```

**Output**:
```
colors[3]: red,green,blue
```

### Tabular Arrays (CSV-like)

```python
products = [
    {"name": "Widget", "price": 9.99, "stock": 100},
    {"name": "Gadget", "price": 14.50, "stock": 50}
]
data = {"products": products}
result = encode(data)
```

**Output**:
```
products[2]{name,price,stock}:
    Widget,9.99,100
    Gadget,14.5,50
```

### Mixed Arrays (List Format)

```python
data = {"mixed": [1, {"nested": True}, "text"]}
result = encode(data)
```

**Output**:
```
mixed[3]:
  - 1
  - nested: true
  - text
```

## Type Support

### Python-Specific Types

```python
from datetime import datetime
from decimal import Decimal
import uuid

data = {
    "timestamp": datetime.now(),
    "price": Decimal("19.99"),
    "id": uuid.uuid4(),
    "binary": b"hello world"
}

result = encode(data)
```

**Output**:
```
timestamp: 2025-01-27T10:30:00
price: 19.99
id: 550e8400-e29b-41d4-a716-446655440000
binary: aGVsbG8gd29ybGQ=
```

## Error Handling

### Strict Mode (Default)

```python
try:
    # Circular reference
    data = {}
    data["self"] = data
    result = encode(data)
except toon_python.CircularReferenceError as e:
    print(f"Circular reference detected: {e}")
```

### Non-serializable Objects

```python
try:
    data = {"func": lambda x: x}
    result = encode(data)
except toon_python.NonSerializableError as e:
    print(f"Non-serializable object: {e}")
```

## Performance Tips

### Maximum Token Reduction

```python
# Use tabular format for uniform object arrays
users = [
    {"name": "Alice", "age": 30, "city": "NY"},
    {"name": "Bob", "age": 25, "city": "SF"},
    {"name": "Carol", "age": 35, "city": "LA"}
]
# Automatically optimized to tabular format

# Choose appropriate delimiter
options = EncodeOptions(delimiter=Delimiter.TAB)  # For tab-separated values
```

### Memory Efficiency

```python
# Streaming is automatic - no memory buildup
# Works well with nested structures up to 10MB
large_data = generate_nested_structure()  # Up to 10MB
result = encode(large_data)  # Linear memory usage
```

## Common Patterns

### Configuration Files

```python
config = {
    "database": {
        "host": "localhost",
        "port": 5432,
        "ssl": True
    },
    "features": ["auth", "logging", "cache"],
    "limits": {
        "max_connections": 100,
        "timeout": 30
    }
}

# Save as TOON config file
toon_config = encode(config, EncodeOptions(indent=2))
with open("config.toon", "w") as f:
    f.write(toon_config)
```

### LLM Prompt Optimization

```python
# Optimize data for LLM context
prompt_data = {
    "context": "Analyze user behavior patterns",
    "users": [
        {"id": 1, "actions": ["login", "view", "logout"]},
        {"id": 2, "actions": ["login", "search", "view"]}
    ],
    "requirements": ["identify_patterns", "generate_insights"]
}

# Maximum token reduction for LLM
optimized_prompt = encode(prompt_data, EncodeOptions(
    delimiter=Delimiter.COMMA,
    length_marker="#"
))

# Use in LLM API call
llm_response = call_llm_api(f"Data: {optimized_prompt}\n\nAnalysis:")
```

## Comparison with JSON

### Token Savings Example

```python
import json
import toon_python

data = {
    "users": [
        {"id": 1, "name": "Alice", "active": True},
        {"id": 2, "name": "Bob", "active": False}
    ],
    "settings": {"theme": "dark", "notifications": True}
}

json_output = json.dumps(data, indent=2)
toon_output = encode(data)

print(f"JSON length: {len(json_output)} characters")
print(f"TOON length: {len(toon_output)} characters")
print(f"Reduction: {(1 - len(toon_output)/len(json_output))*100:.1f}%")
```

**Typical Results**:
- JSON: ~150 characters
- TOON: ~85 characters  
- **43% token reduction**

## Next Steps

1. **Explore Formatting Options**: Try different delimiters and indentation
2. **Optimize Arrays**: Structure data for tabular optimization
3. **Error Handling**: Implement proper exception handling
4. **Performance Testing**: Measure token reduction in your use case
5. **Integration**: Add to your LLM pipeline or configuration system

## Support

- **Documentation**: [Full API docs](https://toon-python.readthedocs.io)
- **Issues**: [GitHub Issues](https://github.com/your-repo/toon-python/issues)
- **Examples**: [Example Repository](https://github.com/your-repo/toon-python-examples)