---

description: "Task list for TOON Python Library - Core Encoding Implementation"
---

# Tasks: TOON Python Library - Core Encoding Implementation

**Input**: Design documents from `/specs/001-toon-encoding/`
**Prerequisites**: plan.md (required), spec.md (required for user stories), research.md, data-model.md, contracts/

**Tests**: Tests are included as this is a library requiring comprehensive test coverage per constitution requirements.

**Organization**: Tasks are grouped by user story to enable independent implementation and testing of each story.

## Format: `[ID] [P?] [Story] Description`

- **[P]**: Can run in parallel (different files, no dependencies)
- **[Story]**: Which user story this task belongs to (e.g., US1, US2, US3)
- Include exact file paths in descriptions

## Path Conventions

- **Single project**: `src/`, `tests/` at repository root
- Paths shown below follow the plan.md structure for Python library

## Phase 1: Setup (Shared Infrastructure)

**Purpose**: Project initialization and basic structure

- [ ] T001 Create project structure per implementation plan
- [ ] T002 Initialize Python project with hatchling build system and dependencies
- [ ] T003 [P] Configure development tools (pytest, mypy, ruff, black) in pyproject.toml
- [ ] T004 [P] Create src/toon_python package structure with __init__.py files
- [ ] T005 [P] Create tests directory structure with conftest.py

---

## Phase 2: Foundational (Blocking Prerequisites)

**Purpose**: Core infrastructure that MUST be complete before ANY user story can be implemented

**⚠️ CRITICAL**: No user story work can begin until this phase is complete

- [ ] T006 Define core type system in src/toon_python/types.py
- [ ] T007 [P] Define configuration classes and enums in src/toon_python/constants.py
- [ ] T008 [P] Define custom exception classes in src/toon_python/__init__.py
- [ ] T009 [P] Create base test fixtures and utilities in tests/conftest.py
- [ ] T010 Setup pytest configuration for hypothesis property-based testing

**Checkpoint**: Foundation ready - user story implementation can now begin in parallel

---

## Phase 3: User Story 1 - Basic TOON Encoding (Priority: P1) 🎯 MVP

**Goal**: Convert Python data structures to TOON format with 30-60% token reduction

**Independent Test**: Encode various Python data structures and verify token reduction compared to JSON, with output matching TOON specification format

### Tests for User Story 1

- [ ] T011 [P] [US1] Create test suite for primitive encoding in tests/test_primitives.py
- [ ] T012 [P] [US1] Create test suite for object encoding in tests/test_encoder.py
- [ ] T013 [P] [US1] Create test suite for basic array encoding in tests/test_encoder.py
- [ ] T014 [P] [US1] Create integration test for end-to-end encoding workflow in tests/test_integration.py
- [ ] T015 [P] [US1] Create test suite for numeric edge cases (-0, NaN, ±inf) in tests/test_normalizer.py

### Implementation for User Story 1

- [ ] T016 [P] [US1] Implement ValueNormalizer class in src/toon_python/normalizer.py
- [ ] T017 [P] [US1] Implement PrimitiveEncoder class in src/toon_python/primitives.py
- [ ] T018 [P] [US1] Implement LineWriter class in src/toon_python/writer.py
- [ ] T019 [P] [US1] Implement OutputFormatter class in src/toon_python/formatter.py
- [ ] T020 [US1] Implement ToonEncoder orchestration class in src/toon_python/encoder.py (depends on T016-T019)
- [ ] T021 [US1] Expose public API in src/toon_python/__init__.py (depends on T020)
- [ ] T022 [US1] Add circular reference detection and error handling
- [ ] T023 [US1] Add dataset size validation (<10MB limit)
- [ ] T024 [P] [US1] Implement -0 to 0 conversion in ValueNormalizer in src/toon_python/normalizer.py
- [ ] T025 [P] [US1] Implement NaN/±inf to null conversion in ValueNormalizer in src/toon_python/normalizer.py

**Checkpoint**: At this point, User Story 1 should be fully functional and testable independently

---

## Phase 4: User Story 2 - Advanced Type Support (Priority: P2)

**Goal**: Encode Python-specific types (datetime, Decimal, UUID, bytes) to JSON-compatible representations

**Independent Test**: Encode objects with Python-specific types and verify they convert to appropriate JSON-compatible representations in TOON format

### Tests for User Story 2

- [ ] T026 [P] [US2] Create test suite for Python-specific type normalization in tests/test_normalizer.py
- [ ] T027 [P] [US2] Create integration tests for advanced type encoding in tests/test_integration.py

### Implementation for User Story 2

- [ ] T028 [P] [US2] Extend ValueNormalizer with datetime/date handling in src/toon_python/normalizer.py
- [ ] T029 [P] [US2] Extend ValueNormalizer with Decimal handling in src/toon_python/normalizer.py
- [ ] T030 [P] [US2] Extend ValueNormalizer with UUID handling in src/toon_python/normalizer.py
- [ ] T031 [P] [US2] Extend ValueNormalizer with bytes handling (Base64) in src/toon_python/normalizer.py
- [ ] T032 [P] [US2] Extend ValueNormalizer with set/frozenset handling in src/toon_python/normalizer.py
- [ ] T033 [US2] Add comprehensive type conversion error handling

**Checkpoint**: At this point, User Stories 1 AND 2 should both work independently

---

## Phase 5: User Story 3 - Flexible Formatting Options (Priority: P2)

**Goal**: Provide control over output formatting (indentation, delimiters, length markers)

**Independent Test**: Encode the same data with different formatting options and verify output matches expected patterns

### Tests for User Story 3

- [ ] T034 [P] [US3] Create test suite for formatting options in tests/test_formatter.py
- [ ] T035 [P] [US3] Create integration tests for different formatting configurations in tests/test_integration.py

### Implementation for User Story 3

- [ ] T036 [P] [US3] Extend OutputFormatter with configurable indentation in src/toon_python/formatter.py
- [ ] T037 [P] [US3] Extend OutputFormatter with delimiter options in src/toon_python/formatter.py
- [ ] T038 [P] [US3] Extend OutputFormatter with length marker support in src/toon_python/formatter.py
- [ ] T039 [US3] Update EncodeOptions class to support all formatting options in src/toon_python/constants.py
- [ ] T040 [US3] Integrate formatting options with ToonEncoder in src/toon_python/encoder.py

**Checkpoint**: All user stories should now be independently functional

---

## Phase 6: User Story 4 - Array Optimization Strategies (Priority: P3)

**Goal**: Implement optimal array encoding (inline, tabular, list) for maximum token reduction

**Independent Test**: Encode different array types and verify the correct optimization strategy is applied

### Tests for User Story 4

- [ ] T041 [P] [US4] Create test suite for array analysis in tests/test_encoder.py
- [ ] T042 [P] [US4] Create test suite for tabular array encoding in tests/test_encoder.py
- [ ] T043 [P] [US4] Create integration tests for array optimization strategies in tests/test_integration.py

### Implementation for User Story 4

- [ ] T044 [P] [US4] Implement ArrayAnalyzer class in src/toon_python/encoder.py
- [ ] T045 [P] [US4] Implement uniform object detection for tabular optimization in src/toon_python/encoder.py
- [ ] T046 [P] [US4] Implement inline array encoding for primitives in src/toon_python/encoder.py
- [ ] T047 [P] [US4] Implement list format encoding for mixed/nested arrays in src/toon_python/encoder.py
- [ ] T048 [US4] Integrate array optimization with ToonEncoder orchestration in src/toon_python/encoder.py

**Checkpoint**: All user stories should now be independently functional

---

## Phase 7: Polish & Cross-Cutting Concerns

**Purpose**: Improvements that affect multiple user stories

- [ ] T049 [P] Add comprehensive property-based tests with hypothesis in tests/test_primitives.py
- [ ] T050 [P] Add performance benchmarks for token reduction verification in tests/test_integration.py
- [ ] T051 [P] Add edge case tests (numeric edge cases, empty values) in tests/test_encoder.py
- [ ] T052 [P] Documentation updates in README.md and docstrings
- [ ] T053 Code cleanup and mypy compliance verification
- [ ] T054 Run quickstart.md validation examples
- [ ] T055 Final integration testing against TypeScript reference implementation

---

## Dependencies & Execution Order

### Phase Dependencies

- **Setup (Phase 1)**: No dependencies - can start immediately
- **Foundational (Phase 2)**: Depends on Setup completion - BLOCKS all user stories
- **User Stories (Phase 3-6)**: All depend on Foundational phase completion
  - User stories can then proceed in parallel (if staffed)
  - Or sequentially in priority order (P1 → P2 → P3)
- **Polish (Final Phase)**: Depends on all desired user stories being complete

### User Story Dependencies

- **User Story 1 (P1)**: Can start after Foundational (Phase 2) - No dependencies on other stories
- **User Story 2 (P2)**: Can start after Foundational (Phase 2) - Extends US1 type system
- **User Story 3 (P2)**: Can start after Foundational (Phase 2) - Extends US1 formatting
- **User Story 4 (P3)**: Can start after Foundational (Phase 2) - Extends US1 array handling

### Within Each User Story

- Tests MUST be written and FAIL before implementation
- Core components before orchestration
- Individual components before integration
- Story complete before moving to next priority

### Parallel Opportunities

- All Setup tasks marked [P] can run in parallel
- All Foundational tasks marked [P] can run in parallel (within Phase 2)
- Once Foundational phase completes, all user stories can start in parallel (if team capacity allows)
- All tests for a user story marked [P] can run in parallel
- Components within a story marked [P] can run in parallel
- Different user stories can be worked on in parallel by different team members

---

## Parallel Example: User Story 1

```bash
# Launch all tests for User Story 1 together:
Task: "Create test suite for primitive encoding in tests/test_primitives.py"
Task: "Create test suite for object encoding in tests/test_encoder.py"
Task: "Create test suite for basic array encoding in tests/test_encoder.py"
Task: "Create integration test for end-to-end encoding workflow in tests/test_integration.py"

# Launch all core components for User Story 1 together:
Task: "Implement ValueNormalizer class in src/toon_python/normalizer.py"
Task: "Implement PrimitiveEncoder class in src/toon_python/primitives.py"
Task: "Implement LineWriter class in src/toon_python/writer.py"
Task: "Implement OutputFormatter class in src/toon_python/formatter.py"
```

---

## Implementation Strategy

### MVP First (User Story 1 Only)

1. Complete Phase 1: Setup
2. Complete Phase 2: Foundational (CRITICAL - blocks all stories)
3. Complete Phase 3: User Story 1
4. **STOP and VALIDATE**: Test User Story 1 independently
5. Deploy/demo if ready

### Incremental Delivery

1. Complete Setup + Foundational → Foundation ready
2. Add User Story 1 → Test independently → Deploy/Demo (MVP!)
3. Add User Story 2 → Test independently → Deploy/Demo
4. Add User Story 3 → Test independently → Deploy/Demo
5. Add User Story 4 → Test independently → Deploy/Demo
6. Each story adds value without breaking previous stories

### Parallel Team Strategy

With multiple developers:

1. Team completes Setup + Foundational together
2. Once Foundational is done:
   - Developer A: User Story 1 (core encoding)
   - Developer B: User Story 2 (advanced types)
   - Developer C: User Story 3 (formatting options)
   - Developer D: User Story 4 (array optimization)
3. Stories complete and integrate independently

---

## Notes

- [P] tasks = different files, no dependencies
- [Story] label maps task to specific user story for traceability
- Each user story should be independently completable and testable
- Verify tests fail before implementing
- Commit after each task or logical group
- Stop at any checkpoint to validate story independently
- Avoid: vague tasks, same file conflicts, cross-story dependencies that break independence