# TOON Python Test Specification

Based on the existing TypeScript tests and Python specification, this document outlines the comprehensive test suite needed for the Python implementation.

## Test Structure

```python
# test/test_toon.py
import pytest
import toon
from toon import Delimiter, EncodeOptions
from datetime import datetime, date
from decimal import Decimal
from uuid import UUID
import base64
```

## 1. Primitive Encoding Tests

### 1.1 String Encoding
```python
def test_encodes_safe_strings_without_quotes():
    assert toon.encode('hello') == 'hello'
    assert toon.encode('Ada_99') == 'Ada_99'
    assert toon.encode('café') == 'café'
    assert toon.encode('你好') == '你好'
    assert toon.encode('🚀') == '🚀'
    assert toon.encode('hello 👋 world') == 'hello 👋 world'

def test_quotes_empty_string():
    assert toon.encode('') == '""'

def test_quotes_ambiguous_strings():
    assert toon.encode('true') == '"true"'
    assert toon.encode('false') == '"false"'
    assert toon.encode('null') == '"null"'
    assert toon.encode('42') == '"42"'
    assert toon.encode('-3.14') == '"-3.14"'
    assert toon.encode('1e-6') == '"1e-6"'
    assert toon.encode('05') == '"05"'

def test_escapes_control_characters():
    assert toon.encode('line1\nline2') == '"line1\\nline2"'
    assert toon.encode('tab\there') == '"tab\\there"'
    assert toon.encode('return\rcarriage') == '"return\\rcarriage"'
    assert toon.encode('C:\\Users\\path') == '"C:\\\\Users\\\\path"'

def test_quotes_structural_strings():
    assert toon.encode('[3]: x,y') == '"[3]: x,y"'
    assert toon.encode('- item') == '"- item"'
    assert toon.encode('[test]') == '"[test]"'
    assert toon.encode('{key}') == '"{key}"'

def test_quotes_whitespace_padded_strings():
    assert toon.encode(' padded ') == '" padded "'
    assert toon.encode('  ') == '"  "'
```

### 1.2 Number Encoding
```python
def test_encodes_numbers():
    assert toon.encode(42) == '42'
    assert toon.encode(3.14) == '3.14'
    assert toon.encode(-7) == '-7'
    assert toon.encode(0) == '0'

def test_handles_special_numeric_values():
    assert toon.encode(-0.0) == '0'
    assert toon.encode(1e6) == '1000000'
    assert toon.encode(1e-6) == '0.000001'
    assert toon.encode(1e20) == '100000000000000000000'

def test_converts_non_finite_numbers_to_null():
    assert toon.encode(float('inf')) == 'null'
    assert toon.encode(float('-inf')) == 'null'
    assert toon.encode(float('nan')) == 'null'

def test_handles_large_integers():
    # Python supports arbitrary precision integers
    large_int = 10**100
    assert toon.encode(large_int) == str(large_int)
```

### 1.3 Boolean and None Encoding
```python
def test_encodes_booleans():
    assert toon.encode(True) == 'true'
    assert toon.encode(False) == 'false'

def test_encodes_none():
    assert toon.encode(None) == 'null'
```

## 2. Python Type Normalization Tests

### 2.1 DateTime and Date Types
```python
def test_normalizes_datetime_to_iso_string():
    dt = datetime(2025, 1, 1, 0, 0, 0, 0)
    assert toon.encode(dt) == '"2025-01-01T00:00:00"'
    
    dt_with_microseconds = datetime(2025, 1, 1, 12, 30, 45, 123456)
    assert toon.encode(dt_with_microseconds) == '"2025-01-01T12:30:45.123456"'

def test_normalizes_date_to_iso_string():
    d = date(2025, 1, 1)
    assert toon.encode(d) == '"2025-01-01"'

def test_normalizes_datetime_in_objects():
    data = {
        "created": datetime(2025, 1, 1, 12, 0, 0),
        "updated": date(2025, 1, 15)
    }
    expected = 'created: "2025-01-01T12:00:00"\nupdated: "2025-01-15"'
    assert toon.encode(data) == expected
```

### 2.2 Decimal Type
```python
def test_normalizes_decimal_to_string():
    decimal_val = Decimal('123.456')
    assert toon.encode(decimal_val) == '"123.456"'
    
    precise_decimal = Decimal('1.23456789012345678901234567890')
    assert toon.encode(precise_decimal) == f'"{precise_decimal}"'

def test_normalizes_decimal_in_objects():
    data = {"price": Decimal('99.99'), "quantity": Decimal('2')}
    expected = 'price: "99.99"\nquantity: "2"'
    assert toon.encode(data) == expected
```

### 2.3 UUID Type
```python
def test_normalizes_uuid_to_string():
    uuid_val = UUID('12345678-1234-5678-9abc-123456789abc')
    assert toon.encode(uuid_val) == '"12345678-1234-5678-9abc-123456789abc"'

def test_normalizes_uuid_in_objects():
    data = {"id": UUID('550e8400-e29b-41d4-a716-446655440000')}
    expected = 'id: "550e8400-e29b-41d4-a716-446655440000"'
    assert toon.encode(data) == expected
```

### 2.4 Bytes Type
```python
def test_normalizes_bytes_to_base64():
    data = b'hello world'
    expected = f'"{base64.b64encode(data).decode()}"'
    assert toon.encode(data) == expected

def test_normalizes_bytes_in_objects():
    data = {"binary": b'hello'}
    expected = f'binary: "{base64.b64encode(b"hello").decode()}"'
    assert toon.encode(data) == expected
```

### 2.5 Collection Types
```python
def test_normalizes_set_to_list():
    data = {"tags": {'a', 'b', 'c'}}
    # Sets are unordered, so we need to check that it's a valid list format
    result = toon.encode(data)
    assert result.startswith('tags[3]:')
    assert 'a' in result and 'b' in result and 'c' in result

def test_normalizes_frozenset_to_list():
    data = {"immutable": frozenset([1, 2, 3])}
    result = toon.encode(data)
    assert result.startswith('immutable[3]:')
    assert '1' in result and '2' in result and '3' in result

def test_normalizes_tuple_to_list():
    data = {"coords": (10, 20, 30)}
    assert toon.encode(data) == 'coords[3]: 10,20,30'

def test_normalizes_nested_collections():
    data = {"nested": [{'a': (1, 2)}, {'b': {3, 4}}]}
    result = toon.encode(data)
    # Should normalize tuple to list and set to list
    assert 'nested[2]:' in result
    assert '[2]: 1,2' in result  # tuple -> list
    assert '[2]: 3,4' in result  # set -> list
```

### 2.6 Invalid Types
```python
def test_converts_functions_to_null():
    def dummy_func():
        pass
    
    class DummyClass:
        def method(self):
            pass
    
    assert toon.encode(dummy_func) == 'null'
    assert toon.encode(DummyClass.method) == 'null'
    assert toon.encode(lambda x: x) == 'null'

def test_converts_custom_objects_to_null():
    class CustomClass:
        def __init__(self):
            self.value = 42
    
    assert toon.encode(CustomClass()) == 'null'
    assert toon.encode({"obj": CustomClass()}) == 'obj: null'

def test_converts_generator_to_null():
    def gen():
        yield 1
        yield 2
    
    assert toon.encode(gen()) == 'null'
```

## 3. Object Encoding Tests

### 3.1 Basic Object Encoding
```python
def test_preserves_key_order_in_objects():
    obj = {
        "id": 123,
        "name": "Ada",
        "active": True,
    }
    assert toon.encode(obj) == 'id: 123\nname: Ada\nactive: true'

def test_encodes_null_values_in_objects():
    obj = {"id": 123, "value": None}
    assert toon.encode(obj) == 'id: 123\nvalue: null'

def test_encodes_empty_objects_as_empty_string():
    assert toon.encode({}) == ''

def test_encodes_empty_nested_object():
    assert toon.encode({"user": {}}) == 'user:'
```

### 3.2 Key Quoting Rules
```python
def test_quotes_keys_with_special_characters():
    assert toon.encode({"order:id": 7}) == '"order:id": 7'
    assert toon.encode({"[index]": 5}) == '"[index]": 5'
    assert toon.encode({"{key}": 5}) == '"{key}": 5'
    assert toon.encode({"a,b": 1}) == '"a,b": 1'

def test_quotes_keys_with_spaces_or_leading_hyphens():
    assert toon.encode({"full name": "Ada"}) == '"full name": Ada'
    assert toon.encode({"-lead": 1}) == '"-lead": 1'
    assert toon.encode({" a ": 1}) == '" a ": 1'

def test_quotes_numeric_keys():
    assert toon.encode({123: "x"}) == '"123": x'

def test_quotes_empty_string_key():
    assert toon.encode({"": 1}) == '"": 1'

def test_escapes_control_characters_in_keys():
    assert toon.encode({"line\nbreak": 1}) == '"line\\nbreak": 1'
    assert toon.encode({"tab\there": 2}) == '"tab\\there": 2'

def test_escapes_quotes_in_keys():
    assert toon.encode({'he said "hi"': 1}) == '"he said \\"hi\\"": 1'

def test_allows_valid_unquoted_keys():
    # Keys matching ^[A-Z_][\w.]*$ pattern should not be quoted
    assert toon.encode({"valid_key": 1}) == 'valid_key: 1'
    assert toon.encode({"_private": 2}) == '_private: 2'
    assert toon.encode({"CamelCase": 3}) == 'CamelCase: 3'
    assert toon.encode({"with.dots": 4}) == 'with.dots: 4'
    assert toon.encode({"with_underscores": 5}) == 'with_underscores: 5'
    assert toon.encode({"WithNumbers123": 6}) == 'WithNumbers123: 6'
```

### 3.3 Nested Objects
```python
def test_encodes_deeply_nested_objects():
    obj = {
        "a": {
            "b": {
                "c": "deep",
            },
        },
    }
    assert toon.encode(obj) == 'a:\n  b:\n    c: deep'

def test_encodes_mixed_nested_types():
    obj = {
        "user": {
            "id": 123,
            "metadata": {
                "created": datetime(2025, 1, 1),
                "tags": ["admin", "user"]
            }
        }
    }
    result = toon.encode(obj)
    assert 'user:' in result
    assert 'id: 123' in result
    assert 'metadata:' in result
    assert 'created:' in result
    assert 'tags[2]:' in result
```

## 4. Array Encoding Tests

### 4.1 Primitive Arrays
```python
def test_encodes_string_arrays_inline():
    obj = {"tags": ["reading", "gaming"]}
    assert toon.encode(obj) == 'tags[2]: reading,gaming'

def test_encodes_number_arrays_inline():
    obj = {"nums": [1, 2, 3]}
    assert toon.encode(obj) == 'nums[3]: 1,2,3'

def test_encodes_mixed_primitive_arrays_inline():
    obj = {"data": ["x", "y", True, 10]}
    assert toon.encode(obj) == 'data[4]: x,y,true,10'

def test_encodes_empty_arrays():
    obj = {"items": []}
    assert toon.encode(obj) == 'items[0]:'

def test_handles_empty_string_in_arrays():
    obj = {"items": [""]}
    assert toon.encode(obj) == 'items[1]: ""'
    
    obj2 = {"items": ["a", "", "b"]}
    assert toon.encode(obj2) == 'items[3]: a,"",b'

def test_handles_whitespace_only_strings_in_arrays():
    obj = {"items": [" ", "  "]}
    assert toon.encode(obj) == 'items[2]: " ","  "'

def test_quotes_array_strings_with_special_characters():
    obj = {"items": ["a", "b,c", "d:e"]}
    assert toon.encode(obj) == 'items[3]: a,"b,c","d:e"'

def test_quotes_ambiguous_strings_in_arrays():
    obj = {"items": ["x", "true", "42", "-3.14"]}
    assert toon.encode(obj) == 'items[4]: x,"true","42","-3.14"'

def test_quotes_structural_strings_in_arrays():
    obj = {"items": ["[5]", "- item", "{key}"]}
    assert toon.encode(obj) == 'items[3]: "[5]","- item","{key}"'
```

### 4.2 Tabular Arrays
```python
def test_encodes_similar_objects_in_tabular_format():
    obj = {
        "items": [
            {"sku": "A1", "qty": 2, "price": 9.99},
            {"sku": "B2", "qty": 1, "price": 14.5},
        ],
    }
    expected = 'items[2]{sku,qty,price}:\n  A1,2,9.99\n  B2,1,14.5'
    assert toon.encode(obj) == expected

def test_handles_null_values_in_tabular_format():
    obj = {
        "items": [
            {"id": 1, "value": None},
            {"id": 2, "value": "test"},
        ],
    }
    expected = 'items[2]{id,value}:\n  1,null\n  2,test'
    assert toon.encode(obj) == expected

def test_quotes_strings_containing_delimiters_in_tabular_rows():
    obj = {
        "items": [
            {"sku": "A,1", "desc": "cool", "qty": 2},
            {"sku": "B2", "desc": "wip: test", "qty": 1},
        ],
    }
    expected = 'items[2]{sku,desc,qty}:\n  "A,1",cool,2\n  B2,"wip: test",1'
    assert toon.encode(obj) == expected

def test_handles_tabular_arrays_with_keys_needing_quotes():
    obj = {
        "items": [
            {"order:id": 1, "full name": "Ada"},
            {"order:id": 2, "full name": "Bob"},
        ],
    }
    expected = 'items[2]{"order:id","full name"}:\n  1,Ada\n  2,Bob'
    assert toon.encode(obj) == expected

def test_uses_field_order_from_first_object():
    obj = {
        "items": [
            {"a": 1, "b": 2, "c": 3},
            {"c": 30, "b": 20, "a": 10},
        ],
    }
    expected = 'items[2]{a,b,c}:\n  1,2,3\n  10,20,30'
    assert toon.encode(obj) == expected
```

### 4.3 Mixed and Non-Uniform Arrays
```python
def test_uses_list_format_for_objects_with_different_fields():
    obj = {
        "items": [
            {"id": 1, "name": "First"},
            {"id": 2, "name": "Second", "extra": True},
        ],
    }
    expected = ('items[2]:\n'
               '  - id: 1\n'
               '    name: First\n'
               '  - id: 2\n'
               '    name: Second\n'
               '    extra: true')
    assert toon.encode(obj) == expected

def test_uses_list_format_for_objects_with_nested_values():
    obj = {
        "items": [
            {"id": 1, "nested": {"x": 1}},
        ],
    }
    expected = ('items[1]:\n'
               '  - id: 1\n'
               '    nested:\n'
               '      x: 1')
    assert toon.encode(obj) == expected

def test_preserves_field_order_in_list_items():
    obj = {"items": [{"nums": [1, 2, 3], "name": "test"}]}
    expected = ('items[1]:\n'
               '  - nums[3]: 1,2,3\n'
               '    name: test')
    assert toon.encode(obj) == expected

def test_preserves_field_order_when_primitive_appears_first():
    obj = {"items": [{"name": "test", "nums": [1, 2, 3]}]}
    expected = ('items[1]:\n'
               '  - name: test\n'
               '    nums[3]: 1,2,3')
    assert toon.encode(obj) == expected
```

### 4.4 Arrays of Arrays
```python
def test_encodes_nested_arrays_of_primitives():
    obj = {
        "pairs": [["a", "b"], ["c", "d"]],
    }
    expected = 'pairs[2]:\n  - [2]: a,b\n  - [2]: c,d'
    assert toon.encode(obj) == expected

def test_quotes_strings_containing_delimiters_in_nested_arrays():
    obj = {
        "pairs": [["a", "b"], ["c,d", "e:f", "true"]],
    }
    expected = 'pairs[2]:\n  - [2]: a,b\n  - [3]: "c,d","e:f","true"'
    assert toon.encode(obj) == expected

def test_handles_empty_inner_arrays():
    obj = {
        "pairs": [[], []],
    }
    expected = 'pairs[2]:\n  - [0]:\n  - [0]:'
    assert toon.encode(obj) == expected

def test_handles_mixed_length_inner_arrays():
    obj = {
        "pairs": [[1], [2, 3]],
    }
    expected = 'pairs[2]:\n  - [1]: 1\n  - [2]: 2,3'
    assert toon.encode(obj) == expected
```

### 4.5 Root Arrays
```python
def test_encodes_arrays_of_primitives_at_root_level():
    arr = ["x", "y", "true", True, 10]
    assert toon.encode(arr) == '[5]: x,y,"true",true,10'

def test_encodes_arrays_of_similar_objects_in_tabular_format():
    arr = [{"id": 1}, {"id": 2}]
    expected = '[2]{id}:\n  1\n  2'
    assert toon.encode(arr) == expected

def test_encodes_arrays_of_different_objects_in_list_format():
    arr = [{"id": 1}, {"id": 2, "name": "Ada"}]
    expected = '[2]:\n  - id: 1\n  - id: 2\n    name: Ada'
    assert toon.encode(arr) == expected

def test_encodes_empty_arrays_at_root_level():
    assert toon.encode([]) == '[0]:'

def test_encodes_arrays_of_arrays_at_root_level():
    arr = [[1, 2], []]
    expected = '[2]:\n  - [2]: 1,2\n  - [0]:'
    assert toon.encode(arr) == expected
```

## 5. Options Tests

### 5.1 Delimiter Options
```python
def test_comma_delimiter_default():
    obj = {"tags": ["reading", "gaming", "coding"]}
    assert toon.encode(obj) == 'tags[3]: reading,gaming,coding'

def test_tab_delimiter():
    obj = {"tags": ["reading", "gaming", "coding"]}
    options = EncodeOptions(delimiter=Delimiter.TAB)
    expected = 'tags[3\t]: reading\tgaming\tcoding'
    assert toon.encode(obj, options) == expected

def test_pipe_delimiter():
    obj = {"tags": ["reading", "gaming", "coding"]}
    options = EncodeOptions(delimiter=Delimiter.PIPE)
    expected = 'tags[3|]: reading|gaming|coding'
    assert toon.encode(obj, options) == expected

def test_tabular_with_different_delimiters():
    obj = {
        "items": [
            {"sku": "A1", "name": "Widget", "qty": 2, "price": 9.99},
            {"sku": "B2", "name": "Gadget", "qty": 1, "price": 14.5}
        ]
    }
    
    # Tab delimiter
    options_tab = EncodeOptions(delimiter=Delimiter.TAB)
    expected_tab = ('items[2\t]{sku\tname\tqty\tprice}:\n'
                  '  A1\tWidget\t2\t9.99\n'
                  '  B2\tGadget\t1\t14.5')
    assert toon.encode(obj, options_tab) == expected_tab
    
    # Pipe delimiter
    options_pipe = EncodeOptions(delimiter=Delimiter.PIPE)
    expected_pipe = ('items[2|]{sku|name|qty|price}:\n'
                   '  A1|Widget|2|9.99\n'
                   '  B2|Gadget|1|14.5')
    assert toon.encode(obj, options_pipe) == expected_pipe

def test_delimiter_aware_quoting():
    # With tab delimiter, commas should not be quoted
    obj = {"items": ["a,b", "c,d"]}
    options = EncodeOptions(delimiter=Delimiter.TAB)
    expected = 'items[2\t]: a,b\tc,d'
    assert toon.encode(obj, options) == expected
    
    # With pipe delimiter, commas should not be quoted
    options_pipe = EncodeOptions(delimiter=Delimiter.PIPE)
    expected_pipe = 'items[2|]: a,b|c,d'
    assert toon.encode(obj, options_pipe) == expected_pipe
    
    # But tabs should be quoted with tab delimiter
    obj_tab = {"items": ["a", "b\tc", "d"]}
    expected_tab_quoted = 'items[3\t]: a\t"b\\tc"\td'
    assert toon.encode(obj_tab, options) == expected_tab_quoted
```

### 5.2 Length Marker Options
```python
def test_length_marker_disabled_by_default():
    obj = {"tags": ["reading", "gaming", "coding"]}
    assert toon.encode(obj) == 'tags[3]: reading,gaming,coding'

def test_length_marker_enabled():
    obj = {"tags": ["reading", "gaming", "coding"]}
    options = EncodeOptions(length_marker='#')
    assert toon.encode(obj, options) == 'tags[#3]: reading,gaming,coding'

def test_length_marker_with_empty_arrays():
    obj = {"items": []}
    options = EncodeOptions(length_marker='#')
    assert toon.encode(obj, options) == 'items[#0]:'

def test_length_marker_with_tabular_arrays():
    obj = {
        "items": [
            {"sku": "A1", "qty": 2, "price": 9.99},
            {"sku": "B2", "qty": 1, "price": 14.5},
        ],
    }
    options = EncodeOptions(length_marker='#')
    expected = ('items[#2]{sku,qty,price}:\n'
               '  A1,2,9.99\n'
               '  B2,1,14.5')
    assert toon.encode(obj, options) == expected

def test_length_marker_with_delimiter():
    obj = {"tags": ["reading", "gaming", "coding"]}
    options = EncodeOptions(length_marker='#', delimiter=Delimiter.PIPE)
    expected = 'tags[#3|]: reading|gaming|coding'
    assert toon.encode(obj, options) == expected
```

### 5.3 Indentation Options
```python
def test_custom_indentation():
    obj = {"user": {"id": 123, "name": "Ada"}}
    options = EncodeOptions(indent=4)
    expected = 'user:\n    id: 123\n    name: Ada'
    assert toon.encode(obj, options) == expected

def test_zero_indentation():
    obj = {"user": {"id": 123, "name": "Ada"}}
    options = EncodeOptions(indent=0)
    expected = 'user:\nid: 123\nname: Ada'
    assert toon.encode(obj, options) == expected
```

## 6. Formatting Invariant Tests

### 6.1 Whitespace Handling
```python
def test_no_trailing_spaces_at_end_of_lines():
    obj = {
        "user": {
            "id": 123,
            "name": "Ada",
        },
        "items": ["a", "b"],
    }
    result = toon.encode(obj)
    lines = result.split('\n')
    for line in lines:
        assert not line.endswith(' '), f"Line has trailing space: '{line}'"

def test_no_trailing_newline_at_end_of_output():
    obj = {"id": 123}
    result = toon.encode(obj)
    assert not result.endswith('\n'), "Output has trailing newline"

def test_consistent_indentation():
    obj = {
        "level1": {
            "level2": {
                "level3": "deep"
            }
        }
    }
    result = toon.encode(obj)
    lines = result.split('\n')
    
    # Check that indentation is consistent
    for i, line in enumerate(lines):
        if line.strip():  # Skip empty lines
            # Count leading spaces
            leading_spaces = len(line) - len(line.lstrip(' '))
            # Should be multiple of 2 (default indent)
            assert leading_spaces % 2 == 0, f"Inconsistent indentation in line: '{line}'"
```

## 7. Complex Structure Tests

### 7.1 Mixed Nested Structures
```python
def test_encodes_objects_with_mixed_arrays_and_nested_objects():
    obj = {
        "user": {
            "id": 123,
            "name": "Ada",
            "tags": ["reading", "gaming"],
            "active": True,
            "prefs": [],
        },
    }
    expected = ('user:\n'
               '  id: 123\n'
               '  name: Ada\n'
               '  tags[2]: reading,gaming\n'
               '  active: true\n'
               '  prefs[0]:')
    assert toon.encode(obj) == expected

def test_encodes_deeply_nested_with_various_types():
    obj = {
        "data": [
            {
                "id": 1,
                "metadata": {
                    "created": datetime(2025, 1, 1),
                    "tags": ["admin", "user"],
                    "score": Decimal("99.5"),
                    "uuid": UUID("550e8400-e29b-41d4-a716-446655440000")
                },
                "binary": b"hello"
            }
        ]
    }
    result = toon.encode(obj)
    
    # Verify structure is correct
    assert 'data[1]:' in result
    assert 'id: 1' in result
    assert 'metadata:' in result
    assert 'created:' in result
    assert 'tags[2]:' in result
    assert 'score:' in result
    assert 'uuid:' in result
    assert 'binary:' in result
```

## 8. Error Handling Tests

### 8.1 Invalid Options
```python
def test_invalid_delimiter_type():
    with pytest.raises(TypeError):
        toon.encode({"a": 1}, EncodeOptions(delimiter="invalid"))

def test_invalid_indentation_type():
    with pytest.raises(TypeError):
        toon.encode({"a": 1}, EncodeOptions(indent="invalid"))

def test_invalid_length_marker_type():
    with pytest.raises(TypeError):
        toon.encode({"a": 1}, EncodeOptions(length_marker=123))

def test_invalid_length_marker_value():
    with pytest.raises(ValueError):
        toon.encode({"a": 1}, EncodeOptions(length_marker="invalid"))
```

### 8.2 Edge Cases
```python
def test_very_deep_nesting():
    # Test with very deep nesting (should handle gracefully)
    deep_obj = {}
    current = deep_obj
    for i in range(100):
        current[f"level{i}"] = {}
        current = current[f"level{i}"]
    current["value"] = "deep"
    
    result = toon.encode(deep_obj)
    # Should produce very long output but not crash
    assert len(result) > 1000
    assert "deep" in result

def test_very_large_arrays():
    # Test with large arrays
    large_array = list(range(10000))
    result = toon.encode(large_array)
    assert result.startswith('[10000]:')
    assert len(result) > 50000  # Should be substantial

def test_circular_references():
    # Python doesn't have built-in circular reference detection like some other languages
    # But we should test that our implementation doesn't crash on self-references
    # (This would require special handling in the normalizer)
    pass  # Implementation dependent
```

## 9. Performance Tests

### 9.1 Benchmark Tests
```python
def test_performance_large_objects():
    import time
    
    large_obj = {
        "users": [
            {"id": i, "name": f"user{i}", "email": f"user{i}@example.com"}
            for i in range(1000)
        ]
    }
    
    start_time = time.time()
    result = toon.encode(large_obj)
    end_time = time.time()
    
    # Should complete within reasonable time (adjust threshold as needed)
    assert end_time - start_time < 1.0, f"Encoding took too long: {end_time - start_time}s"
    assert len(result) > 0

def test_memory_usage():
    import psutil
    import os
    
    process = psutil.Process(os.getpid())
    initial_memory = process.memory_info().rss
    
    # Create and encode a large structure
    large_data = {
        "data": [
            {"id": i, "values": list(range(100))}
            for i in range(1000)
        ]
    }
    
    result = toon.encode(large_data)
    final_memory = process.memory_info().rss
    
    # Memory usage should be reasonable (within 10MB of initial + data size)
    memory_increase = (final_memory - initial_memory) / 1024 / 1024
    assert memory_increase < 10, f"Memory usage too high: {memory_increase:.2f}MB"
    
    # Clean up
    del result
    del large_data
```

## 10. Property-Based Tests

### 10.1 String Property Tests
```python
from hypothesis import given, strategies as st

@given(st.text())
def test_string_encoding_roundtrip(s):
    """Test that encoding doesn't crash on any string"""
    try:
        result = toon.encode(s)
        # Result should be a string
        assert isinstance(result, str)
    except Exception as e:
        pytest.fail(f"Encoding failed for string '{s}': {e}")

@given(st.lists(st.text(), max_size=10))
def test_list_encoding_properties(lst):
    """Test properties of list encoding"""
    result = toon.encode(lst)
    
    # Should start with [N]: where N is the length
    assert result.startswith(f'[{len(lst)}]:')
    
    # Should contain all string representations of elements
    for item in lst:
        assert str(item) in result

@given(st.dictionaries(st.text(), st.text(), max_size=10))
def test_dict_encoding_properties(d):
    """Test properties of dictionary encoding"""
    result = toon.encode(d)
    
    # Should contain all keys
    for key in d:
        assert str(key) in result
    
    # Should contain all values
    for value in d.values():
        assert str(value) in result
```

## 11. Integration Tests

### 11.1 Real-world Data Tests
```python
def test_github_api_response_format():
    """Test encoding of realistic GitHub API response data"""
    github_data = {
        "id": 28457823,
        "name": "freeCodeCamp",
        "full_name": "freeCodeCamp/freeCodeCamp",
        "description": "freeCodeCamp.org's open-source codebase and curriculum.",
        "created_at": datetime(2014, 12, 24, 17, 49, 19),
        "updated_at": datetime(2025, 10, 27, 7, 40, 58),
        "stargazers_count": 430828,
        "language": "JavaScript",
        "topics": ["education", "javascript", "react", "nodejs"],
        "owner": {
            "login": "freecodecamp",
            "id": 9892522,
            "type": "Organization"
        }
    }
    
    result = toon.encode(github_data)
    
    # Verify structure
    assert 'id: 28457823' in result
    assert 'name: freeCodeCamp' in result
    assert 'created_at:' in result
    assert 'topics[4]:' in result
    assert 'owner:' in result
    assert 'login: freecodecamp' in result

def test_analytics_time_series():
    """Test encoding of time series data"""
    import random
    
    metrics = {
        "metrics": [
            {
                "date": date(2024, 12, 31 + i),
                "views": random.randint(1000, 5000),
                "clicks": random.randint(100, 500),
                "revenue": round(random.uniform(50, 500), 2)
            }
            for i in range(30)
        ]
    }
    
    result = toon.encode(metrics)
    
    # Should use tabular format for uniform objects
    assert 'metrics[30]{date,views,clicks,revenue}:' in result
    assert result.count('\n') == 31  # Header + 30 data rows
```

This comprehensive test specification ensures that the Python implementation:

1. **Maintains feature parity** with the TypeScript version
2. **Handles Python-specific types** correctly (datetime, Decimal, UUID, bytes, etc.)
3. **Follows Python conventions** for error handling and type safety
4. **Provides comprehensive coverage** of all encoding scenarios
5. **Includes performance and property-based tests** for robustness
6. **Tests real-world scenarios** to ensure practical utility

The test suite can be implemented incrementally, starting with basic functionality and expanding to cover all edge cases and Python-specific features.