"""
Tests for main encoder functionality and array handling.
"""

import pytest
from hypothesis import given
from hypothesis import strategies as st

from src.toon_python.constants import ArrayType, Delimiter, EncodeOptions
from src.toon_python.encoder import (
    CircularReferenceError,
    ToonEncoder,
)


class TestToonEncoder:
    """Test cases for ToonEncoder class."""

    def setup_method(self):
        """Set up test fixtures."""
        self.options = EncodeOptions()
        self.encoder = ToonEncoder(self.options)

    def test_encoder_initialization(self):
        """Test encoder initialization with options."""
        assert self.encoder.options == self.options

    def test_encoder_initialization_with_custom_options(self):
        """Test encoder initialization with custom options."""
        custom_options = EncodeOptions(indent=4, delimiter=Delimiter.TAB)
        encoder = ToonEncoder(custom_options)
        assert encoder.options == custom_options

    def test_encode_simple_object(self):
        """Test encoding a simple object."""
        data = {"name": "Alice", "age": 30}
        result = self.encoder.encode(data)
        expected = "name: Alice\nage: 30"
        assert result == expected

    def test_encode_empty_object(self):
        """Test encoding an empty object."""
        data = {}
        result = self.encoder.encode(data)
        assert result == ""

    def test_encode_nested_object(self):
        """Test encoding a nested object."""
        data = {"user": {"name": "Bob", "age": 25}}
        result = self.encoder.encode(data)
        expected = "user:\n  name: Bob\n  age: 25"
        assert result == expected

    def test_encode_array(self):
        """Test encoding an array."""
        data = [1, 2, 3]
        result = self.encoder.encode(data)
        expected = "1,2,3"
        assert result == expected

    def test_encode_primitive(self):
        """Test encoding a primitive value."""
        assert self.encoder.encode("hello") == "hello"
        assert self.encoder.encode(42) == "42"
        assert self.encoder.encode(True) == "true"
        assert self.encoder.encode(None) == "null"


class TestArrayAnalyzer:
    """Test cases for array analysis functionality."""

    def setup_method(self):
        """Set up test fixtures."""
        from src.toon_python.encoder import ArrayAnalyzer

        self.analyzer = ArrayAnalyzer()

    def test_analyze_primitive_array(self):
        """Test analysis of primitive arrays."""
        arr = [1, 2, 3, 4, 5]
        result = self.analyzer.analyze_array(arr)
        assert result == ArrayType.INLINE

    def test_analyze_uniform_object_array(self):
        """Test analysis of uniform object arrays."""
        arr = [
            {"sku": "A1", "qty": 2, "price": 9.99},
            {"sku": "B2", "qty": 1, "price": 14.5},
        ]
        result = self.analyzer.analyze_array(arr)
        assert result == ArrayType.TABULAR

    def test_analyze_mixed_array(self):
        """Test analysis of mixed arrays."""
        arr = ["a", 1, True, None]
        result = self.analyzer.analyze_array(arr)
        # Mixed primitives should be INLINE
        assert result == ArrayType.INLINE

    def test_analyze_empty_array(self):
        """Test analysis of empty arrays."""
        arr = []
        result = self.analyzer.analyze_array(arr)
        assert result == ArrayType.LIST

    def test_is_uniform_objects(self):
        """Test uniform object detection."""
        uniform_arr = [{"name": "Alice", "age": 30}, {"name": "Bob", "age": 25}]
        non_uniform_arr = [{"name": "Alice", "age": 30}, {"name": "Bob", "height": 180}]

        assert self.analyzer.is_uniform_objects(uniform_arr) is True
        assert self.analyzer.is_uniform_objects(non_uniform_arr) is False

    def test_get_tabular_headers(self):
        """Test tabular header extraction."""
        arr = [
            {"sku": "A1", "qty": 2, "price": 9.99},
            {"sku": "B2", "qty": 1, "price": 14.5},
        ]
        headers = self.analyzer.get_tabular_headers(arr)
        assert headers == ["sku", "qty", "price"]


class TestArrayOptimization:
    """Test cases for array optimization strategies."""

    def test_inline_array_encoding(self):
        """Test inline array encoding for primitives."""
        data = {"tags": ["admin", "ops", "dev"]}
        encoder = ToonEncoder(EncodeOptions())
        result = encoder.encode(data)
        expected = "tags[3]: admin,ops,dev"
        assert result == expected

    def test_tabular_array_encoding(self):
        """Test tabular array encoding for uniform objects."""
        data = {
            "items": [
                {"sku": "A1", "qty": 2, "price": 9.99},
                {"sku": "B2", "qty": 1, "price": 14.5},
            ]
        }
        encoder = ToonEncoder(EncodeOptions())
        result = encoder.encode(data)
        expected = "items[2]{sku,qty,price}:\n  A1,2,9.99\n  B2,1,14.5"
        assert result == expected

    def test_list_array_encoding(self):
        """Test list array encoding for mixed/nested arrays."""
        data = {"mixed": ["a", {"nested": "value"}, 1]}
        encoder = ToonEncoder(EncodeOptions())
        result = encoder.encode(data)
        expected = "mixed[3]:\n  - a\n  nested: value\n  - 1"
        assert result == expected


class TestErrorHandling:
    """Test cases for error handling."""

    def test_circular_reference_detection(self):
        """Test circular reference detection."""
        data = {}
        data["self"] = data

        encoder = ToonEncoder(EncodeOptions())
        with pytest.raises(CircularReferenceError):
            encoder.encode(data)

    def test_non_serializable_objects(self):
        """Test handling of non-serializable objects."""

        def test_func():
            pass

        encoder = ToonEncoder(EncodeOptions())
        # Functions should be converted to None by normalizer
        result = encoder.encode(test_func)
        assert result == "null"


# Property-based tests
@given(st.dictionaries(keys=st.text(), values=st.integers()))
def test_property_based_object_encoding(data):
    """Property-based test for object encoding."""
    encoder = ToonEncoder(EncodeOptions())
    result = encoder.encode(data)
    assert isinstance(result, str)


@given(st.lists(st.integers()))
def test_property_based_array_encoding(data):
    """Property-based test for array encoding."""
    encoder = ToonEncoder(EncodeOptions())
    result = encoder.encode(data)
    assert isinstance(result, str)


@given(st.one_of(st.integers(), st.text(), st.booleans(), st.none()))
def test_property_based_primitive_encoding(data):
    """Property-based test for primitive encoding."""
    encoder = ToonEncoder(EncodeOptions())
    result = encoder.encode(data)
    assert isinstance(result, str)
