"""
Integration tests for end-to-end encoding workflow.
"""

from hypothesis import given
from hypothesis import strategies as st

from toon_python import Delimiter, EncodeOptions, encode


class TestEndToEndEncoding:
    """Test cases for end-to-end encoding workflow."""

    def test_encode_simple_dictionary(self):
        """Test encoding a simple dictionary."""
        data = {"name": "Alice", "age": 30, "active": True}
        result = encode(data)
        expected = "name: Alice\nage: 30\nactive: true"
        assert result == expected

    def test_encode_nested_structures(self):
        """Test encoding nested structures."""
        data = {
            "user": {
                "profile": {
                    "name": "Bob",
                    "preferences": {"theme": "dark", "notifications": True},
                }
            }
        }
        result = encode(data)
        expected = "user:\n  profile:\n    name: Bob\n    preferences:\n      theme: dark\n      notifications: true"
        assert result == expected

    def test_encode_arrays(self):
        """Test encoding various array types."""
        data = {
            "tags": ["admin", "ops", "dev"],
            "numbers": [1, 2, 3, 4, 5],
            "mixed": ["a", 1, True, None],
        }
        result = encode(data)
        expected = (
            "tags[3]: admin,ops,dev\nnumbers[5]: 1,2,3,4,5\nmixed[4]: a,1,true,null"
        )
        assert result == expected

    def test_encode_with_custom_options(self):
        """Test encoding with custom options."""
        data = {"items": ["a", "b", "c"]}
        options = EncodeOptions(indent=4, delimiter=Delimiter.TAB, length_marker="#")
        result = encode(data, options)
        expected = "items#[3]: a\tb\tc"
        assert result == expected

    def test_encode_empty_structures(self):
        """Test encoding empty structures."""
        test_cases = [
            ({}, ""),
            ([], ""),
            ({"empty_dict": {}, "empty_list": []}, "empty_dict:\nempty_list[0]:"),
        ]

        for data, expected in test_cases:
            result = encode(data)
            assert result == expected

    def test_encode_special_values(self):
        """Test encoding special values."""
        data = {
            "null_value": None,
            "boolean_true": True,
            "boolean_false": False,
            "zero": 0,
            "negative": -42,
            "float_val": 3.14,
        }
        result = encode(data)
        expected = "null_value: null\nboolean_true: true\nboolean_false: false\nzero: 0\nnegative: -42\nfloat_val: 3.14"
        assert result == expected


class TestTokenReduction:
    """Test cases for token reduction verification."""

    def test_token_reduction_basic(self):
        """Test basic token reduction verification."""
        # This will be implemented when we have actual encoding
        # For now, just ensure the function runs
        data = {"key": "value", "number": 42}
        result = encode(data)
        assert isinstance(result, str)

    def test_token_reduction_with_arrays(self):
        """Test token reduction with arrays."""
        # This will be implemented when we have actual encoding
        data = {"items": ["a", "b", "c", "d", "e"]}
        result = encode(data)
        assert isinstance(result, str)

    def test_token_reduction_with_nested_objects(self):
        """Test token reduction with nested objects."""
        # This will be implemented when we have actual encoding
        data = {"users": [{"name": "Alice", "age": 30}, {"name": "Bob", "age": 25}]}
        result = encode(data)
        assert isinstance(result, str)


class TestFormattingOptions:
    """Test cases for different formatting options."""

    def test_comma_delimiter(self):
        """Test comma delimiter formatting."""
        data = {"items": [1, 2, 3]}
        options = EncodeOptions(delimiter=Delimiter.COMMA)
        result = encode(data, options)
        expected = "items[3]: 1,2,3"
        assert result == expected

    def test_tab_delimiter(self):
        """Test tab delimiter formatting."""
        data = {"items": [1, 2, 3]}
        options = EncodeOptions(delimiter=Delimiter.TAB)
        result = encode(data, options)
        expected = "items[3]: 1\t2\t3"
        assert result == expected

    def test_pipe_delimiter(self):
        """Test pipe delimiter formatting."""
        data = {"items": [1, 2, 3]}
        options = EncodeOptions(delimiter=Delimiter.PIPE)
        result = encode(data, options)
        expected = "items[3]: 1|2|3"
        assert result == expected

    def test_different_indentation(self):
        """Test different indentation levels."""
        data = {"nested": {"key": "value"}}

        test_cases = [
            (0, "nested:\nkey: value"),
            (2, "nested:\n  key: value"),
            (4, "nested:\n    key: value"),
            (8, "nested:\n        key: value"),
        ]

        for indent, expected in test_cases:
            options = EncodeOptions(indent=indent)
            result = encode(data, options)
            assert result == expected

    def test_length_marker(self):
        """Test length marker option."""
        data = {"items": [1, 2, 3]}
        options = EncodeOptions(length_marker="#")
        result = encode(data, options)
        expected = "items#[3]: 1,2,3"
        assert result == expected


# Property-based integration tests
@given(st.dictionaries(keys=st.text(), values=st.integers()))
def test_property_based_dictionary_encoding(data):
    """Property-based test for dictionary encoding."""
    result = encode(data)
    assert isinstance(result, str)


@given(st.lists(st.one_of([st.integers(), st.text(), st.booleans()])))
def test_property_based_list_encoding(data):
    """Property-based test for list encoding."""
    result = encode(data)
    assert isinstance(result, str)


@given(
    st.one_of(
        [
            st.dictionaries(keys=st.text(), values=st.integers()),
            st.lists(st.integers()),
            st.integers(),
            st.text(),
            st.booleans(),
            st.none(),
        ]
    )
)
def test_property_based_mixed_encoding(data):
    """Property-based test for mixed data encoding."""
    result = encode(data)
    assert isinstance(result, str)
