"""
Tests for type normalization and numeric edge cases.
"""

from datetime import date, datetime
from decimal import Decimal
from uuid import UUID

from hypothesis import given
from hypothesis import strategies as st

from src.toon_python.normalizer import ValueNormalizer


class TestValueNormalizer:
    """Test cases for ValueNormalizer class."""

    def setup_method(self):
        """Set up test fixtures."""
        self.normalizer = ValueNormalizer()

    def test_normalize_none(self):
        """Test None normalization."""
        result = self.normalizer.normalize(None)
        assert result is None

    def test_normalize_string(self):
        """Test string normalization."""
        result = self.normalizer.normalize("hello")
        assert result == "hello"

    def test_normalize_integer(self):
        """Test integer normalization."""
        result = self.normalizer.normalize(42)
        assert result == 42

        result = self.normalizer.normalize(-17)
        assert result == -17

    def test_normalize_boolean(self):
        """Test boolean normalization."""
        assert self.normalizer.normalize(True) is True
        assert self.normalizer.normalize(False) is False

    def test_normalize_negative_zero(self):
        """Test -0 to 0 conversion."""
        result = self.normalizer.normalize(-0.0)
        assert result == 0
        assert str(result) == "0"

    def test_normalize_nan(self):
        """Test NaN to null conversion."""
        result = self.normalizer.normalize(float('nan'))
        assert result is None

    def test_normalize_positive_infinity(self):
        """Test +inf to null conversion."""
        result = self.normalizer.normalize(float('inf'))
        assert result is None

    def test_normalize_negative_infinity(self):
        """Test -inf to null conversion."""
        result = self.normalizer.normalize(float('-inf'))
        assert result is None

    def test_normalize_regular_floats(self):
        """Test regular float normalization."""
        assert self.normalizer.normalize(3.14) == 3.14
        assert self.normalizer.normalize(-2.5) == -2.5
        assert self.normalizer.normalize(0.0) == 0.0

    def test_normalize_dictionary(self):
        """Test dictionary normalization."""
        data = {"key": "value", "number": 42}
        result = self.normalizer.normalize(data)
        expected = {"key": "value", "number": 42}
        assert result == expected

    def test_normalize_nested_dictionary(self):
        """Test nested dictionary normalization."""
        data = {"outer": {"inner": "value"}}
        result = self.normalizer.normalize(data)
        expected = {"outer": {"inner": "value"}}
        assert result == expected

    def test_normalize_list(self):
        """Test list normalization."""
        data = [1, "two", True]
        result = self.normalizer.normalize(data)
        expected = [1, "two", True]
        assert result == expected

    def test_normalize_tuple(self):
        """Test tuple normalization."""
        data = (1, "two", True)
        result = self.normalizer.normalize(data)
        expected = [1, "two", True]
        assert result == expected

    def test_normalize_datetime(self):
        """Test datetime normalization."""
        dt = datetime(2023, 1, 15, 12, 30, 45)
        result = self.normalizer.normalize(dt)
        assert result == "2023-01-15T12:30:45"

    def test_normalize_date(self):
        """Test date normalization."""
        d = date(2023, 1, 15)
        result = self.normalizer.normalize(d)
        assert result == "2023-01-15"

    def test_normalize_decimal(self):
        """Test Decimal normalization."""
        dec = Decimal("123.45")
        result = self.normalizer.normalize(dec)
        assert result == "123.45"

    def test_normalize_uuid(self):
        """Test UUID normalization."""
        uid = UUID("12345678-1234-5678-9abc-123456789abc")
        result = self.normalizer.normalize(uid)
        assert result == "12345678-1234-5678-9abc-123456789abc"

    def test_normalize_bytes(self):
        """Test bytes normalization."""
        data = b"hello"
        result = self.normalizer.normalize(data)
        assert result == "aGVsbG8="  # Base64 encoding

    def test_normalize_set(self):
        """Test set normalization."""
        data = {1, 2, 3}
        result = self.normalizer.normalize(data)
        assert set(result) == {1, 2, 3}  # Order doesn't matter for sets
        assert isinstance(result, list)

    def test_normalize_frozenset(self):
        """Test frozenset normalization."""
        data = frozenset([1, 2, 3])
        result = self.normalizer.normalize(data)
        assert set(result) == {1, 2, 3}  # Order doesn't matter for frozensets
        assert isinstance(result, list)

    def test_normalize_unsupported_object(self):
        """Test unsupported object normalization."""
        class CustomClass:
            pass

        obj = CustomClass()
        result = self.normalizer.normalize(obj)
        assert result is None

    def test_normalize_function(self):
        """Test function normalization."""
        def test_func():
            pass

        result = self.normalizer.normalize(test_func)
        assert result is None

    def test_normalize_complex_nested_structure(self):
        """Test complex nested structure normalization."""
        data = {
            "users": [
                {
                    "id": 1,
                    "name": "Alice",
                    "created": datetime(2023, 1, 1),
                    "tags": {"admin", "user"},
                    "metadata": None
                }
            ],
            "count": 1.0,
            "active": True
        }

        result = self.normalizer.normalize(data)


        # Check that sets are converted to lists (order may vary)
        assert isinstance(result, dict)
        assert isinstance(result["users"], list)
        assert isinstance(result["users"][0]["tags"], list)
        assert set(result["users"][0]["tags"]) == {"admin", "user"}


class TestNumericEdgeCases:
    """Specific tests for numeric edge cases."""

    def setup_method(self):
        """Set up test fixtures."""
        self.normalizer = ValueNormalizer()

    def test_all_nan_representations(self):
        """Test different NaN representations."""
        import math

        nan_values = [
            float('nan'),
            math.nan,
            float('inf') - float('inf'),  # inf - inf = nan
        ]

        for nan_val in nan_values:
            result = self.normalizer.normalize(nan_val)
            assert result is None

    def test_all_infinity_representations(self):
        """Test different infinity representations."""
        inf_values = [
            float('inf'),
            float('-inf'),
            1e308 * 2,  # Overflow to inf
            -1e308 * 2,  # Underflow to -inf
        ]

        for inf_val in inf_values:
            result = self.normalizer.normalize(inf_val)
            assert result is None

    def test_negative_zero_variations(self):
        """Test different negative zero representations."""
        negative_zeros = [
            -0.0,
            -0.0 + 0.0,
            float('-0.0'),
        ]

        for neg_zero in negative_zeros:
            result = self.normalizer.normalize(neg_zero)
            assert result == 0
            # Note: str(0) returns "0", but str(0.0) returns "0.0"
            # Both are valid representations of zero

    def test_very_large_numbers(self):
        """Test very large numbers."""
        large_numbers = [
            10**100,
            2**1023,
            999999999999999999999,
        ]

        for large_num in large_numbers:
            result = self.normalizer.normalize(large_num)
            assert result == large_num

    def test_very_small_numbers(self):
        """Test very small numbers."""
        small_numbers = [
            1e-100,
            1e-323,
            0.0000000000000001,
        ]

        for small_num in small_numbers:
            result = self.normalizer.normalize(small_num)
            assert result == small_num


# Property-based tests
@given(st.integers())
def test_property_based_integer_normalization(value):
    """Property-based test for integer normalization."""
    normalizer = ValueNormalizer()
    result = normalizer.normalize(value)
    assert result == value


@given(st.floats(allow_nan=False, allow_infinity=False))
def test_property_based_float_normalization(value):
    """Property-based test for float normalization."""
    normalizer = ValueNormalizer()
    result = normalizer.normalize(value)
    assert result == value


@given(st.text())
def test_property_based_string_normalization(value):
    """Property-based test for string normalization."""
    normalizer = ValueNormalizer()
    result = normalizer.normalize(value)
    assert result == value


@given(st.booleans())
def test_property_based_boolean_normalization(value):
    """Property-based test for boolean normalization."""
    normalizer = ValueNormalizer()
    result = normalizer.normalize(value)
    assert result == value


@given(st.lists(st.integers()))
def test_property_based_list_normalization(value):
    """Property-based test for list normalization."""
    normalizer = ValueNormalizer()
    result = normalizer.normalize(value)
    assert result == value


@given(st.dictionaries(keys=st.text(), values=st.integers()))
def test_property_based_dict_normalization(value):
    """Property-based test for dictionary normalization."""
    normalizer = ValueNormalizer()
    result = normalizer.normalize(value)
    assert result == value
