"""
Tests for primitive encoding and quoting.
"""

from hypothesis import given
from hypothesis import strategies as st

from src.toon_python.primitives import PrimitiveEncoder


class TestPrimitiveEncoder:
    """Test cases for PrimitiveEncoder class."""

    def setup_method(self):
        """Set up test fixtures."""
        self.encoder = PrimitiveEncoder()

    def test_encode_null(self):
        """Test null encoding."""
        result = self.encoder.encode_primitive(None, ',')
        assert result == 'null'

    def test_encode_boolean(self):
        """Test boolean encoding."""
        assert self.encoder.encode_primitive(True, ',') == 'true'
        assert self.encoder.encode_primitive(False, ',') == 'false'

    def test_encode_integer(self):
        """Test integer encoding."""
        assert self.encoder.encode_primitive(42, ',') == '42'
        assert self.encoder.encode_primitive(-17, ',') == '-17'
        assert self.encoder.encode_primitive(0, ',') == '0'

    def test_encode_float(self):
        """Test float encoding."""
        assert self.encoder.encode_primitive(3.14, ',') == '3.14'
        assert self.encoder.encode_primitive(-2.5, ',') == '-2.5'
        assert self.encoder.encode_primitive(0.0, ',') == '0.0'

    def test_encode_simple_string(self):
        """Test simple string encoding without quotes."""
        result = self.encoder.encode_primitive('hello', ',')
        assert result == 'hello'

    def test_encode_string_with_delimiter(self):
        """Test string encoding with delimiter requires quotes."""
        result = self.encoder.encode_primitive('hello,world', ',')
        assert result == '"hello,world"'

    def test_encode_string_with_colon(self):
        """Test string encoding with colon requires quotes."""
        result = self.encoder.encode_primitive('key:value', ',')
        assert result == '"key:value"'

    def test_encode_string_with_quotes(self):
        """Test string encoding with quotes requires escaping."""
        result = self.encoder.encode_primitive('say "hello"', ',')
        assert result == '"say \\"hello\\""'

    def test_encode_empty_string(self):
        """Test empty string requires quotes."""
        result = self.encoder.encode_primitive('', ',')
        assert result == '""'

    def test_encode_string_with_leading_trailing_spaces(self):
        """Test string with leading/trailing spaces requires quotes."""
        result = self.encoder.encode_primitive('  hello  ', ',')
        assert result == '"  hello  "'

    def test_encode_boolean_like_strings(self):
        """Test strings that look like booleans require quotes."""
        assert self.encoder.encode_primitive('true', ',') == '"true"'
        assert self.encoder.encode_primitive('false', ',') == '"false"'
        assert self.encoder.encode_primitive('TRUE', ',') == '"TRUE"'
        assert self.encoder.encode_primitive('False', ',') == '"False"'

    def test_encode_null_like_strings(self):
        """Test strings that look like null require quotes."""
        assert self.encoder.encode_primitive('null', ',') == '"null"'
        assert self.encoder.encode_primitive('NULL', ',') == '"NULL"'

    def test_encode_number_like_strings(self):
        """Test strings that look like numbers require quotes."""
        assert self.encoder.encode_primitive('42', ',') == '"42"'
        assert self.encoder.encode_primitive('-17', ',') == '"-17"'
        assert self.encoder.encode_primitive('3.14', ',') == '"3.14"'
        assert self.encoder.encode_primitive('1e5', ',') == '"1e5"'

    def test_encode_list_like_strings(self):
        """Test strings that look like list items require quotes."""
        assert self.encoder.encode_primitive('- item', ',') == '"- item"'
        assert self.encoder.encode_primitive('- 123', ',') == '"- 123"'

    def test_encode_structural_token_strings(self):
        """Test strings that look like structural tokens require quotes."""
        assert self.encoder.encode_primitive('[5]', ',') == '"[5]"'
        assert self.encoder.encode_primitive('{key}', ',') == '"{key}"'
        assert self.encoder.encode_primitive('[3]: x,y', ',') == '"[3]: x,y"'

    def test_escape_special_characters(self):
        """Test escaping of special characters."""
        test_cases = [
            ('\\', '\\\\'),
            ('"', '\\"'),
            ('\n', '\\n'),
            ('\r', '\\r'),
            ('\t', '\\t'),
            ('\b', '\\b'),
            ('\f', '\\f'),
        ]

        for input_char, expected in test_cases:
            result = self.encoder.escape_string(input_char)
            assert result == expected

    def test_escape_complex_string(self):
        """Test escaping of complex string with multiple special characters."""
        input_str = 'Hello\nWorld\t"Test"'
        expected = 'Hello\\nWorld\\t\\"Test\\"'
        result = self.encoder.escape_string(input_str)
        assert result == expected

    def test_different_delimiters(self):
        """Test encoding with different delimiters."""
        # String with comma should be quoted when using comma delimiter
        result_comma = self.encoder.encode_primitive('a,b', ',')
        assert result_comma == '"a,b"'

        # Same string should not be quoted when using tab delimiter
        result_tab = self.encoder.encode_primitive('a,b', '\t')
        assert result_tab == 'a,b'

        # String with tab should be quoted when using tab delimiter
        result_tab2 = self.encoder.encode_primitive('a\tb', '\t')
        assert result_tab2 == '"a\\tb"'

    @given(st.text(min_size=1, max_size=10))
    def test_property_based_string_encoding(self, text):
        """Property-based test for string encoding."""
        result = self.encoder.encode_primitive(text, ',')

        # Result should be a string
        assert isinstance(result, str)

        # If quoted, should start and end with quotes
        if result.startswith('"'):
            assert result.endswith('"')
            # Length should be at least 2 for quotes
            assert len(result) >= 2

    @given(st.integers())
    def test_property_based_integer_encoding(self, value):
        """Property-based test for integer encoding."""
        result = self.encoder.encode_primitive(value, ',')
        assert result == str(value)

    @given(st.floats(allow_nan=False, allow_infinity=False))
    def test_property_based_float_encoding(self, value):
        """Property-based test for float encoding."""
        result = self.encoder.encode_primitive(value, ',')
        assert result == str(value)

    @given(st.booleans())
    def test_property_based_boolean_encoding(self, value):
        """Property-based test for boolean encoding."""
        result = self.encoder.encode_primitive(value, ',')
        expected = 'true' if value else 'false'
        assert result == expected
