"""TrainLoop Evaluations CLI 'init' command."""

import json
import os
import shutil
import subprocess
import sys
from pathlib import Path


def init_command():
    """Scaffold data/ and eval/ directories, create sample metrics and suites."""
    print("Initializing TrainLoop Evaluations...")

    # Get the path to the scaffold directory
    current_file = Path(__file__)
    cli_dir = current_file.parent.parent.parent
    scaffold_dir = cli_dir / "scaffold"

    # Get destination directory (current working directory)
    dest_dir = Path.cwd()
    trainloop_dir = dest_dir / "trainloop"

    # Check if trainloop directory already exists
    if trainloop_dir.exists():
        print(
            f"Error: {trainloop_dir} already exists. Please remove it or use a different directory."
        )
        sys.exit(1)

    # Copy the scaffold directory to the trainloop directory
    shutil.copytree(scaffold_dir / "trainloop", trainloop_dir)

    # Check for .env file and create or update if needed
    env_file = dest_dir / ".env"
    env_var_name = "TRAINLOOP_DATA_FOLDER"
    env_var_value = "trainloop/data"

    if not env_file.exists():
        # Try .env.local first
        env_file = dest_dir / ".env.local"

    if not env_file.exists():
        # Create new .env file
        with open(env_file, "w", encoding="utf-8") as f:
            f.write("# Remove the below to stop collecting data\n")
            f.write(f"{env_var_name}={env_var_value}\n")
        print("Created .env file with TRAINLOOP_DATA_FOLDER environment variable")
    else:
        # Check if variable exists in env file
        with open(env_file, "r", encoding="utf-8") as f:
            env_content = f.read()

        if f"{env_var_name}=" not in env_content:
            # Append variable if not found
            with open(env_file, "a", encoding="utf-8") as f:
                f.write(
                    f"\n# Remove the below to stop collecting data\n{env_var_name}={env_var_value}\n"
                )
            print(f"Added {env_var_name} to existing .env file")
        else:
            print(f"{env_var_name} already exists in .env file, not modifying")

    # Install appropriate SDK based on project type
    install_appropriate_sdk(dest_dir)

    # Print the directory tree structure dynamically
    print("\nCreated trainloop directory with the following structure:")
    print_directory_tree(trainloop_dir)

    print("\nInitialization complete!")
    print("\nFollow the instructions in trainloop/README.md to start collecting data.")


def install_appropriate_sdk(project_dir):
    """Detect project type and install appropriate SDK."""
    # Check for TypeScript/JavaScript project
    package_json = project_dir / "package.json"
    requirements_txt = project_dir / "requirements.txt"

    if package_json.exists():
        # It's a Node.js project
        try:
            with open(package_json, "r", encoding="utf-8") as f:
                package_data = json.load(f)

            # Check if the SDK is already installed
            dependencies = package_data.get("dependencies", {})
            dev_dependencies = package_data.get("devDependencies", {})

            if (
                "trainloop-evals-sdk" not in dependencies
                and "trainloop-evals-sdk" not in dev_dependencies
            ):
                # Try to install the SDK using npm
                print(
                    "\nDetected Node.js project, installing trainloop-evals-sdk package..."
                )
                try:
                    result = subprocess.run(
                        ["npm", "install", "trainloop-evals-sdk"],
                        cwd=project_dir,
                        capture_output=True,
                        text=True,
                        check=True,
                    )
                    print("Successfully installed trainloop-evals-sdk package")
                except subprocess.CalledProcessError as e:
                    print(f"Failed to install trainloop-evals-sdk package: {e.stderr}")
                    print(
                        "Please manually install with: npm install trainloop-evals-sdk"
                    )
            else:
                print(
                    "trainloop-evals-sdk package is already installed in package.json"
                )

        except (json.JSONDecodeError, IOError) as e:
            print(f"Error reading package.json: {e}")

    elif requirements_txt.exists():
        # It's a Python project
        try:
            with open(requirements_txt, "r", encoding="utf-8") as f:
                requirements = f.read()

            if "trainloop-evals-sdk" not in requirements:
                # Add the SDK to requirements.txt
                print(
                    "\nDetected Python project, adding trainloop-evals-sdk to requirements.txt..."
                )
                with open(requirements_txt, "a", encoding="utf-8") as f:
                    f.write("\n# TrainLoop evaluation SDK\ntrainloop-evals-sdk\n")
                print("Added trainloop-evals-sdk to requirements.txt")
                print("Please run: pip install -r requirements.txt")
            else:
                print("trainloop-evals-sdk is already in requirements.txt")

        except IOError as e:
            print(f"Error reading/writing requirements.txt: {e}")

    else:
        # Create a new requirements.txt file if no project type detected
        print(
            "\nNo package.json or requirements.txt found, creating requirements.txt..."
        )
        with open(requirements_txt, "w", encoding="utf-8") as f:
            f.write("# TrainLoop evaluation SDK\ntrainloop-evals-sdk\n")
        print("Created requirements.txt with trainloop-evals-sdk dependency")
        print("Please run: pip install -r requirements.txt")


def print_directory_tree(directory, prefix="", is_last=True, is_root=True):
    """Print a directory tree structure.

    Args:
        directory: Path object of the directory to print
        prefix: Prefix to use for current line (used for recursion)
        is_last: Whether this is the last item in its parent directory
        is_root: Whether this is the root directory
    """
    # Define directory and file display names with special descriptions
    descriptions = {
        "data": "# git-ignored",
        "events": "# append-only *.jsonl shards of raw calls",
        "results": "# verdicts; one line per test per event",
        "_registry.json": "",
        "helpers.py": "# tiny DSL (tag, etc.)",
        "types.py": "# Sample / Result dataclasses",
        "runner.py": "# CLI engine",
        "metrics": "# user-defined primitives",
        "suites": "# user-defined test collections",
    }

    # Get path name for display
    path_name = directory.name
    if is_root:
        print(f"  {path_name}/")
        new_prefix = "  "
    else:
        connector = "└── " if is_last else "├── "
        description = (
            f" {descriptions.get(path_name, '')}" if path_name in descriptions else ""
        )
        print(f"{prefix}{connector}{path_name}/{description}")
        new_prefix = prefix + ("    " if is_last else "│   ")

    # Get all items in the directory and sort them
    items = list(directory.iterdir())
    dirs = sorted([item for item in items if item.is_dir()])
    files = sorted([item for item in items if item.is_file()])

    # Process directories first
    for i, dir_path in enumerate(dirs):
        print_directory_tree(
            dir_path, new_prefix, i == len(dirs) - 1 and len(files) == 0, False
        )

    # Then process files
    for i, file_path in enumerate(files):
        connector = "└── " if i == len(files) - 1 else "├── "
        description = (
            f" {descriptions.get(file_path.name, '')}"
            if file_path.name in descriptions
            else ""
        )
        print(f"{new_prefix}{connector}{file_path.name}{description}")
