r"""\
Quiver
======
The motivation behind this module is to properly plot vectors in 3d space using matplotlib quivers.

The way quiver method works in 3d is to ask a 6-tuple of list which indicate the x, y and z coordinates
of the origins of the arrows to be drawn (tail by default) and u, v and w component of the arrows themselves.

3 of these methods take vectors and matrices as inputs, one function for drawing the same vector, after
applying transformations to it; one for drawing many vectors in the same coordinate system; and one for
drawing each input vector with a corresponding coordinate system.

The last method, `draw_axes_from_matrices`, takes a list of matrices and returns the information to draw
the x, y and z axis for each input matrix. As it returns 18 values instead of 6, it has slight customization
on how the values are returned.
"""

import transformations as tr
import numpy as np

from .vectors import extend_vector, reduce_vector, extend_matrix, reduce_matrix

def one_vector_many_systems(vector, matrices: list):
    """
    This method takes a single input vector and returns the position and direction it would have if it was in the coordinate systems
    generated by the input transformation matrices. When a single matrix is input, it is **required** to be in a list.
    """
    V = vector
    if len(V) == 4: V = reduce_vector(V)
    x,y,z,u,v,w=[],[],[],[],[],[]
    
    for mat in matrices:
        tmpMat = extend_matrix(mat) if np.shape(mat) == (3,3) else mat
        pos = tr.translation_from_matrix(tmpMat)
        x.append(float(pos[0]))
        y.append(float(pos[1])) 
        z.append(float(pos[2]))
        direct = np.dot(reduce_matrix(tmpMat),V).tolist()
        u.append(float(direct[0][0]))
        v.append(float(direct[0][1]))
        w.append(float(direct[0][2]))
    return x,y,z,u,v,w

def many_vectors_one_system(vectors: list, matrix: list):
    """
    This method takes a each input vector and returns the position and direction they would have if it was in the coordinate system
    generated by the input transformation matrix. When a single vector is input, it is **required** to be in a list.
    """
    M = extend_matrix(matrix) if np.shape(matrix) == (3,3) else matrix
    x,y,z,u,v,w=[],[],[],[],[],[]

    pos = tr.translation_from_matrix(M)
    for vec in vectors:
        tmpVec = reduce_vector(vec) if len(vec) == 4 else vec
        direct = np.dot(reduce_matrix(M),tmpVec).tolist()
        x.append(float(pos[0]))
        y.append(float(pos[1])) 
        z.append(float(pos[2]))
        u.append(float(direct[0][0]))
        v.append(float(direct[0][1]))
        w.append(float(direct[0][2]))
    return x,y,z,u,v,w

def many_vectors_many_systems(vectors: list, matrices: list):
    """
    This method takes a each input vector and returns the position and direction they would have if it was in the coordinate system
    generated by the corresponding input matrix. Input vector list and input matrix list must be the same size. When a single vector and thus
    a single matrix are input, they are both **required** to be on a list each.
    """
    if len(vectors) != len(matrices): raise ValueError("List of vectors and list of matrices have to be of same size")
    x=[]
    y=[]
    z=[]
    u=[]
    v=[]
    w=[]
    for i in range(len(vectors)):
        V = vectors[i].tolist() if type(vectors[i]) != list else vectors[i]
        if len(V) == 4: V = reduce_vector(V)
        M = extend_matrix(matrices[i]) if np.shape(matrices[i]) == (3,3) else matrices[i]
        pos = tr.translation_from_matrix(matrices[i])
        direct = np.dot(reduce_matrix(M),V).tolist()
        x.append(float(pos[0]))
        y.append(float(pos[1])) 
        z.append(float(pos[2]))
        u.append(float(direct[0][0]))
        v.append(float(direct[0][1]))
        w.append(float(direct[0][2]))
    return x,y,z,u,v,w

def draw_axes_from_matrices(matrices: list, orderByAxes = False, separateXYZ = False):
    """
    This method returns a 6-tuple of lists, the first 3 indicate the origin of the coordinate systems, the last 3 indicate
    the direction of the axes. Basis vectors of the coordinate systems are generated by the input transformation matrices.\n
    By default all x-axes come first, then all y-axes and finally all z-axes. By making `orderByAxes = False`,
    the order will be by coordinate system input, i.e., x, y and z axis from the first system; then x, y and z axis
    from the second system, etc.\n
    When separated a 12-tuple is returned instead, representing: origin of system, direction of X-axis,
    direction of Y-axis and direction of Z-axis, respectively.
    """
    x,y,z,uX,vX,wX = one_vector_many_systems([1,0,0],matrices)
    _,_,_,uY,vY,wY = one_vector_many_systems([0,1,0],matrices)
    _,_,_,uZ,vZ,wZ = one_vector_many_systems([0,0,1],matrices)
    if separateXYZ:
        return x, y, z, uX, vX, wX, uY, vY, wY, uZ, vZ, wZ
    else:
        if orderByAxes:
            X = x*3
            Y = y*3
            Z = z*3
            U = [uX,uY,uZ]
            V = [vX,vY,vZ]
            W = [wX,wY,wZ]
            U = list(np.ravel(U))
            V = list(np.ravel(V))
            W = list(np.ravel(W))
        else:
            X = [item for tri in zip (x,x,x) for item in tri]
            Y = [item for tri in zip (y,y,y) for item in tri]
            Z = [item for tri in zip (z,z,z) for item in tri]
            U = [item for tri in zip (uX,uY,uZ) for item in tri]
            V = [item for tri in zip (vX,vY,vZ) for item in tri]
            W = [item for tri in zip (wX,wY,wZ) for item in tri]
            U = list(np.ravel(U))
            V = list(np.ravel(V))
            W = list(np.ravel(W))
        return X,Y,Z,U,V,W
    
__all__ = [
    "one_vector_many_systems",
    "many_vectors_one_system",
    "many_vectors_many_systems",
    "draw_axes_from_matrices",
]