#!/usr/bin/env python3
"""
Trello CLI - Main command-line interface
"""

import sys
import warnings

# Suppress urllib3 NotOpenSSLWarning - this is a system-level SSL library issue
# that doesn't affect Trello CLI functionality with HTTPS requests
warnings.filterwarnings('ignore', message='.*urllib3 v2 only supports OpenSSL.*')

from . import __version__
from .config import configure_interactive
from .plugins import cmd_plugin_list, cmd_plugin_info, cmd_plugin_run
from .commands import (
    # Basic commands
    cmd_boards, cmd_create_board,
    cmd_lists, cmd_create_list, cmd_archive_list,
    cmd_cards, cmd_add_card, cmd_show_card,
    cmd_update_card, cmd_move_card, cmd_rename_card,
    cmd_add_checklist, cmd_add_checkitem,
    cmd_set_due, cmd_add_comment, cmd_delete_card,
    cmd_add_label, cmd_remove_label, cmd_delete_label, cmd_rename_label,
    # Help & Discovery
    cmd_help, cmd_help_json,
    cmd_board_overview, cmd_board_ids, cmd_search_cards,
    # Bulk operations
    cmd_bulk_move_cards, cmd_bulk_add_label, cmd_bulk_set_due,
    cmd_bulk_archive_cards, cmd_bulk_create_cards,
    cmd_bulk_relabel, cmd_label_backup, cmd_label_restore,
    # Quick commands
    cmd_quick_start, cmd_quick_test, cmd_quick_done,
    cmd_my_cards, cmd_card_age,
    # Sprint planning
    cmd_sprint_start, cmd_sprint_status, cmd_sprint_close, cmd_sprint_velocity,
    # Advanced queries
    cmd_cards_by_label, cmd_cards_due_soon, cmd_cards_overdue,
    cmd_list_metrics, cmd_board_health,
    # Board standardization
    cmd_standardize_lists, cmd_scrum_check, cmd_migrate_cards, cmd_list_templates,
    # Board migration
    cmd_migrate_board, cmd_archive_board,
    # Audit commands
    cmd_board_audit, cmd_list_audit, cmd_list_snapshot, cmd_sprint_audit, cmd_label_audit,
    # Member management
    cmd_assign_card, cmd_unassign_card, cmd_card_log,
    # Export
    cmd_export_board
)

HELP_TEXT = """
Trello CLI v{version} - Official Python command-line interface for Trello

Usage: trello <command> [arguments]

HELP & CONFIGURATION:
  config                      Configure API credentials
  help                        Show this help message
  help-json                   Get all commands in JSON format (for Claude Code)

DISCOVERY COMMANDS (for exploration):
  board-overview <board_id>   Complete board structure with card counts
  board-ids <board_id>        Quick reference of all IDs in a board
  search-cards <board_id> "query"   Search cards across all lists

QUICK COMMANDS (shortcuts):
  quick-start <card_id>       Move to "In Progress" + add comment
  quick-test <card_id>        Move to "Testing" + add comment
  quick-done <card_id>        Move to "Done" + add comment
  my-cards <board_id>         Show all your assigned cards
  card-age <list_id>          Show how long cards have been in list

SPRINT PLANNING:
  sprint-start <board_id>     Start sprint (move Ready → Sprint)
  sprint-status <board_id>    Show current sprint status
  sprint-close <board_id>     Close sprint and move unfinished cards
  sprint-velocity <board_id>  Calculate sprint velocity

BULK OPERATIONS:
  bulk-move-cards <source_list> <target_list> ["filter"]
  bulk-add-label <file> <color> ["name"]
  bulk-set-due <file> <date>
  bulk-archive-cards <list_id> ["filter"]
  bulk-create-cards <list_id> <csv/json_file>
  bulk-relabel <board_id> <from_label> <to_label> [--dry-run]

LABEL BACKUP & RECOVERY:
  label-backup <board_id> [output_file]         Backup all label assignments
  label-restore <board_id> <backup_file>        Restore labels from backup

ADVANCED QUERIES:
  cards-by-label <board_id> <color> ["name"]
  cards-due-soon <board_id> [days]
  cards-overdue <board_id>
  list-metrics <list_id>
  board-health <board_id>

BOARD STANDARDIZATION (Agile/Scrum):
  list-templates              Show available board templates
  standardize-lists <board_id> <template> [--dry-run]
  scrum-check <board_id>      Validate Agile/Scrum conformity
  migrate-cards <list_id> <target_board_id> ["target_list"]

MEMBER MANAGEMENT:
  assign-card <card_id> <member>        Assign member to card (use 'me' for self)
  unassign-card <card_id> <member>      Remove member from card
  card-log <card_id> [limit]            Show card action history

AUDIT & ANALYSIS (Expose Structural Chaos):
  board-audit <board_id> ["pattern"] [--report-json] [--fix-labels]
                                        🔍 Comprehensive workflow audit:
                                        - Cards in Done without due dates
                                        - Cards in Done with incomplete checklists
                                        - Active cards without due dates
                                        - Overdue zombie tasks
                                        - Execution cards without owners
                                        - Empty checklists (fake productivity)
                                        - Pattern violations & missing descriptions
                                        Flags: --report-json (JSON output)
                                               --fix-labels (auto-fix duplicates)
  list-audit <list_id> ["pattern"]      Detailed list audit
  list-snapshot <list_id> ["file.json"] Export list to JSON snapshot
  sprint-audit <board_id> ["sprint"]    Sprint-specific audit (dates, overdue)
  label-audit <board_id>                Label audit (duplicates, unused, typos)

EXPORT & REPORTING:
  export-board <board_id> <format> ["file"]  Export board (json/csv/md)

PLUGINS (EXTENSIBILITY):
  plugin list [--plugin-dir DIR]             List available plugins
  plugin info <name> [--plugin-dir DIR]      Show plugin details
  plugin run <name> [args] [--plugin-dir DIR] Execute a plugin

BASIC BOARD/LIST/CARD COMMANDS:
  boards                      List all boards
  lists <board_id>            List all lists
  cards <list_id>             List cards in list
  add-card <list_id> "title" ["desc"]
  show-card <card_id>
  update-card <card_id> "desc"
  rename-card <card_id> "title"         Rename card (update title)
  move-card <card_id> <list_id>
  delete-card <card_id>                 Delete a card permanently
  add-label <card_id> "color" ["name"]
  remove-label <card_id> "label"        Remove label from card
  delete-label <board_id> "label"       Delete label from board
  rename-label <board_id> "label" "new" Rename label on board
  add-checklist <card_id> "name"
  set-due <card_id> "YYYY-MM-DD"
  add-comment <card_id> "text"

For detailed help, run: trello help-json
For more information, see: README.md and CLAUDE.md
""".format(version=__version__)


def main():
    """Main CLI entry point"""
    if len(sys.argv) < 2:
        print(HELP_TEXT)
        sys.exit(1)

    command = sys.argv[1]

    try:
        if command == 'config':
            configure_interactive()

        elif command in ['-h', '--help', 'help']:
            cmd_help()

        elif command == 'help-json':
            cmd_help_json()

        elif command == 'board-overview':
            if len(sys.argv) < 3:
                print("❌ Usage: trello board-overview <board_id>")
                sys.exit(1)
            cmd_board_overview(sys.argv[2])

        elif command == 'board-ids':
            if len(sys.argv) < 3:
                print("❌ Usage: trello board-ids <board_id>")
                sys.exit(1)
            cmd_board_ids(sys.argv[2])

        elif command == 'search-cards':
            if len(sys.argv) < 4:
                print("❌ Usage: trello search-cards <board_id> \"query\"")
                sys.exit(1)
            cmd_search_cards(sys.argv[2], sys.argv[3])

        elif command == 'boards':
            cmd_boards()

        elif command == 'create-board':
            if len(sys.argv) < 3:
                print("❌ Usage: trello create-board \"name\"")
                sys.exit(1)
            cmd_create_board(sys.argv[2])

        elif command == 'lists':
            if len(sys.argv) < 3:
                print("❌ Usage: trello lists <board_id>")
                sys.exit(1)
            cmd_lists(sys.argv[2])

        elif command == 'create-list':
            if len(sys.argv) < 4:
                print("❌ Usage: trello create-list <board_id> \"name\"")
                sys.exit(1)
            cmd_create_list(sys.argv[2], sys.argv[3])

        elif command == 'archive-list':
            if len(sys.argv) < 3:
                print("❌ Usage: trello archive-list <list_id>")
                sys.exit(1)
            cmd_archive_list(sys.argv[2])

        elif command == 'cards':
            if len(sys.argv) < 3:
                print("❌ Usage: trello cards <list_id>")
                sys.exit(1)
            cmd_cards(sys.argv[2])

        elif command == 'add-card':
            if len(sys.argv) < 4:
                print("❌ Usage: trello add-card <list_id> \"title\" [--description \"description\"]")
                sys.exit(1)

            # Parse description - support both positional and --description flag
            description = ""
            if len(sys.argv) > 4:
                if sys.argv[4] == '--description' and len(sys.argv) > 5:
                    description = sys.argv[5]
                else:
                    # Legacy support: positional description argument
                    description = sys.argv[4]

            cmd_add_card(sys.argv[2], sys.argv[3], description)

        elif command == 'show-card':
            if len(sys.argv) < 3:
                print("❌ Usage: trello show-card <card_id>")
                sys.exit(1)
            cmd_show_card(sys.argv[2])

        elif command == 'update-card':
            if len(sys.argv) < 4:
                print("❌ Usage: trello update-card <card_id> \"description\"")
                sys.exit(1)
            cmd_update_card(sys.argv[2], sys.argv[3])

        elif command == 'rename-card':
            if len(sys.argv) < 4:
                print("❌ Usage: trello rename-card <card_id> \"new_title\"")
                sys.exit(1)
            cmd_rename_card(sys.argv[2], sys.argv[3])

        elif command == 'move-card':
            if len(sys.argv) < 4:
                print("❌ Usage: trello move-card <card_id> <list_id>")
                sys.exit(1)
            cmd_move_card(sys.argv[2], sys.argv[3])

        elif command == 'add-label':
            if len(sys.argv) < 4:
                print("❌ Usage: trello add-label <card_id> \"color\" [\"name\"]")
                sys.exit(1)
            label_name = sys.argv[4] if len(sys.argv) > 4 else ""
            cmd_add_label(sys.argv[2], sys.argv[3], label_name)

        elif command == 'remove-label':
            if len(sys.argv) < 4:
                print("❌ Usage: trello remove-label <card_id> \"label_name|color|id\"")
                sys.exit(1)
            cmd_remove_label(sys.argv[2], sys.argv[3])

        elif command == 'delete-label':
            if len(sys.argv) < 4:
                print("❌ Usage: trello delete-label <board_id> \"label_name|color|id\"")
                sys.exit(1)
            cmd_delete_label(sys.argv[2], sys.argv[3])

        elif command == 'rename-label':
            if len(sys.argv) < 5:
                print("❌ Usage: trello rename-label <board_id> \"current_label\" \"new_name\"")
                sys.exit(1)
            cmd_rename_label(sys.argv[2], sys.argv[3], sys.argv[4])

        elif command == 'add-checklist':
            if len(sys.argv) < 4:
                print("❌ Usage: trello add-checklist <card_id> \"name\"")
                sys.exit(1)
            cmd_add_checklist(sys.argv[2], sys.argv[3])

        elif command == 'add-checkitem':
            if len(sys.argv) < 5:
                print("❌ Usage: trello add-checkitem <card_id> \"checklist\" \"item\"")
                sys.exit(1)
            cmd_add_checkitem(sys.argv[2], sys.argv[3], sys.argv[4])

        elif command == 'set-due':
            if len(sys.argv) < 4:
                print("❌ Usage: trello set-due <card_id> \"YYYY-MM-DD\"")
                sys.exit(1)
            cmd_set_due(sys.argv[2], sys.argv[3])

        elif command == 'add-comment':
            if len(sys.argv) < 4:
                print("❌ Usage: trello add-comment <card_id> \"comment\"")
                sys.exit(1)
            cmd_add_comment(sys.argv[2], sys.argv[3])

        elif command == 'delete-card':
            if len(sys.argv) < 3:
                print("❌ Usage: trello delete-card <card_id>")
                sys.exit(1)
            cmd_delete_card(sys.argv[2])

        # Quick Commands
        elif command == 'quick-start':
            if len(sys.argv) < 3:
                print("❌ Usage: trello quick-start <card_id> [\"comment\"]")
                sys.exit(1)
            comment = sys.argv[3] if len(sys.argv) > 3 else "Started working on this"
            cmd_quick_start(sys.argv[2], comment)

        elif command == 'quick-test':
            if len(sys.argv) < 3:
                print("❌ Usage: trello quick-test <card_id> [\"comment\"]")
                sys.exit(1)
            comment = sys.argv[3] if len(sys.argv) > 3 else "Ready for testing"
            cmd_quick_test(sys.argv[2], comment)

        elif command == 'quick-done':
            if len(sys.argv) < 3:
                print("❌ Usage: trello quick-done <card_id> [\"comment\"]")
                sys.exit(1)
            comment = sys.argv[3] if len(sys.argv) > 3 else "Completed and verified"
            cmd_quick_done(sys.argv[2], comment)

        elif command == 'my-cards':
            if len(sys.argv) < 3:
                print("❌ Usage: trello my-cards <board_id> [\"member_name\"]")
                sys.exit(1)
            member_name = sys.argv[3] if len(sys.argv) > 3 else ""
            cmd_my_cards(sys.argv[2], member_name)

        elif command == 'card-age':
            if len(sys.argv) < 3:
                print("❌ Usage: trello card-age <list_id>")
                sys.exit(1)
            cmd_card_age(sys.argv[2])

        # Sprint Planning Commands
        elif command == 'sprint-start':
            if len(sys.argv) < 3:
                print("❌ Usage: trello sprint-start <board_id>")
                sys.exit(1)
            cmd_sprint_start(sys.argv[2])

        elif command == 'sprint-status':
            if len(sys.argv) < 3:
                print("❌ Usage: trello sprint-status <board_id>")
                sys.exit(1)
            cmd_sprint_status(sys.argv[2])

        elif command == 'sprint-close':
            if len(sys.argv) < 3:
                print("❌ Usage: trello sprint-close <board_id>")
                sys.exit(1)
            cmd_sprint_close(sys.argv[2])

        elif command == 'sprint-velocity':
            if len(sys.argv) < 3:
                print("❌ Usage: trello sprint-velocity <board_id> [num_sprints]")
                sys.exit(1)
            num_sprints = int(sys.argv[3]) if len(sys.argv) > 3 else 3
            cmd_sprint_velocity(sys.argv[2], num_sprints)

        # Bulk Operations
        elif command == 'bulk-move-cards':
            if len(sys.argv) < 4:
                print("❌ Usage: trello bulk-move-cards <source_list_id> <target_list_id> [\"filter\"]")
                sys.exit(1)
            filter_query = sys.argv[4] if len(sys.argv) > 4 else ""
            cmd_bulk_move_cards(sys.argv[2], sys.argv[3], filter_query)

        elif command == 'bulk-add-label':
            if len(sys.argv) < 4:
                print("❌ Usage: trello bulk-add-label <card_ids_file> <color> [\"name\"]")
                sys.exit(1)
            label_name = sys.argv[4] if len(sys.argv) > 4 else ""
            cmd_bulk_add_label(sys.argv[2], sys.argv[3], label_name)

        elif command == 'bulk-set-due':
            if len(sys.argv) < 4:
                print("❌ Usage: trello bulk-set-due <card_ids_file> <date>")
                sys.exit(1)
            cmd_bulk_set_due(sys.argv[2], sys.argv[3])

        elif command == 'bulk-archive-cards':
            if len(sys.argv) < 3:
                print("❌ Usage: trello bulk-archive-cards <list_id> [\"filter\"]")
                sys.exit(1)
            filter_query = sys.argv[3] if len(sys.argv) > 3 else ""
            cmd_bulk_archive_cards(sys.argv[2], filter_query)

        elif command == 'bulk-create-cards':
            if len(sys.argv) < 4:
                print("❌ Usage: trello bulk-create-cards <list_id> <csv/json_file>")
                sys.exit(1)
            cmd_bulk_create_cards(sys.argv[2], sys.argv[3])

        elif command == 'bulk-relabel':
            if len(sys.argv) < 5:
                print("❌ Usage: trello bulk-relabel <board_id> <from_label> <to_label> [--dry-run]")
                sys.exit(1)
            dry_run = '--dry-run' in sys.argv
            cmd_bulk_relabel(sys.argv[2], sys.argv[3], sys.argv[4], dry_run)

        elif command == 'label-backup':
            if len(sys.argv) < 3:
                print("❌ Usage: trello label-backup <board_id> [output_file]")
                sys.exit(1)
            output_file = sys.argv[3] if len(sys.argv) > 3 else "label_backup.json"
            cmd_label_backup(sys.argv[2], output_file)

        elif command == 'label-restore':
            if len(sys.argv) < 4:
                print("❌ Usage: trello label-restore <board_id> <backup_file>")
                sys.exit(1)
            cmd_label_restore(sys.argv[2], sys.argv[3])

        # Advanced Query Commands
        elif command == 'cards-by-label':
            if len(sys.argv) < 4:
                print("❌ Usage: trello cards-by-label <board_id> <color> [\"name\"]")
                sys.exit(1)
            label_name = sys.argv[4] if len(sys.argv) > 4 else ""
            cmd_cards_by_label(sys.argv[2], sys.argv[3], label_name)

        elif command == 'cards-due-soon':
            if len(sys.argv) < 3:
                print("❌ Usage: trello cards-due-soon <board_id> [days]")
                sys.exit(1)
            days = int(sys.argv[3]) if len(sys.argv) > 3 else 7
            cmd_cards_due_soon(sys.argv[2], days)

        elif command == 'cards-overdue':
            if len(sys.argv) < 3:
                print("❌ Usage: trello cards-overdue <board_id>")
                sys.exit(1)
            cmd_cards_overdue(sys.argv[2])

        elif command == 'list-metrics':
            if len(sys.argv) < 3:
                print("❌ Usage: trello list-metrics <list_id>")
                sys.exit(1)
            cmd_list_metrics(sys.argv[2])

        elif command == 'board-health':
            if len(sys.argv) < 3:
                print("❌ Usage: trello board-health <board_id>")
                sys.exit(1)
            cmd_board_health(sys.argv[2])

        # Board Standardization Commands
        elif command == 'list-templates':
            cmd_list_templates()

        elif command == 'standardize-lists':
            if len(sys.argv) < 3:
                print("❌ Usage: trello standardize-lists <board_id> [template] [--dry-run]")
                sys.exit(1)
            template = sys.argv[3] if len(sys.argv) > 3 else "agile"
            dry_run = "--dry-run" in sys.argv or "-n" in sys.argv
            cmd_standardize_lists(sys.argv[2], template, dry_run)

        elif command == 'scrum-check':
            if len(sys.argv) < 3:
                print("❌ Usage: trello scrum-check <board_id>")
                sys.exit(1)
            cmd_scrum_check(sys.argv[2])

        elif command == 'migrate-cards':
            if len(sys.argv) < 4:
                print("❌ Usage: trello migrate-cards <source_list_id> <target_board_id> [\"target_list\"]")
                sys.exit(1)
            target_list = sys.argv[4] if len(sys.argv) > 4 else ""
            cmd_migrate_cards(sys.argv[2], sys.argv[3], target_list)

        elif command == 'migrate-board':
            if len(sys.argv) < 4:
                print("❌ Usage: trello migrate-board <source_board_id> <target_board_id> [--dry-run]")
                sys.exit(1)
            dry_run = '--dry-run' in sys.argv
            cmd_migrate_board(sys.argv[2], sys.argv[3], dry_run)

        elif command == 'archive-board':
            if len(sys.argv) < 3:
                print("❌ Usage: trello archive-board <board_id>")
                sys.exit(1)
            cmd_archive_board(sys.argv[2])

        # Audit Commands
        elif command == 'board-audit':
            if len(sys.argv) < 3:
                print("❌ Usage: trello board-audit <board_id> [\"pattern\"] [--report-json] [--fix-labels]")
                print("\nFlags:")
                print("  --report-json    Output audit results in JSON format")
                print("  --fix-labels     Automatically fix duplicate labels")
                sys.exit(1)

            # Parse arguments and flags
            board_id = sys.argv[2]
            pattern = None
            report_json = False
            fix_labels = False

            for i in range(3, len(sys.argv)):
                arg = sys.argv[i]
                if arg == '--report-json':
                    report_json = True
                elif arg == '--fix-labels':
                    fix_labels = True
                elif not pattern and not arg.startswith('--'):
                    pattern = arg

            cmd_board_audit(board_id, pattern, fix_labels, report_json)

        elif command == 'list-audit':
            if len(sys.argv) < 3:
                print("❌ Usage: trello list-audit <list_id> [\"pattern\"]")
                sys.exit(1)
            pattern = sys.argv[3] if len(sys.argv) > 3 else None
            cmd_list_audit(sys.argv[2], pattern)

        elif command == 'list-snapshot':
            if len(sys.argv) < 3:
                print("❌ Usage: trello list-snapshot <list_id> [\"output_file.json\"]")
                sys.exit(1)
            output_file = sys.argv[3] if len(sys.argv) > 3 else None
            cmd_list_snapshot(sys.argv[2], output_file)

        elif command == 'sprint-audit':
            if len(sys.argv) < 3:
                print("❌ Usage: trello sprint-audit <board_id> [\"sprint_label\"]")
                sys.exit(1)
            sprint_label = sys.argv[3] if len(sys.argv) > 3 else None
            cmd_sprint_audit(sys.argv[2], sprint_label)

        elif command == 'label-audit':
            if len(sys.argv) < 3:
                print("❌ Usage: trello label-audit <board_id>")
                sys.exit(1)
            cmd_label_audit(sys.argv[2])

        # Member Management Commands
        elif command == 'assign-card':
            if len(sys.argv) < 4:
                print("❌ Usage: trello assign-card <card_id> <member_username|name|'me'>")
                sys.exit(1)
            cmd_assign_card(sys.argv[2], sys.argv[3])

        elif command == 'unassign-card':
            if len(sys.argv) < 4:
                print("❌ Usage: trello unassign-card <card_id> <member_username|name|'me'>")
                sys.exit(1)
            cmd_unassign_card(sys.argv[2], sys.argv[3])

        elif command == 'card-log':
            if len(sys.argv) < 3:
                print("❌ Usage: trello card-log <card_id> [limit]")
                sys.exit(1)
            limit = int(sys.argv[3]) if len(sys.argv) > 3 else 50
            cmd_card_log(sys.argv[2], limit)

        # Export Commands
        elif command == 'export-board':
            if len(sys.argv) < 4:
                print("❌ Usage: trello export-board <board_id> <json|csv|md> [\"output_file\"]")
                sys.exit(1)
            output_file = sys.argv[4] if len(sys.argv) > 4 else None
            cmd_export_board(sys.argv[2], sys.argv[3], output_file)

        # Plugin Commands
        elif command == 'plugin':
            if len(sys.argv) < 3:
                print("❌ Usage: trello plugin <list|info|run> [args]")
                print("\n  plugin list                    - List all plugins")
                print("  plugin info <name>             - Show plugin details")
                print("  plugin run <name> [args]       - Execute a plugin")
                print("\n  Optional: --plugin-dir <path>  - Use custom plugin directory")
                sys.exit(1)

            subcommand = sys.argv[2]

            # Extract --plugin-dir if present
            plugin_dir = None
            if '--plugin-dir' in sys.argv:
                idx = sys.argv.index('--plugin-dir')
                if idx + 1 < len(sys.argv):
                    plugin_dir = sys.argv[idx + 1]
                    # Remove --plugin-dir and its value from argv
                    sys.argv.pop(idx)
                    sys.argv.pop(idx)

            if subcommand == 'list':
                cmd_plugin_list(plugin_dir)

            elif subcommand == 'info':
                if len(sys.argv) < 4:
                    print("❌ Usage: trello plugin info <name>")
                    sys.exit(1)
                cmd_plugin_info(sys.argv[3], plugin_dir)

            elif subcommand == 'run':
                if len(sys.argv) < 4:
                    print("❌ Usage: trello plugin run <name> [args]")
                    sys.exit(1)
                plugin_name = sys.argv[3]
                plugin_args = sys.argv[4:]
                cmd_plugin_run(plugin_name, plugin_args, plugin_dir)

            else:
                print(f"❌ Unknown plugin subcommand: {subcommand}")
                print("   Valid subcommands: list, info, run")
                sys.exit(1)

        elif command in ['-v', '--version', 'version']:
            print(f"Trello CLI v{__version__}")

        else:
            print(f"❌ Unknown command: {command}")
            print()
            print(HELP_TEXT)
            sys.exit(1)

        # Show help reminder after successful command execution
        # (only for non-help, non-version commands)
        if command not in ['help', '-h', '--help', 'help-json', 'version', '-v', '--version']:
            print("\n💡 Run 'trello help' to see all capabilities")

    except KeyboardInterrupt:
        print("\n⚠️  Operation cancelled by user")
        sys.exit(130)
    except Exception as e:
        print(f"❌ Error: {str(e)}")
        sys.exit(1)


if __name__ == '__main__':
    main()
