from httpx import Response

from tsugu_api_core import settings
from tsugu_api_core._network import Api
from tsugu_api_core._typing import (
    _Update,
    _ServerId,
    _GetUserDataResponse,
    _ChangeUserDataResponse,
    _BindPlayerRequestResponse,
    _BindPlayerVerificationResponse
)

async def get_user_data(platform: str, user_id: str) -> _GetUserDataResponse:
    '''获取用户数据

    参数:
        platform (str): 平台名称
        user_id (str): 用户 ID

    返回:
        _GetUserDataResponse: API 返回响应
    '''
    
    # 构建 URL
    url = settings.userdata_backend_url + '/user/getUserData'
    
    # 构建数据
    data = {
        'platform': platform,
        'user_id': user_id
    }
    
    # 发送请求
    response = await Api(
        url,
        proxy=settings.userdata_backend_proxy
    ).apost(data)
    if isinstance(response, Response): return response.json()
    return await response.json()

async def change_user_data(platform: str, user_id: str, update: _Update) -> _ChangeUserDataResponse:
    '''修改用户数据

    参数:
        platform (str): 平台名称
        user_id (str): 用户 ID
        update (_Update): 更新数据

    返回:
        _ChangeUserDataResponse: API 返回响应
    '''
    
    # 构建 URL
    url = settings.userdata_backend_url + '/user/changeUserData'
    
    # 构建数据
    data = {
        'platform': platform,
        'user_id': user_id,
        'update': update
    }
    
    # 发送请求
    response = await Api(
        url,
        proxy=settings.userdata_backend_proxy
    ).apost(data)
    if isinstance(response, Response): return response.json()
    return await response.json()

async def bind_player_request(
    platform: str,
    user_id: str,
    server: _ServerId,
    bind_type: bool
) -> _BindPlayerRequestResponse:
    '''绑定玩家请求

    参数:
        platform (str): 平台名称
        user_id (str): 用户 ID
        server (_ServerId): 服务器编号 0 - 日服 1 - 国际服 2 - 台服 3 - 国服 4 - 韩服
        bind_type (bool): 绑定类型， `true` 为绑定， `false` 为解绑

    返回:
        _BindPlayerRequestResponse: 请求返回数据
    '''
    
    # 构建 URL
    url = settings.userdata_backend_url + '/user/bindPlayerRequest'
    
    # 构建数据
    data = {
        'platform': platform,
        'user_id': user_id,
        'server': server,
        'bindType': bind_type
    }
    
    # 发送请求
    response = await Api(
        url,
        proxy=settings.userdata_backend_proxy
    ).apost(data)
    if isinstance(response, Response): return response.json()
    return await response.json()

async def bind_player_verification(
    platform: str,
    user_id: str,
    server: _ServerId,
    player_id: int,
    bind_type: bool
) -> _BindPlayerVerificationResponse:
    '''绑定玩家验证

    参数:
        platform (str): 平台名称
        user_id (str): 用户 ID
        server (_ServerId): 服务器编号 0 - 日服 1 - 国际服 2 - 台服 3 - 国服 4 - 韩服
        player_id (int): 玩家 ID
        bind_type (bool): 绑定类型， `true` 为绑定， `false` 为解绑

    返回:
        _BindPlayerVerificationResponse: 验证返回数据
    '''
    
    # 构建 URL
    url = settings.userdata_backend_url + '/user/bindPlayerVerification'
    
    # 构建数据
    data = {
        'platform': platform,
        'user_id': user_id,
        'server': server,
        'playerId': player_id,
        'bindType': bind_type
    }
    
    # 发送请求
    response = await Api(
        url,
        proxy=settings.userdata_backend_proxy
    ).apost(data)
    if isinstance(response, Response): return response.json()
    return await response.json()
