"""
This module contains functionality for translating various tagsets to the
Typecraft tagset.
"""

import six

# noinspection SpellCheckingInspection
POS_CONVERSIONS = {
    "$": "PUN",
    "$(": "PUN",
    "$.": "PUN",
    "$,": "PUN",
    ":": "PUN",
    "adj": "ADJ",
    "adja": "ADJ",
    "adjd": "ADJ",
    "adv": "ADV",
    "advprt": "PRT",
    "advs": "ADV",
    "appo": "PPOS",
    "appo": "PPOST",
    "appr": "PREP",
    "apprart": "PREP",
    "apzr": "",
    "apzr": "",
    "art": "ART",
    "art": "DET",
    "c": "COMP",
    "card": "CARD",
    "cc": "CONJC",
    "cd": "CARD",
    "conj": "CONJ",
    "deg": "PRT",
    "dt": "DET",
    "ex": "",
    "expl": "EXPL",
    "fm": "",
    "fm": "",
    "fw": "",
    "in": "PREP",
    "in/that": "COMP",
    "inf": "PRTinf",
    "interjct": "INTRJCT",
    "itj": "INTRJCT",
    "jj": "ADJ",
    "jjr": "ADJC",
    "jjs": "ADJS",
    "kokom": "PRT",
    "kon": "CONJC",
    "koui": "CONJS",
    "koui": "CONJSINF",
    "kous": "CONJS",
    "ls": "",
    "md": "Vmod",
    "ne": "Np",
    "net": "",
    "nn": "N",
    "nn": "N",
    "nns": "N",
    "np": "N",
    "nprop": "Np",
    "nps": "N",
    "ord": "ORD",
    "p": "PREP",
    "pav": "ADV",
    "pav": "PN",
    "pdat": "PN",
    "pdat": "PNdem",
    "pds": "PN",
    "pds": "PNdem",
    "pdt": "DET",
    "piat": "PN",
    "pidat": "",
    "pis": "PN",
    "pninterr": "wh",
    "pninterr": "Wh",
    "pnrefl": "PNrefl",
    "pnresmptv": "PNrel",
    "pos": "PRTposs",
    "poss": "PRTposs",
    "PP$": "PNposs",
    "pp": "PN",
    "pper": "",
    "pper": "PN",
    "pposat": "PNposs",
    "pposat": "Pnposs",
    "pposs": "PNposs",
    "pposs": "Pnposs",
    "prelat": "",
    "prelat": "PNrel",
    "prels": "PNrel",
    "prels": "Pnrel",
    "prf": "PNrefl",
    "prf": "Pnrefl",
    "prtcmpr": "PRT",
    "ptka": "PRT",
    "ptkant": "PRT",
    "ptkant": "PRTresp",
    "ptkneg": "PRT",
    "ptkneg": "PRTneg",
    "ptkvz": "PRTv",
    "ptkzu": "PRTinf",
    "pun": "PUN",
    "punct": "PUN",
    "pwat": "PN",
    "pwat": "PROint",
    "pwav": "PN",
    "pwav": "PROint",
    "pws": "PROint",
    "pws": "Wh",
    "quantinterr": "QUANT",
    "rb": "ADV",
    "rbr": "ADV",
    "rbs": "ADV",
    "rp": "PRT",
    "sent": "PUN",
    "sgml": "",
    "spell": "",
    "sym": "PUN",
    "to": "ADV",
    "trunc": "",
    "trunc": "TRUNC",
    "uh": "INTRJCT",
    "v-ditr": "Vdtr",
    "v-ditrobl": "V",
    "v-extrapos": "V",
    "v-extraposobl": "V",
    "v-intr": "Vitr",
    "v-introbl": "VitrOBL",
    "v-intrscpr": "V",
    "v-obextrapos": "V",
    "v-presntn": "V",
    "v-presntnobl": "V",
    "v-tr": "Vtr",
    "v-trobl": "VtrOBL",
    "v-trscpr": "V",
    "vafin": "AUX",
    "vaimp": "AUX",
    "vainf": "AUX",
    "vapp": "AUX",
    "vapp": "PTCP",
    "vaux": "AUX",
    "vb": "V",
    "vbd": "V",
    "vbg": "V",
    "vbn": "V",
    "vbp": "V",
    "vbz": "V",
    "vcopa": "COP",
    "vcopn": "COP",
    "vd": "V",
    "vdd": "V",
    "vdg": "V",
    "vdn": "V",
    "vdp": "V",
    "vdz": "V",
    "vh": "V",
    "vhd": "V",
    "vhg": "V",
    "vhn": "V",
    "vhp": "V",
    "vhz": "V",
    "vmfin": "Vmod",
    "vminf": "Vmod",
    "vmpp": "PTCP",
    "vmpp": "Vmod",
    "vrefl": "V",
    "vrefl-ob": "V",
    "vrefl-obl": "V",
    "vrefl-presobl": "V",
    "vrefl-scpr": "V",
    "vrefl-trobl": "V",
    "vv": "V",
    "vvd": "V",
    "vvfin": "V",
    "vvg": "V",
    "vvimp": "V",
    "vvinf": "V",
    "vvizu": "V",
    "vvn": "V",
    "vvp": "V",
    "vvpp": "V",
    "vvz": "V",
    "wdt": "Wh",
    "wp": "Wh",
    "WP$": "Wh",
    "wrb": "Wh",
    "xy": "XY"
}


def get_pos_conversions(pos, tagset='tc'):
    """
    Gets a pos conversion.

    :param pos:
    :param tagset: Currently not used, as the only mapping supported is 'tc'
    :return:
    """
    assert isinstance(pos, six.string_types)
    potential_tag = POS_CONVERSIONS.get(pos.lower())
    if potential_tag:
        return potential_tag
    else:
        return pos


def get_gloss_conversions(gloss, tagset='tc'):
    """
    Gets a gloss conversion into the given tagset.
    Currently acts as the identity, as we don't have any conversions registered yet.

    :param gloss:
    :param tagset: Currently not used, as the only mapping supported is 'tc'
    :return:
    """
    return gloss
