# UpNote Python Client

[![PyPI version](https://badge.fury.io/py/upnote-python-client.svg)](https://badge.fury.io/py/upnote-python-client)
[![Python versions](https://img.shields.io/pypi/pyversions/upnote-python-client.svg)](https://pypi.org/project/upnote-python-client/)
[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)

A powerful Python client for UpNote that uses URL schemes (x-callback-url) to create and manage notes programmatically.

## 기능

- 노트 생성 (제목, 내용, 태그, 노트북 지정)
- 노트 열기
- 노트 검색
- 노트북 생성
- UpNote 앱 열기
- 마크다운 헬퍼 기능 (체크리스트, 테이블 등)

## Installation

Install from PyPI:

```bash
pip install upnote-python-client
```

Or install from source:

```bash
git clone https://github.com/upnote-python/upnote-python-client.git
cd upnote-python-client
pip install -e .

## Quick Start

### Basic Setup

```python
from upnote_python_client import UpNoteClient

# Initialize client (no API key required)
client = UpNoteClient()
```

### Creating Notes

```python
# Basic note creation
client.create_note(
    text="Note content",
    title="Note title"
)

# Advanced note with multiple parameters
client.create_note(
    text="Markdown **supported** content",
    title="Advanced Note",
    notebook="Projects",
    tags=["important", "work"],
    pinned=True,
    color="blue",
    priority="high"
)
```

### Using Helper Functions

```python
from upnote_python_client import UpNoteHelper

# Create checklist
checklist = UpNoteHelper.create_checklist([
    "Task 1",
    "Task 2", 
    "Task 3"
])

# Create table
table = UpNoteHelper.create_table(
    headers=["Name", "Age", "Role"],
    rows=[
        ["John Doe", "30", "Developer"],
        ["Jane Smith", "25", "Designer"]
    ]
)

# Add timestamp
content = UpNoteHelper.format_markdown_content(
    "Original content",
    add_timestamp=True
)

# Create complete note
client.create_note(
    text=f"# Task List\n\n{checklist}\n\n# Team Info\n\n{table}",
    title="Project Status"
)
```

## 주요 메서드

### UpNoteClient

**노트 관련:**
- `create_note()`: 새 노트 생성 (마크다운, 색상, 고정, 즐겨찾기, 알림 등 지원)
- `create_markdown_note()`: 마크다운 최적화 노트 생성
- `open_note()`: 특정 노트 열기 (편집 모드 지원)
- `quick_note()`: 빠른 노트 추가

**노트북 관련:**
- `create_notebook()`: 노트북 생성 (색상, 하위 노트북 지원)
- `open_notebook()`: 노트북 열기

**검색 및 기타:**
- `search_notes()`: 고급 검색 (노트북, 태그 필터링)
- `import_note()`: 파일에서 노트 가져오기
- `export_note()`: 노트 내보내기
- `open_upnote()`: UpNote 앱 열기

### UpNoteHelper

- `format_markdown_content()`: 마크다운 콘텐츠 포맷팅
- `create_checklist()`: 체크리스트 생성
- `create_table()`: 테이블 생성

## URL Scheme 예제

생성되는 URL 예시:
```
upnote://x-callback-url/note/new?text=Hello%20UpNote!&title=Test%20Note
upnote://x-callback-url/note/new?text=Content&title=Title&notebook=Project&tags=tag1,tag2
upnote://x-callback-url/search?query=keyword
```

## 지원 플랫폼

- macOS (open 명령어)
- Windows (start 명령어)
- Linux (xdg-open 명령어)

## 예제

자세한 사용 예제는 `examples/` 디렉토리의 파일들을 참고하세요:

- `examples/example_usage.py`: 기본 사용법
- `examples/advanced_example.py`: 고급 기능
- `examples/comprehensive_example.py`: 종합 예제

## 테스트

프로젝트의 모든 기능을 테스트하려면:

```bash
# 전체 테스트 실행
python tests/test_all_features.py

# 마크다운 기능 테스트
python tests/test_markdown.py

# 최종 검증
python tests/final_test_summary.py
```

## 프로젝트 구조

```
upnote_client/
├── upnote_client.py          # 메인 클라이언트 모듈
├── requirements.txt          # 의존성 (없음)
├── setup.py                 # 설치 스크립트
├── README.md                # 프로젝트 문서
├── LICENSE                  # MIT 라이선스
├── CHANGELOG.md             # 변경 로그
├── examples/                # 사용 예제들
│   ├── README.md
│   ├── example_usage.py
│   ├── advanced_example.py
│   └── comprehensive_example.py
├── tests/                   # 테스트 파일들
│   ├── README.md
│   ├── test_all_features.py
│   ├── test_markdown.py
│   └── final_test_summary.py
└── docs/                    # 문서
    └── API_REFERENCE.md
```

## 주의사항

- UpNote 앱이 설치되어 있어야 합니다
- URL scheme 방식이므로 앱이 열리면서 노트가 생성됩니다
- 콜백 URL을 설정하여 성공/실패 처리를 할 수 있습니다


### 고급 노트 생성
```python
# 색상, 고정, 즐겨찾기가 있는 중요한 노트
client.create_note(
    title="긴급 공지",
    text="# 중요한 내용\n\n긴급히 확인이 필요합니다.",
    markdown=True,
    pinned=True,
    favorite=True,
    color="red",
    reminder="2024-01-20T14:00:00",
    tags=["긴급", "공지"]
)

# 마크다운 최적화 노트
client.create_markdown_note(
    title="프로젝트 계획",
    content="# 계획\n\n- [ ] 작업 1\n- [ ] 작업 2",
    notebook="프로젝트",
    color="blue",
    add_timestamp=True
)
```

### 고급 검색
```python
# 특정 노트북에서 태그로 필터링하여 검색
client.search_notes(
    query="회의",
    notebook="업무",
    tags=["중요", "진행중"],
    limit=10
)
```

### 노트북 관리
```python
# 색상이 있는 노트북 생성
client.create_notebook("데이터 분석", color="purple")

# 하위 노트북 생성
client.create_notebook("월간 리포트", parent="데이터 분석")

# 노트북 열기
client.open_notebook("데이터 분석")
```

### 빠른 작업
```python
# 기존 노트에 내용 추가
client.quick_note("새로운 아이디어 추가", append=True)

# 편집 모드로 노트 열기
client.open_note(title="회의록", edit=True)
```

### 파일 작업
```python
# 마크다운 파일 가져오기
client.import_note("document.md", notebook="문서", format_type="markdown")

# 노트를 PDF로 내보내기
client.export_note(title="보고서", format_type="pdf", destination="~/Documents/")
```

## 확장된 파라미터 지원

### 노트 속성
- `pinned`: 노트 고정
- `favorite`: 즐겨찾기
- `starred`: 별표 표시
- `priority`: 우선순위 (high, medium, low, urgent)
- `category`: 카테고리 분류
- `readonly`: 읽기 전용 설정

### 시간 관리
- `due_date`: 마감일 설정
- `created_date`: 생성일 지정
- `modified_date`: 수정일 지정
- `reminder`: 알림 시간

### 보안 및 공유
- `encrypted`: 암호화 여부
- `password`: 노트 비밀번호
- `shared`: 공유 여부
- `public`: 공개 여부

### 메타데이터
- `author`: 작성자 정보
- `source`: 출처 정보
- `url`: 관련 URL 링크
- `location`: 위치 정보
- `attachments`: 첨부파일 목록

### 지원하는 색상
- `red`: 빨간색 (긴급, 중요)
- `blue`: 파란색 (정보, 계획)
- `green`: 녹색 (완료, 성공)
- `yellow`: 노란색 (주의, 대기)
- `purple`: 보라색 (창작, 아이디어)
- `gray`: 회색 (보관, 참고)
- `orange`: 주황색 (경고, 알림)
- `pink`: 분홍색 (개인, 취미)

## 알림 시간 형식
- ISO 8601: `"2024-01-20T14:00:00"`
- 자연어: `"tomorrow 2pm"`, `"next friday"`
- 상대시간: `"in 1 hour"`, `"in 30 minutes"`#
# 특수 노트 생성 메서드

### 할 일 노트
```python
client.create_task_note(
    title="주간 업무 계획",
    tasks=["작업 1", "작업 2", "작업 3"],
    due_date="2024-01-30",
    priority="high",
    reminder="2024-01-25T09:00:00"
)
```

### 회의록 노트
```python
client.create_meeting_note(
    title="팀 미팅",
    date="2024-01-25 14:00",
    attendees=["김팀장", "박개발", "이디자인"],
    agenda=["프로젝트 진행상황", "다음 스프린트 계획"],
    location="회의실 A"
)
```

### 프로젝트 노트
```python
client.create_project_note(
    project_name="웹사이트 리뉴얼",
    description="기존 웹사이트의 UI/UX 개선",
    milestones=["기획", "디자인", "개발", "테스트", "배포"],
    team_members=["기획자", "디자이너", "개발자"],
    due_date="2024-06-30"
)
```

### 일일 노트
```python
client.create_daily_note(
    mood="😊 좋음",
    weather="☀️ 맑음",
    goals=["운동하기", "독서하기", "프로젝트 진행"],
    reflections="오늘은 생산적인 하루였다."
)
```

## 고급 사용 예제

### 암호화된 기밀 노트
```python
client.create_note(
    title="🔒 서버 정보",
    text="중요한 서버 접속 정보...",
    encrypted=True,
    password="secure123!",
    color="red",
    shared=False,
    readonly=True
)
```

### 위치 정보가 있는 여행 노트
```python
client.create_note(
    title="✈️ 제주도 여행",
    text="여행 계획 및 일정...",
    location="제주특별자치도",
    attachments=["항공권.pdf", "호텔예약.pdf"],
    tags=["여행", "제주도"],
    color="green"
)
```

### 우선순위가 높은 업무 노트
```python
client.create_note(
    title="🚨 긴급 업무",
    text="즉시 처리해야 할 업무...",
    priority="urgent",
    pinned=True,
    favorite=True,
    color="red",
    reminder="2024-01-20T09:00:00",
    due_date="2024-01-20T18:00:00"
)
```