"""Parse the JSON file and generate the validation modules (parser.py, record.py, validator.py, validate_file.py)."""
import click
import logging
import os
import pathlib
import sys
import yaml

from pathlib import Path
from rich.console import Console

from .json_manager import Manager as JSONManager
from . import constants
from .file_utils import check_infile_status
from .console_helper import print_yellow, print_green, print_red


DEFAULT_OUTDIR = os.path.join(
    constants.DEFAULT_OUTDIR_BASE,
    os.path.splitext(os.path.basename(__file__))[0],
    constants.DEFAULT_TIMESTAMP,
)


error_console = Console(stderr=True, style="bold red")

console = Console()



def validate_verbose(ctx, param, value):
    """Validate the validate option.

    Args:
        ctx (Context): The click context.
        param (str): The parameter.
        value (bool): The value.

    Returns:
        bool: The value.
    """

    if value is None:
        click.secho("--verbose was not specified and therefore was set to 'True'", fg='yellow')
        return constants.DEFAULT_VERBOSE
    return value


@click.command()  # type: ignore
@click.option(
    "--config_file",
    type=click.Path(exists=True),
    help=f"Optional: The configuration file for this project - default is '{constants.DEFAULT_CONFIG_FILE}'",
)  # type: ignore
@click.option("--data_file_type", help="Required: The type of the data files to be processed")  # type: ignore
@click.option("--infile", help="Required: The primary input file")  # type: ignore
@click.option("--logfile", help="Optional: The log file")  # type: ignore
@click.option("--namespace", help="Required: The namespace will dictate relative placement of the modules")  # type: ignore
@click.option(
    "--outdir",
    help=f"Optional: The default is the current working directory - default is '{DEFAULT_OUTDIR}'",
)  # type: ignore
@click.option(
    "--template_path",
    help=f"Optional: The directory containing the Jinja2 template files - default is '{constants.DEFAULT_TEMPLATE_PATH}'",
)  # type: ignore
@click.option(
    '--verbose',
    is_flag=True,
    help=f"Will print more info to STDOUT - default is '{constants.DEFAULT_VERBOSE}'.",
    callback=validate_verbose
)
def main(
    config_file: str,
    data_file_type: str,
    infile: str,
    logfile: str,
    namespace: str,
    outdir: str,
    outfile: str,
    template_path: str,
    verbose: bool,
) -> None:
    """Parse the JSON file and generate the validation modules (parser.py, record.py, validator.py, validate_file.py)."""
    error_ctr = 0

    if infile is None:
        error_console.print("--infile was not specified")
        error_ctr += 1

    if data_file_type is None:
        error_console.print("--data_file_type was not specified")
        error_ctr += 1

    if namespace is None:
        error_console.print("--namespace was not specified")
        error_ctr += 1

    if error_ctr > 0:
        print_red("Required parameter(s) not defined")
        click.echo(click.get_current_context().get_help())
        sys.exit(1)

    check_infile_status(infile)

    if config_file is None:
        config_file = constants.DEFAULT_CONFIG_FILE
        print_yellow(f"--config_file was not specified and therefore was set to '{config_file}'")

    check_infile_status(config_file, extension="yaml")

    if outdir is None:
        outdir = DEFAULT_OUTDIR
        print_yellow(f"--outdir was not specified and therefore was set to '{outdir}'")

    if not os.path.exists(outdir):
        pathlib.Path(outdir).mkdir(parents=True, exist_ok=True)
        print_yellow(f"Created output directory '{outdir}'")

    if template_path is None:
        template_path = constants.DEFAULT_TEMPLATE_PATH
        print_yellow(f"--template_path was not specified and therefore was set to '{template_path}'")

    if not os.path.exists(template_path):
        print_red(f"template_path '{template_path}' does not exist")
        sys.exit(1)

    if not os.path.isdir(template_path):
        print_red(f"template_path '{template_path}' is not a regular directory")
        sys.exit(1)

    if verbose is None:
        verbose = constants.DEFAULT_VERBOSE
        print_yellow(f"--verbose was not specified and therefore was set to '{verbose}'")

    if logfile is None:
        logfile = os.path.join(
            outdir, os.path.splitext(os.path.basename(__file__))[0] + ".log"
        )
        print_yellow(f"--logfile was not specified and therefore was set to '{logfile}'")

    logfile = os.path.abspath(logfile)

    logging.basicConfig(
        format=constants.DEFAULT_LOGGING_FORMAT,
        level=constants.DEFAULT_LOGGING_LEVEL,
        filename=logfile,
    )

    # Read the configuration from the YAML file and
    # load into dictionary.
    logging.info(f"Will load contents of config file '{config_file}'")
    config = yaml.safe_load(Path(config_file).read_text())

    manager = JSONManager(
        config=config,
        config_file=config_file,
        data_file_type=data_file_type,
        logfile=logfile,
        outdir=outdir,
        outfile=outfile,
        namespace=namespace,
        template_path=template_path,
        verbose=verbose,
    )

    manager.generate_validation_modules(infile=os.path.abspath(infile))

    if verbose:
        console.print(f"The log file is '{logfile}'")
        print_green(f"Execution of '{os.path.abspath(__file__)}' completed")


if __name__ == "__main__":
    main()
