"""
CivilGuardVehiclePlateValueObject value object.
"""

from re import Pattern, compile as re_compile
from typing import NoReturn

from value_object_pattern.decorators import process, validation
from value_object_pattern.usables import NotEmptyStringValueObject, TrimmedStringValueObject


class CivilGuardVehiclePlateValueObject(NotEmptyStringValueObject, TrimmedStringValueObject):
    """
    CivilGuardVehiclePlateValueObject value object ensures the provided value is a valid Spanish civil guard plate. The plate
    format is PGC, followed by 5 digits with a final letter identifying the vehicle type, and it can contain spaces,
    hyphens, or no separators.

    References:
        Plates: https://matriculasdelmundo.com/espana.html

    Example:
    ```python
    from value_object_pattern.usables.identifiers.world.europe.spain.plates import CivilGuardVehiclePlateValueObject

    plate = CivilGuardVehiclePlateValueObject(value='PGC-12345E')

    print(repr(plate))
    # >>> CivilGuardVehiclePlateValueObject(value=PGC12345E)
    ```
    """  # noqa: E501  # fmt: skip

    _IDENTIFICATION_REGEX: Pattern[str] = re_compile(pattern=r'([pP][gG][cC])[-\s]?([0-9]{5})[-\s]?([a-zA-Z]{1})')  # noqa: E501  # fmt: skip

    @process(order=0)
    def _ensure_value_is_upper(self, value: str) -> str:
        """
        Ensures the value object `value` is stored in upper case.

        Args:
            value (str): The provided value.

        Returns:
            str: Upper case value.
        """
        return value.upper()

    @process(order=1)
    def _ensure_value_is_formatted(self, value: str) -> str:
        """
        Ensures the value object `value` is stored without separators.

        Args:
            value (str): The provided value.

        Returns:
            str: Formatted value.
        """
        return self._IDENTIFICATION_REGEX.sub(repl=r'\1\2\3', string=value)

    @validation(order=0)
    def _ensure_value_follows_identification_regex(self, value: str) -> None:
        """
        Ensures the value object `value` follows the identification regex.

        Args:
            value (str): The provided value.

        Raises:
            ValueError: If the `value` does not follow the identification regex.
        """
        if not self._IDENTIFICATION_REGEX.fullmatch(string=value):
            self._raise_value_is_not_civil_guard_plate(value=value)

    def _raise_value_is_not_civil_guard_plate(self, value: str) -> NoReturn:
        """
        Raises a ValueError if the value object `value` is not a valid Spanish civil guard plate.

        Args:
            value (str): The provided value.

        Raises:
            ValueError: If the `value` is not a valid Spanish civil guard plate.
        """
        raise ValueError(
            f'CivilGuardVehiclePlateValueObject value <<<{value}>>> is not a valid Spanish civil guard plate.'
        )

    @classmethod
    def regex(cls) -> Pattern[str]:
        """
        Returns a list of regex patterns used for validation.

        Returns:
            Pattern[str]: List of regex patterns.
        """
        return cls._IDENTIFICATION_REGEX
