import type { Context } from 'aws-lambda';
import { Utility } from '@aws-lambda-powertools/commons';
import type { ClassThatLogs, HandlerMethodDecorator, LogAttributes, ConstructorOptions, LogItemExtraInput, LogItemMessage, HandlerOptions } from './types';
/**
 * ## Intro
 * The Logger utility provides an opinionated logger with output structured as JSON.
 *
 * ## Key features
 *  * Capture key fields from Lambda context, cold start and structures logging output as JSON
 *  * Log Lambda context when instructed (disabled by default)
 *  * Log sampling prints all logs for a percentage of invocations (disabled by default)
 *  * Append additional keys to structured log at any point in time
 *
 * ## Usage
 *
 * For more usage examples, see [our documentation](https://awslabs.github.io/aws-lambda-powertools-typescript/latest/core/logger/).
 *
 * ### Basic usage
 *
 * @example
 * ```typescript
 * import { Logger } from '@aws-lambda-powertools/logger';
 *
 * // Logger parameters fetched from the environment variables:
 * const logger = new Logger();
 * ```
 *
 * ### Functions usage with middleware
 *
 * If you use function-based Lambda handlers you can use the [injectLambdaContext()](#injectLambdaContext)
 * middy middleware to automatically add context to your Lambda logs.
 *
 * @example
 * ```typescript
 * import { Logger, injectLambdaContext } from '@aws-lambda-powertools/logger';
 * import middy from '@middy/core';
 *
 * const logger = new Logger();
 *
 * const lambdaHandler = async (_event: any, _context: any) => {
 *     logger.info('This is an INFO log with some context');
 * };
 *
 * export const handler = middy(lambdaHandler).use(injectLambdaContext(logger));
 * ```
 *
 * ### Object oriented usage with decorators
 *
 * If instead you use TypeScript classes to wrap your Lambda handler you can use the [@logger.injectLambdaContext()](./_aws_lambda_powertools_logger.Logger.html#injectLambdaContext) decorator.
 *
 * @example
 * ```typescript
 * import { Logger } from '@aws-lambda-powertools/logger';
 * import { LambdaInterface } from '@aws-lambda-powertools/commons';
 *
 * const logger = new Logger();
 *
 * class Lambda implements LambdaInterface {
 *
 *   // FYI: Decorator might not render properly in VSCode mouse over due to https://github.com/microsoft/TypeScript/issues/47679 and might show as *@logger* instead of `@logger.injectLambdaContext`
 *
 *     // Decorate your handler class method
 *     @logger.injectLambdaContext()
 *     public async handler(_event: any, _context: any): Promise<void> {
 *         logger.info('This is an INFO log with some context');
 *     }
 * }
 *
 * const handlerClass = new Lambda();
 * export const handler = handlerClass.handler.bind(handlerClass);
 * ```
 *
 * ### Functions usage with manual instrumentation
 *
 * If you prefer to manually instrument your Lambda handler you can use the methods in the Logger class directly.
 *
 * @example
 * ```typescript
 * import { Logger } from '@aws-lambda-powertools/logger';
 *
 * const logger = new Logger();
 *
 * export const handler = async (_event, context) => {
 *     logger.addContext(context);
 *     logger.info('This is an INFO log with some context');
 * };
 * ```
 *
 * @class
 * @implements {ClassThatLogs}
 * @see https://awslabs.github.io/aws-lambda-powertools-typescript/latest/core/logger/
 */
declare class Logger extends Utility implements ClassThatLogs {
    private console;
    private customConfigService?;
    private static readonly defaultLogLevel;
    private envVarsService;
    private logEvent;
    private logFormatter?;
    private logIndentation;
    private logLevel?;
    private readonly logLevelThresholds;
    private logsSampled;
    private persistentLogAttributes?;
    private powertoolLogData;
    /**
     * It initializes the Logger class with an optional set of options (settings).
     * *
     * @param {ConstructorOptions} options
     */
    constructor(options?: ConstructorOptions);
    /**
     * It adds the current Lambda function's invocation context data to the powertoolLogData property of the instance.
     * This context data will be part of all printed log items.
     *
     * @param {Context} context
     * @returns {void}
     */
    addContext(context: Context): void;
    /**
     * It adds the given attributes (key-value pairs) to all log items generated by this Logger instance.
     *
     * @param {LogAttributes} attributes
     * @returns {void}
     */
    addPersistentLogAttributes(attributes?: LogAttributes): void;
    /**
     * Alias for addPersistentLogAttributes.
     *
     * @param {LogAttributes} attributes
     * @returns {void}
     */
    appendKeys(attributes?: LogAttributes): void;
    /**
     * It creates a separate Logger instance, identical to the current one
     * It's possible to overwrite the new instance options by passing them.
     *
     * @param {ConstructorOptions} options
     * @returns {Logger}
     */
    createChild(options?: ConstructorOptions): Logger;
    /**
     * It prints a log item with level DEBUG.
     *
     * @param {LogItemMessage} input
     * @param {Error | LogAttributes | string} extraInput
     * @returns {void}
     */
    debug(input: LogItemMessage, ...extraInput: LogItemExtraInput): void;
    /**
     * It prints a log item with level ERROR.
     *
     * @param {LogItemMessage} input
     * @param {Error | LogAttributes | string} extraInput
     * @returns {void}
     */
    error(input: LogItemMessage, ...extraInput: LogItemExtraInput): void;
    /**
     * It returns a boolean value. True means that the Lambda invocation events
     * are printed in the logs.
     *
     * @returns {boolean}
     */
    getLogEvent(): boolean;
    /**
     * It returns a boolean value, if true all the logs will be printed.
     *
     * @returns {boolean}
     */
    getLogsSampled(): boolean;
    /**
     * It returns the persistent log attributes, which are the attributes
     * that will be logged in all log items.
     *
     * @private
     * @returns {LogAttributes}
     */
    getPersistentLogAttributes(): LogAttributes;
    /**
     * It prints a log item with level INFO.
     *
     * @param {LogItemMessage} input
     * @param {Error | LogAttributes | string} extraInput
     * @returns {void}
     */
    info(input: LogItemMessage, ...extraInput: LogItemExtraInput): void;
    /**
     * Method decorator that adds the current Lambda function context as extra
     * information in all log items.
     *
     * The decorator can be used only when attached to a Lambda function handler which
     * is written as method of a class, and should be declared just before the handler declaration.
     *
     * Note: Currently TypeScript only supports decorators on classes and methods. If you are using the
     * function syntax, you should use the middleware instead.
     *
     * @example
     * ```typescript
     * import { Logger } from '@aws-lambda-powertools/logger';
     * import { LambdaInterface } from '@aws-lambda-powertools/commons';
     *
     * const logger = new Logger();
     *
     * class Lambda implements LambdaInterface {
     *     // Decorate your handler class method
     *     @logger.injectLambdaContext()
     *     public async handler(_event: any, _context: any): Promise<void> {
     *         logger.info('This is an INFO log with some context');
     *     }
     * }
     *
     * const handlerClass = new Lambda();
     * export const handler = handlerClass.handler.bind(handlerClass);
     * ```
     *
     * @see https://www.typescriptlang.org/docs/handbook/decorators.html#method-decorators
     * @returns {HandlerMethodDecorator}
     */
    injectLambdaContext(options?: HandlerOptions): HandlerMethodDecorator;
    static injectLambdaContextAfterOrOnError(logger: Logger, initialPersistentAttributes: LogAttributes, options?: HandlerOptions): void;
    static injectLambdaContextBefore(logger: Logger, event: unknown, context: Context, options?: HandlerOptions): void;
    /**
     * Logs a Lambda invocation event, if it *should*.
     *
     ** @param {unknown} event
     * @param {boolean} [overwriteValue]
     * @returns {void}
     */
    logEventIfEnabled(event: unknown, overwriteValue?: boolean): void;
    /**
     * If the sample rate feature is enabled, the calculation that determines whether the logs
     * will actually be printed or not for this invocation is done when the Logger class is
     * initialized.
     * This method will repeat that calculation (with possible different outcome).
     *
     * @returns {void}
     */
    refreshSampleRateCalculation(): void;
    /**
     * Alias for removePersistentLogAttributes.
     *
     * @param {string[]} keys
     * @returns {void}
     */
    removeKeys(keys: string[]): void;
    /**
     * It removes attributes based on provided keys to all log items generated by this Logger instance.
     *
     * @param {string[]} keys
     * @returns {void}
     */
    removePersistentLogAttributes(keys: string[]): void;
    /**
     * It sets the given attributes (key-value pairs) to all log items generated by this Logger instance.
     * Note: this replaces the pre-existing value.
     *
     * @param {LogAttributes} attributes
     * @returns {void}
     */
    setPersistentLogAttributes(attributes: LogAttributes): void;
    /**
     * It sets the user-provided sample rate value.
     *
     * @param {number} [sampleRateValue]
     * @returns {void}
     */
    setSampleRateValue(sampleRateValue?: number): void;
    /**
     * It checks whether the current Lambda invocation event should be printed in the logs or not.
     *
     * @private
     * @param {boolean} [overwriteValue]
     * @returns {boolean}
     */
    shouldLogEvent(overwriteValue?: boolean): boolean;
    /**
     * It prints a log item with level WARN.
     *
     * @param {LogItemMessage} input
     * @param {Error | LogAttributes | string} extraInput
     * @returns {void}
     */
    warn(input: LogItemMessage, ...extraInput: LogItemExtraInput): void;
    /**
     * It stores information that is printed in all log items.
     *
     * @param {Partial<PowertoolLogData>} attributesArray
     * @private
     * @returns {void}
     */
    private addToPowertoolLogData;
    /**
     * It processes a particular log item so that it can be printed to stdout:
     * - Merges ephemeral log attributes with persistent log attributes (printed for all logs) and additional info;
     * - Formats all the log attributes;
     *
     * @private
     * @param {LogLevel} logLevel
     * @param {LogItemMessage} input
     * @param {LogItemExtraInput} extraInput
     * @returns {LogItem}
     */
    private createAndPopulateLogItem;
    /**
     * It returns the custom config service, an abstraction used to fetch environment variables.
     *
     * @private
     * @returns {ConfigServiceInterface | undefined}
     */
    private getCustomConfigService;
    /**
     * It returns the instance of a service that fetches environment variables.
     *
     * @private
     * @returns {EnvironmentVariablesService}
     */
    private getEnvVarsService;
    /**
     * It returns the instance of a service that formats the structure of a
     * log item's keys and values in the desired way.
     *
     * @private
     * @returns {LogFormatterInterface}
     */
    private getLogFormatter;
    /**
     * It returns the log level set for the Logger instance.
     *
     * @private
     * @returns {LogLevel}
     */
    private getLogLevel;
    /**
     * It returns information that will be added in all log item by
     * this Logger instance (different from user-provided persistent attributes).
     *
     * @private
     * @returns {LogAttributes}
     */
    private getPowertoolLogData;
    /**
     * It returns the numeric sample rate value.
     *
     * @private
     * @returns {number}
     */
    private getSampleRateValue;
    /**
     * It returns true if the provided log level is valid.
     *
     * @param {LogLevel} logLevel
     * @private
     * @returns {boolean}
     */
    private isValidLogLevel;
    /**
     * It prints a given log with given log level.
     *
     * @param {LogLevel} logLevel
     * @param {LogItem} log
     * @private
     */
    private printLog;
    /**
     * It prints a given log with given log level.
     *
     * @param {LogLevel} logLevel
     * @param {LogItem} log
     * @private
     */
    private processLogItem;
    /**
     * When the data added in the log item contains object references,
     * JSON.stringify() doesn't try to solve them and instead throws an error: TypeError: cyclic object value.
     * To mitigate this issue, this method will find and remove all cyclic references.
     *
     * @see https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Errors/Cyclic_object_value
     * @private
     */
    private removeCircularDependencies;
    /**
     * It initializes console property as an instance of the internal version of Console() class (PR #748)
     * or as the global node console if the `POWERTOOLS_DEV' env variable is set and has truthy value.
     *
     * @private
     * @returns {void}
     */
    private setConsole;
    /**
     * Sets the Logger's customer config service instance, which will be used
     * to fetch environment variables.
     *
     * @private
     * @param {ConfigServiceInterface} customConfigService
     * @returns {void}
     */
    private setCustomConfigService;
    /**
     * Sets the Logger's custom config service instance, which will be used
     * to fetch environment variables.
     *
     * @private
     * @param {ConfigServiceInterface} customConfigService
     * @returns {void}
     */
    private setEnvVarsService;
    /**
     * If the log event feature is enabled via env variable, it sets a property that tracks whether
     * the event passed to the Lambda function handler should be logged or not.
     *
     * @private
     * @returns {void}
     */
    private setLogEvent;
    /**
     * It sets the log formatter instance, in charge of giving a custom format
     * to the structured logs
     *
     * @private
     * @param {LogFormatterInterface} logFormatter
     * @returns {void}
     */
    private setLogFormatter;
    /**
     * If the `POWERTOOLS_DEV' env variable is set,
     * it adds JSON indentation for pretty printing logs.
     *
     * @private
     * @returns {void}
     */
    private setLogIndentation;
    /**
     * It sets the Logger's instance log level.
     *
     * @private
     * @param {LogLevel} logLevel
     * @returns {void}
     */
    private setLogLevel;
    /**
     * If the sample rate feature is enabled, it sets a property that tracks whether this Lambda function invocation
     * will print logs or not.
     *
     * @private
     * @returns {void}
     */
    private setLogsSampled;
    /**
     * It configures the Logger instance settings that will affect the Logger's behaviour
     * and the content of all logs.
     *
     * @private
     * @param {ConstructorOptions} options
     * @returns {Logger}
     */
    private setOptions;
    /**
     * It adds important data to the Logger instance that will affect the content of all logs.
     *
     * @param {string} serviceName
     * @param {Environment} environment
     * @param {LogAttributes} persistentLogAttributes
     * @private
     * @returns {void}
     */
    private setPowertoolLogData;
    /**
     * It checks whether the current log item should/can be printed.
     *
     * @param {string} serviceName
     * @param {Environment} environment
     * @param {LogAttributes} persistentLogAttributes
     * @private
     * @returns {boolean}
     */
    private shouldPrint;
}
export { Logger };
