"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.injectLambdaContext = void 0;
const Logger_1 = require("../Logger");
/**
 * A middy middleware that helps emitting CloudWatch EMF metrics in your logs.
 *
 * Using this middleware on your handler function will automatically add context information to logs, as well as optionally log the event and clear attributes set during the invocation.
 *
 * @example
 * ```typescript
 * import { Logger, injectLambdaContext } from '@aws-lambda-powertools/logger';
 * import middy from '@middy/core';
 *
 *
 * const logger = new Logger();
 *
 * const lambdaHandler = async (_event: any, _context: any) => {
 *     logger.info('This is an INFO log with some context');
 * };
 *
 * export const handler = middy(lambdaHandler).use(injectLambdaContext(logger));
 * ```
 *
 * @param target - The Logger instance(s) to use for logging
 * @param options - (_optional_) Options for the middleware
 * @returns - The middy middleware object
 */
const injectLambdaContext = (target, options) => {
    const loggers = target instanceof Array ? target : [target];
    const persistentAttributes = [];
    const injectLambdaContextBefore = async (request) => {
        loggers.forEach((logger) => {
            if (options && options.clearState === true) {
                persistentAttributes.push({ ...logger.getPersistentLogAttributes() });
            }
            Logger_1.Logger.injectLambdaContextBefore(logger, request.event, request.context, options);
        });
    };
    const injectLambdaContextAfterOrOnError = async () => {
        if (options && options.clearState === true) {
            loggers.forEach((logger, index) => {
                Logger_1.Logger.injectLambdaContextAfterOrOnError(logger, persistentAttributes[index], options);
            });
        }
    };
    return {
        before: injectLambdaContextBefore,
        after: injectLambdaContextAfterOrOnError,
        onError: injectLambdaContextAfterOrOnError
    };
};
exports.injectLambdaContext = injectLambdaContext;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibWlkZHkuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvbWlkZGxld2FyZS9taWRkeS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxzQ0FBbUM7QUFJbkM7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O0dBdUJHO0FBQ0gsTUFBTSxtQkFBbUIsR0FBRyxDQUFDLE1BQXlCLEVBQUUsT0FBd0IsRUFBdUIsRUFBRTtJQUV2RyxNQUFNLE9BQU8sR0FBRyxNQUFNLFlBQVksS0FBSyxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDNUQsTUFBTSxvQkFBb0IsR0FBb0IsRUFBRSxDQUFDO0lBRWpELE1BQU0seUJBQXlCLEdBQUcsS0FBSyxFQUFFLE9BQXNCLEVBQWlCLEVBQUU7UUFDaEYsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDLE1BQWMsRUFBRSxFQUFFO1lBQ2pDLElBQUksT0FBTyxJQUFJLE9BQU8sQ0FBQyxVQUFVLEtBQUssSUFBSSxFQUFFO2dCQUMxQyxvQkFBb0IsQ0FBQyxJQUFJLENBQUMsRUFBRSxHQUFHLE1BQU0sQ0FBQywwQkFBMEIsRUFBRSxFQUFFLENBQUMsQ0FBQzthQUN2RTtZQUNELGVBQU0sQ0FBQyx5QkFBeUIsQ0FBQyxNQUFNLEVBQUUsT0FBTyxDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsT0FBTyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQ3BGLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDO0lBRUYsTUFBTSxpQ0FBaUMsR0FBRyxLQUFLLElBQW1CLEVBQUU7UUFDbEUsSUFBSSxPQUFPLElBQUksT0FBTyxDQUFDLFVBQVUsS0FBSyxJQUFJLEVBQUU7WUFDMUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDLE1BQWMsRUFBRSxLQUFhLEVBQUUsRUFBRTtnQkFDaEQsZUFBTSxDQUFDLGlDQUFpQyxDQUFDLE1BQU0sRUFBRSxvQkFBb0IsQ0FBQyxLQUFLLENBQUMsRUFBRSxPQUFPLENBQUMsQ0FBQztZQUN6RixDQUFDLENBQUMsQ0FBQztTQUNKO0lBQ0gsQ0FBQyxDQUFDO0lBRUYsT0FBTztRQUNMLE1BQU0sRUFBRSx5QkFBeUI7UUFDakMsS0FBSyxFQUFFLGlDQUFpQztRQUN4QyxPQUFPLEVBQUUsaUNBQWlDO0tBQzNDLENBQUM7QUFDSixDQUFDLENBQUM7QUFHQSxrREFBbUIifQ==