"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.logMetrics = void 0;
/**
 * A middy middleware automating capture of metadata and annotations on segments or subsegments for a Lambda Handler.
 *
 * Using this middleware on your handler function will automatically flush metrics after the function returns or throws an error.
 * Additionally, you can configure the middleware to easily:
 * * ensure that at least one metric is emitted before you flush them
 * * capture a `ColdStart` a metric
 * * set default dimensions for all your metrics
 *
 * @example
 * ```typescript
 * import { Metrics, logMetrics } from '@aws-lambda-powertools/metrics';
 * import middy from '@middy/core';
 *
 * const metrics = new Metrics({ namespace: 'serverlessAirline', serviceName: 'orders' });
 *
 * const lambdaHandler = async (_event: any, _context: any) => {
 *   ...
 * };
 *
 * export const handler = middy(lambdaHandler).use(logMetrics(metrics));
 * ```
 *
 * @param target - The Metrics instance to use for emitting metrics
 * @param options - (_optional_) Options for the middleware
 * @returns middleware - The middy middleware object
 */
const logMetrics = (target, options = {}) => {
    const metricsInstances = target instanceof Array ? target : [target];
    const logMetricsBefore = async (request) => {
        metricsInstances.forEach((metrics) => {
            metrics.setFunctionName(request.context.functionName);
            const { throwOnEmptyMetrics, defaultDimensions, captureColdStartMetric } = options;
            if (throwOnEmptyMetrics) {
                metrics.throwOnEmptyMetrics();
            }
            if (defaultDimensions !== undefined) {
                metrics.setDefaultDimensions(defaultDimensions);
            }
            if (captureColdStartMetric) {
                metrics.captureColdStartMetric();
            }
        });
    };
    const logMetricsAfterOrError = async () => {
        metricsInstances.forEach((metrics) => {
            metrics.publishStoredMetrics();
        });
    };
    return {
        before: logMetricsBefore,
        after: logMetricsAfterOrError,
        onError: logMetricsAfterOrError
    };
};
exports.logMetrics = logMetrics;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibWlkZHkuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvbWlkZGxld2FyZS9taWRkeS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFJQTs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0EwQkc7QUFDSCxNQUFNLFVBQVUsR0FBRyxDQUFDLE1BQTJCLEVBQUUsVUFBd0IsRUFBRSxFQUF1QixFQUFFO0lBQ2xHLE1BQU0sZ0JBQWdCLEdBQUcsTUFBTSxZQUFZLEtBQUssQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDO0lBRXJFLE1BQU0sZ0JBQWdCLEdBQUcsS0FBSyxFQUFFLE9BQXNCLEVBQWlCLEVBQUU7UUFDdkUsZ0JBQWdCLENBQUMsT0FBTyxDQUFDLENBQUMsT0FBZ0IsRUFBRSxFQUFFO1lBQzVDLE9BQU8sQ0FBQyxlQUFlLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsQ0FBQztZQUN0RCxNQUFNLEVBQUUsbUJBQW1CLEVBQUUsaUJBQWlCLEVBQUUsc0JBQXNCLEVBQUUsR0FBRyxPQUFPLENBQUM7WUFDbkYsSUFBSSxtQkFBbUIsRUFBRTtnQkFDdkIsT0FBTyxDQUFDLG1CQUFtQixFQUFFLENBQUM7YUFDL0I7WUFDRCxJQUFJLGlCQUFpQixLQUFLLFNBQVMsRUFBRTtnQkFDbkMsT0FBTyxDQUFDLG9CQUFvQixDQUFDLGlCQUFpQixDQUFDLENBQUM7YUFDakQ7WUFDRCxJQUFJLHNCQUFzQixFQUFFO2dCQUMxQixPQUFPLENBQUMsc0JBQXNCLEVBQUUsQ0FBQzthQUNsQztRQUNILENBQUMsQ0FBQyxDQUFDO0lBRUwsQ0FBQyxDQUFDO0lBRUYsTUFBTSxzQkFBc0IsR0FBRyxLQUFLLElBQW1CLEVBQUU7UUFDdkQsZ0JBQWdCLENBQUMsT0FBTyxDQUFDLENBQUMsT0FBZ0IsRUFBRSxFQUFFO1lBQzVDLE9BQU8sQ0FBQyxvQkFBb0IsRUFBRSxDQUFDO1FBQ2pDLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDO0lBRUYsT0FBTztRQUNMLE1BQU0sRUFBRSxnQkFBZ0I7UUFDeEIsS0FBSyxFQUFFLHNCQUFzQjtRQUM3QixPQUFPLEVBQUUsc0JBQXNCO0tBQ2hDLENBQUM7QUFDSixDQUFDLENBQUM7QUFHQSxnQ0FBVSJ9