"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.captureLambdaHandler = void 0;
/**
 * A middy middleware automating capture of metadata and annotations on segments or subsegments for a Lambda Handler.
 *
 * Using this middleware on your handler function will automatically:
 * * handle the subsegment lifecycle
 * * add the `ColdStart` annotation
 * * add the function response as metadata
 * * add the function error as metadata (if any)
 *
 * @example
 * ```typescript
 * import { Tracer, captureLambdaHandler } from '@aws-lambda-powertools/tracer';
 * import middy from '@middy/core';
 *
 * const tracer = new Tracer({ serviceName: 'serverlessAirline' });
 *
 * const lambdaHandler = async (_event: any, _context: any) => {
 *   ...
 * };
 *
 * export const handler = middy(lambdaHandler).use(captureLambdaHandler(tracer));
 * ```
 *
 * @param target - The Tracer instance to use for tracing
 * @param options - (_optional_) Options for the middleware
 * @returns middleware - The middy middleware object
 */
const captureLambdaHandler = (target, options) => {
    let lambdaSegment;
    const open = () => {
        lambdaSegment = target.getSegment();
        const handlerSegment = lambdaSegment.addNewSubsegment(`## ${process.env._HANDLER}`);
        target.setSegment(handlerSegment);
    };
    const close = () => {
        const subsegment = target.getSegment();
        subsegment.close();
        target.setSegment(lambdaSegment);
    };
    const captureLambdaHandlerBefore = async () => {
        if (target.isTracingEnabled()) {
            open();
            target.annotateColdStart();
            target.addServiceNameAnnotation();
        }
    };
    const captureLambdaHandlerAfter = async (request) => {
        var _a;
        if (target.isTracingEnabled()) {
            if ((_a = options === null || options === void 0 ? void 0 : options.captureResponse) !== null && _a !== void 0 ? _a : true) {
                target.addResponseAsMetadata(request.response, process.env._HANDLER);
            }
            close();
        }
    };
    const captureLambdaHandlerError = async (request) => {
        if (target.isTracingEnabled()) {
            target.addErrorAsMetadata(request.error);
            close();
        }
    };
    return {
        before: captureLambdaHandlerBefore,
        after: captureLambdaHandlerAfter,
        onError: captureLambdaHandlerError
    };
};
exports.captureLambdaHandler = captureLambdaHandler;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibWlkZHkuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvbWlkZGxld2FyZS9taWRkeS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFLQTs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0EwQkc7QUFDSCxNQUFNLG9CQUFvQixHQUFHLENBQUMsTUFBYyxFQUFFLE9BQXFDLEVBQXVCLEVBQUU7SUFDMUcsSUFBSSxhQUFtQyxDQUFDO0lBRXhDLE1BQU0sSUFBSSxHQUFHLEdBQVMsRUFBRTtRQUN0QixhQUFhLEdBQUcsTUFBTSxDQUFDLFVBQVUsRUFBRSxDQUFDO1FBQ3BDLE1BQU0sY0FBYyxHQUFHLGFBQWEsQ0FBQyxnQkFBZ0IsQ0FBQyxNQUFNLE9BQU8sQ0FBQyxHQUFHLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQztRQUNwRixNQUFNLENBQUMsVUFBVSxDQUFDLGNBQWMsQ0FBQyxDQUFDO0lBQ3BDLENBQUMsQ0FBQztJQUVGLE1BQU0sS0FBSyxHQUFHLEdBQVMsRUFBRTtRQUN2QixNQUFNLFVBQVUsR0FBRyxNQUFNLENBQUMsVUFBVSxFQUFFLENBQUM7UUFDdkMsVUFBVSxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQ25CLE1BQU0sQ0FBQyxVQUFVLENBQUMsYUFBd0IsQ0FBQyxDQUFDO0lBQzlDLENBQUMsQ0FBQztJQUVGLE1BQU0sMEJBQTBCLEdBQUcsS0FBSyxJQUFtQixFQUFFO1FBQzNELElBQUksTUFBTSxDQUFDLGdCQUFnQixFQUFFLEVBQUU7WUFDN0IsSUFBSSxFQUFFLENBQUM7WUFDUCxNQUFNLENBQUMsaUJBQWlCLEVBQUUsQ0FBQztZQUMzQixNQUFNLENBQUMsd0JBQXdCLEVBQUUsQ0FBQztTQUNuQztJQUNILENBQUMsQ0FBQztJQUVGLE1BQU0seUJBQXlCLEdBQUcsS0FBSyxFQUFFLE9BQXNCLEVBQWlCLEVBQUU7O1FBQ2hGLElBQUksTUFBTSxDQUFDLGdCQUFnQixFQUFFLEVBQUU7WUFDN0IsSUFBSSxNQUFBLE9BQU8sYUFBUCxPQUFPLHVCQUFQLE9BQU8sQ0FBRSxlQUFlLG1DQUFJLElBQUksRUFBRTtnQkFDcEMsTUFBTSxDQUFDLHFCQUFxQixDQUFDLE9BQU8sQ0FBQyxRQUFRLEVBQUUsT0FBTyxDQUFDLEdBQUcsQ0FBQyxRQUFRLENBQUMsQ0FBQzthQUN0RTtZQUNELEtBQUssRUFBRSxDQUFDO1NBQ1Q7SUFDSCxDQUFDLENBQUM7SUFFRixNQUFNLHlCQUF5QixHQUFHLEtBQUssRUFBRSxPQUFzQixFQUFpQixFQUFFO1FBQ2hGLElBQUksTUFBTSxDQUFDLGdCQUFnQixFQUFFLEVBQUU7WUFDN0IsTUFBTSxDQUFDLGtCQUFrQixDQUFDLE9BQU8sQ0FBQyxLQUFjLENBQUMsQ0FBQztZQUNsRCxLQUFLLEVBQUUsQ0FBQztTQUNUO0lBQ0gsQ0FBQyxDQUFDO0lBRUYsT0FBTztRQUNMLE1BQU0sRUFBRSwwQkFBMEI7UUFDbEMsS0FBSyxFQUFFLHlCQUF5QjtRQUNoQyxPQUFPLEVBQUUseUJBQXlCO0tBQ25DLENBQUM7QUFDSixDQUFDLENBQUM7QUFHQSxvREFBb0IifQ==