from functools import wraps
from velocity.db import exceptions


def retry_on_dup_key(function):
    # retry_on_dup_key is needed to skip past existing autogenerated ids when inserting
    @wraps(function)
    def retry_decorator(self, *args, **kwds):
        sp = self.tx.create_savepoint(cursor=self.cursor)
        while True:
            try:
                return function(self, *args, **kwds)
            except exceptions.DbDuplicateKeyError:
                self.tx.rollback_savepoint(sp, cursor=self.cursor)
                continue

    return retry_decorator


def reset_id_on_dup_key(function):
    # reset_id_on_dup_key sets sys_id to max(sys_id) + 1 when inserting
    @wraps(function)
    def reset_decorator(self, *args, **kwds):
        sp = self.tx.create_savepoint(cursor=self.cursor)
        try:
            return function(self, *args, **kwds)
        except exceptions.DbDuplicateKeyError:
            self.tx.rollback_savepoint(sp, cursor=self.cursor)
            self.set_sequence(self.max("sys_id") + 1)
            return function(self, *args, **kwds)

    return reset_decorator


def return_default(
    default=None,
    exceptions=(
        StopIteration,
        exceptions.DbApplicationError,
        exceptions.DbTableMissingError,
        exceptions.DbColumnMissingError,
        exceptions.DbTruncationError,
        exceptions.DbObjectExistsError,
    ),
):
    def decorator(f):
        f.default = default
        f.exceptions = exceptions

        @wraps(f)
        def return_default(self, *args, **kwds):
            sp = self.tx.create_savepoint(cursor=self.cursor)
            try:
                result = f(self, *args, **kwds)
                if result is None:
                    result = default
            except f.exceptions as e:
                self.tx.rollback_savepoint(sp, cursor=self.cursor)
                return f.default
            self.tx.release_savepoint(sp, cursor=self.cursor)
            return result

        return return_default

    return decorator


def create_missing(function):
    @wraps(function)
    def create_missing_decorator(self, *args, **kwds):
        sp = self.tx.create_savepoint(cursor=self.cursor)
        try:
            result = function(self, *args, **kwds)
            self.tx.release_savepoint(sp, cursor=self.cursor)
        except exceptions.DbColumnMissingError:
            self.tx.rollback_savepoint(sp, cursor=self.cursor)
            data = {}
            if "pk" in kwds:
                data.update(kwds["pk"])
            if "data" in kwds:
                data.update(kwds["data"])
            for i in range(len(args)):
                if args[i]:
                    data.update(args[i])
            self.alter(data)
            result = function(self, *args, **kwds)
        except exceptions.DbTableMissingError:
            self.tx.rollback_savepoint(sp, cursor=self.cursor)
            data = {}
            if "pk" in kwds:
                data.update(kwds["pk"])
            if "data" in kwds:
                data.update(kwds["data"])
            for i in range(len(args)):
                data.update(args[i])
            self.create(data)
            result = function(self, *args, **kwds)
        return result

    return create_missing_decorator
