import json
from pathlib import Path
import vl_convert as vlc
import pytest
from io import BytesIO
from skimage.io import imread
from skimage.metrics import structural_similarity as ssim

tests_dir = Path(__file__).parent
root_dir = tests_dir.parent.parent
specs_dir = root_dir / "vl-convert-rs" / "tests" / "vl-specs"
fonts_dir = root_dir / "vl-convert-rs" / "tests" / "fonts"

BACKGROUND_COLOR = "#abc"


def setup_module(module):
    vlc.register_font_directory(str(fonts_dir))


def load_vl_spec(name):
    spec_path = specs_dir / f"{name}.vl.json"
    with open(spec_path, "rt") as f:
        spec_str = f.read()
    return spec_str


def load_expected_vg_spec(name, vl_version):
    filename = f"{name}.vg.json"
    spec_path = specs_dir / "expected" / vl_version / filename
    if spec_path.exists():
        with spec_path.open("rt", encoding="utf8") as f:
            return json.load(f)
    else:
        return None


def load_expected_svg(name, vl_version):
    filename = f"{name}.svg"
    spec_path = specs_dir / "expected" / vl_version / filename
    with spec_path.open("rt", encoding="utf8") as f:
        return f.read()


def load_expected_png(name, vl_version, theme=None):
    filename = f"{name}-{theme}.png" if theme else f"{name}.png"
    spec_path = specs_dir / "expected" / vl_version / filename
    with spec_path.open("rb") as f:
        return f.read()


@pytest.mark.parametrize("name", ["circle_binned", "seattle-weather", "stacked_bar_h"])
@pytest.mark.parametrize(
    "vl_version",
    [
        "v4_17",
        "v5_6",
        "v5_7",
        "v5_8",
        "v5_9",
        "v5_10",
        "v5_11",
        "v5_12",
        "v5_13",
        "v5_14",
    ],
)
@pytest.mark.parametrize("as_dict", [False, True])
def test_vega(name, vl_version, as_dict):
    vl_spec = load_vl_spec(name)

    if as_dict:
        vl_spec = json.loads(vl_spec)

    expected_vg_spec = load_expected_vg_spec(name, vl_version)

    if expected_vg_spec is None:
        with pytest.raises(ValueError):
            vlc.vegalite_to_vega(vl_spec, vl_version=vl_version)
    else:
        vg_spec = vlc.vegalite_to_vega(vl_spec, vl_version=vl_version)
        assert expected_vg_spec == vg_spec


@pytest.mark.parametrize("name", ["circle_binned", "stacked_bar_h"])
@pytest.mark.parametrize("as_dict", [False, True])
def test_svg(name, as_dict):
    vl_version = "v5_8"
    vl_spec = load_vl_spec(name)

    if as_dict:
        vl_spec = json.loads(vl_spec)

    expected_svg = load_expected_svg(name, vl_version)

    # Convert to vega first
    vg_spec = vlc.vegalite_to_vega(vl_spec, vl_version=vl_version)
    svg = vlc.vega_to_svg(vg_spec)
    check_svg(svg, expected_svg)

    # Convert directly to image
    svg = vlc.vegalite_to_svg(vl_spec, vl_version=vl_version)
    check_svg(svg, expected_svg)


@pytest.mark.parametrize(
    "name,scale",
    [
        ("circle_binned", 1.0),
        ("stacked_bar_h", 2.0),
        ("remote_images", 1.0),
        ("maptile_background", 1.0),
        ("no_text_in_font_metrics", 1.0),
        ("lookup_urls", 1.0),
    ],
)
@pytest.mark.parametrize("as_dict", [False])
def test_png(name, scale, as_dict):
    vl_version = "v5_8"
    vl_spec = load_vl_spec(name)

    if as_dict:
        vl_spec = json.loads(vl_spec)

    expected_png = load_expected_png(name, vl_version)

    # Convert to vega first
    vg_spec = vlc.vegalite_to_vega(vl_spec, vl_version=vl_version)
    png = vlc.vega_to_png(vg_spec, scale=scale)
    check_png(png, expected_png)

    # Convert directly to image
    png = vlc.vegalite_to_png(vl_spec, vl_version=vl_version, scale=scale)
    check_png(png, expected_png)


@pytest.mark.parametrize(
    "name,scale,theme", [("circle_binned", 1.0, "dark"), ("stacked_bar_h", 2.0, "vox")]
)
def test_png_theme_config(name, scale, theme):
    vl_version = "v5_8"
    vl_spec = json.loads(load_vl_spec(name))

    expected_png = load_expected_png(name, vl_version, theme)

    # Convert directly to image
    config = dict(background=BACKGROUND_COLOR)
    png = vlc.vegalite_to_png(
        vl_spec,
        vl_version=vl_version,
        scale=scale,
        theme=theme,
        config=config,
    )
    check_png(png, expected_png)


def test_gh_78():
    vl_version = "v5_8"
    name = "lookup_urls"
    vl_spec = json.loads(load_vl_spec(name))

    png = None
    for i in range(30):
        png = vlc.vegalite_to_png(vl_spec, vl_version=vl_version)

    expected_png = load_expected_png(name, vl_version)
    check_png(png, expected_png)


def check_png(png, expected_png):
    png_img = imread(BytesIO(png))
    expected_png_img = imread(BytesIO(expected_png))
    similarity_value = ssim(png_img, expected_png_img, channel_axis=2)
    if similarity_value < 0.995:
        pytest.fail(f"png mismatch with similarity: {similarity_value}")


def check_svg(svg, expected_svg):
    if svg != expected_svg:
        pytest.fail(f"svg image mismatch")
