#![feature(once_cell_try)]
#![feature(trusted_len)]
#![feature(substr_range)]
//! Vortex crate containing core logic for encoding and memory representation of [arrays](Array).
//!
//! At the heart of Vortex are [arrays](Array) and [encodings](vtable::EncodingVTable).
//! Arrays are typed views of memory buffers that hold [scalars](vortex_scalar::Scalar). These
//! buffers can be held in a number of physical encodings to perform lightweight compression that
//! exploits the particular data distribution of the array's values.
//!
//! Every data type recognized by Vortex also has a canonical physical encoding format, which
//! arrays can be [canonicalized](Canonical) into for ease of access in compute functions.
//!

pub use canonical::*;
pub use children::*;
pub use context::*;
pub use data::*;
pub use encoding::*;
pub use metadata::*;
pub use paste;

pub mod accessor;
pub mod aliases;
pub mod array;
pub mod arrow;
pub mod builders;
mod canonical;
mod children;
pub mod compress;
pub mod compute;
mod context;
mod data;
mod encoding;
pub mod iter;
mod macros;
mod metadata;
pub mod nbytes;
pub mod parts;
pub mod patches;
pub mod stats;
pub mod stream;
#[cfg(feature = "test-harness")]
pub mod test_harness;
pub mod tree;
pub mod validity;
pub mod variants;
pub mod visitor;
pub mod vtable;

pub mod flatbuffers {
    //! Re-exported autogenerated code from the core Vortex flatbuffer definitions.
    pub use vortex_flatbuffers::array::*;
}

/// A depth-first pre-order iterator over a Array.
pub struct ArrayChildrenIterator {
    stack: Vec<Array>,
}

impl ArrayChildrenIterator {
    pub fn new(array: Array) -> Self {
        Self { stack: vec![array] }
    }
}

impl Iterator for ArrayChildrenIterator {
    type Item = Array;

    fn next(&mut self) -> Option<Self::Item> {
        let next = self.stack.pop()?;
        for child in next.children().into_iter().rev() {
            self.stack.push(child);
        }
        Some(next)
    }
}

/// Consume `self` and turn it into an [`Array`] infallibly.
///
/// Implementation of this array should never fail.
pub trait IntoArray {
    fn into_array(self) -> Array;
}
