#
# Copyright (c) nexB Inc. and others. All rights reserved.
# VulnerableCode is a trademark of nexB Inc.
# SPDX-License-Identifier: Apache-2.0
# See http://www.apache.org/licenses/LICENSE-2.0 for the license text.
# See https://github.com/aboutcode-org/vulnerablecode for support or download.
# See https://aboutcode.org for more information about nexB OSS projects.
#

import os
from datetime import datetime
from unittest import mock

import pytest

from vulnerabilities.improvers.vulnerability_status import VulnerabilityStatusImprover
from vulnerabilities.models import Advisory
from vulnerabilities.models import Alias
from vulnerabilities.models import Vulnerability
from vulnerabilities.models import VulnerabilityStatusType
from vulnerabilities.pipelines.nvd_importer import NVDImporterPipeline

BASE_DIR = os.path.dirname(os.path.abspath(__file__))


TEST_DATA = os.path.join(
    BASE_DIR,
    "test_data/vulnerability_status_improver",
)


@pytest.mark.django_db(transaction=True)
def test_interesting_advisories():
    Advisory.objects.create(
        aliases=["CVE-1"],
        created_by=NVDImporterPipeline.pipeline_id,
        summary="1",
        date_collected=datetime.now(),
    )
    Advisory.objects.create(
        aliases=["CVE-1"],
        created_by=NVDImporterPipeline.pipeline_id,
        summary="2",
        date_collected=datetime.now(),
    )
    advs = VulnerabilityStatusImprover().interesting_advisories
    assert len(list(advs)) == 1


@pytest.mark.django_db(transaction=True)
@mock.patch("vulnerabilities.utils.fetch_response")
def test_improver_end_to_end(mock_response):
    response = os.path.join(TEST_DATA, "CVE-2023-35866.json")
    mock_response.return_value = response
    adv = Advisory.objects.create(
        aliases=["CVE-2023-35866"],
        created_by=NVDImporterPipeline.pipeline_id,
        summary="1",
        date_collected=datetime.now(),
    )
    v1 = Vulnerability.objects.create(summary="test")
    Alias.objects.create(alias="CVE-2023-35866", vulnerability=v1)
    VulnerabilityStatusImprover().get_inferences(advisory_data=adv)
    v1 = Vulnerability.objects.get(summary="test")
    assert v1.status == VulnerabilityStatusType.DISPUTED
