"""CloudKit client for WatchCode CLI.

Enables 4-character prefix pairing by querying CloudKit for the full auth token.
The Watch app syncs authToken and deviceToken to CloudKit, and this client
queries that data to enable simplified setup.

The CloudKit S2S key is bundled with the CLI for zero-friction setup.
Advanced users can override by placing their own key at ~/.watchcode/cloudkit_key.p8
"""

import base64
import datetime
import hashlib
import json
import os
import time
from pathlib import Path
from typing import Optional, Tuple, Dict, Any

# Check for required dependencies
try:
    import requests
    import ecdsa
except ImportError:
    requests = None
    ecdsa = None


class CloudKitAuth(requests.auth.AuthBase):
    """CloudKit Server-to-Server authentication.

    Custom implementation that's Python 3.12+ compatible.
    Replaces requests_cloudkit which has compatibility issues.
    """

    def __init__(self, key_id: str, private_key_pem: str):
        """Initialize CloudKit auth.

        Args:
            key_id: CloudKit S2S key ID from Dashboard.
            private_key_pem: EC private key in PEM format.
        """
        self.key_id = key_id
        self.signing_key = ecdsa.SigningKey.from_pem(private_key_pem)

    def __call__(self, r):
        """Add CloudKit authentication headers to request."""
        # ISO 8601 date without microseconds
        dt = datetime.datetime.now(tz=datetime.timezone.utc)
        dt = dt.replace(microsecond=0)
        formatted_date = dt.isoformat().replace("+00:00", "Z")

        # Compute signature
        signature = self._make_signature(formatted_date, r.body, r.path_url)

        # Set CloudKit headers
        r.headers.update({
            'Content-Type': 'text/plain',
            'X-Apple-CloudKit-Request-SignatureV1': signature,
            'X-Apple-CloudKit-Request-KeyID': self.key_id,
            'X-Apple-CloudKit-Request-ISO8601Date': formatted_date,
        })
        return r

    def _make_signature(self, formatted_date: str, body, path: str) -> str:
        """Create CloudKit request signature.

        Format: {date}:{body_hash}:{path}
        Signed with EC private key using SHA-256.
        """
        # Encode body to get hash
        body_hash = self._encode_body(body)

        # Create signature payload
        signature_payload = f"{formatted_date}:{body_hash}:{path}"

        # Sign with EC key
        signature_bytes = self.signing_key.sign(
            signature_payload.encode('utf-8'),
            hashfunc=hashlib.sha256,
            sigencode=ecdsa.util.sigencode_der
        )

        return base64.b64encode(signature_bytes).decode('utf-8')

    def _encode_body(self, body) -> str:
        """Hash the request body for signature.

        Returns base64-encoded SHA-256 hash.
        """
        if body is None:
            body_str = ""
        elif isinstance(body, bytes):
            body_str = body.decode('utf-8')
        elif isinstance(body, str):
            body_str = body
        else:
            body_str = json.dumps(body, separators=(',', ':'))

        h = hashlib.sha256(body_str.encode('utf-8'))
        return base64.b64encode(h.digest()).decode('utf-8')

# Bundled CloudKit Server-to-Server credentials
# Safe to embed: read-only access to public database only
_BUNDLED_KEY_ID = "28f19635e1b70a0257ead31a50db60c81fc0c2ecd291c8f8f04527e8afee0e30"
_BUNDLED_PRIVATE_KEY = """-----BEGIN EC PRIVATE KEY-----
MHcCAQEEIGvtmSkQipC2r0L2oI1LAoIDL+SGJfrYkdwjH5AeQ4rEoAoGCCqGSM49
AwEHoUQDQgAEcRHNHiuhzuLXMmoP0fApfYjV3m2t63BvYp3ohBNN5Xz24DUoRLUK
xUE1SKx/RINpCeKsMCnuj8mZUREVJ1xhvA==
-----END EC PRIVATE KEY-----"""


class CloudKitError(Exception):
    """Base exception for CloudKit operations."""
    pass


class CloudKitNotConfigured(CloudKitError):
    """CloudKit credentials not configured."""
    pass


class CloudKitUnavailable(CloudKitError):
    """CloudKit service unavailable."""
    pass


class DeviceNotFound(CloudKitError):
    """No device found matching the prefix."""
    pass


class CloudKitClient:
    """Client for querying CloudKit WatchDevice records.

    Used to find a device by auth token prefix (first 4 characters),
    enabling simplified setup without typing the full 12-character code.
    """

    # CloudKit configuration
    CONTAINER_ID = "iCloud.com.watchcode.app"
    ENVIRONMENT = "production"  # Use production for App Store app
    API_VERSION = "1"
    BASE_URL = f"https://api.apple-cloudkit.com/database/{API_VERSION}/{CONTAINER_ID}/{ENVIRONMENT}"

    # Record type and fields
    RECORD_TYPE = "WatchDevice"
    RECORD_NAME = "watch_device"  # Fixed record name used by Watch app

    # Local config paths
    CONFIG_DIR = Path.home() / ".watchcode"
    CONFIG_FILE = CONFIG_DIR / "config.json"
    KEY_FILE = CONFIG_DIR / "cloudkit_key.p8"

    # Retry configuration
    MAX_RETRIES = 3
    BASE_DELAY = 1.0  # seconds
    MAX_DELAY = 10.0  # seconds

    def __init__(self):
        """Initialize CloudKit client."""
        self._auth: Optional[CloudKitAuth] = None
        self._config: Optional[Dict[str, Any]] = None

    @classmethod
    def is_available(cls) -> bool:
        """Check if CloudKit client dependencies are available.

        Returns:
            True if requests and ecdsa are installed.
        """
        return requests is not None and ecdsa is not None

    @classmethod
    def is_configured(cls) -> bool:
        """Check if CloudKit credentials are configured.

        Returns:
            True if bundled key exists OR custom key is configured.
        """
        if not cls.is_available():
            return False

        # Bundled key is always available
        if _BUNDLED_KEY_ID and _BUNDLED_PRIVATE_KEY:
            return True

        # Check for custom key override
        config = cls._load_config_static()
        key_id = config.get("cloudkit_key_id")
        key_file = config.get("cloudkit_key_file", str(cls.KEY_FILE))

        return bool(key_id) and os.path.exists(key_file)

    @staticmethod
    def _load_config_static() -> Dict[str, Any]:
        """Load configuration from config.json (static method)."""
        config_file = Path.home() / ".watchcode" / "config.json"
        if not config_file.exists():
            return {}
        try:
            with open(config_file, "r") as f:
                return json.load(f)
        except (json.JSONDecodeError, IOError):
            return {}

    def _load_config(self) -> Dict[str, Any]:
        """Load configuration from config.json."""
        if self._config is not None:
            return self._config

        if not self.CONFIG_FILE.exists():
            self._config = {}
            return self._config

        try:
            with open(self.CONFIG_FILE, "r") as f:
                self._config = json.load(f)
        except (json.JSONDecodeError, IOError):
            self._config = {}

        return self._config

    def _get_auth(self) -> CloudKitAuth:
        """Get CloudKit authentication handler.

        Uses bundled key by default, with optional custom key override.

        Returns:
            CloudKitAuth instance.

        Raises:
            CloudKitNotConfigured: If credentials are not set up.
        """
        if self._auth is not None:
            return self._auth

        if not self.is_available():
            raise CloudKitNotConfigured(
                "CloudKit dependencies not installed. "
                "Run: pip install requests ecdsa"
            )

        config = self._load_config()

        # Check for custom key override first
        custom_key_id = config.get("cloudkit_key_id")
        custom_key_file = config.get("cloudkit_key_file", str(self.KEY_FILE))

        if custom_key_id and os.path.exists(custom_key_file):
            # Use custom key from file
            with open(custom_key_file, 'r') as f:
                private_key_pem = f.read()
            self._auth = CloudKitAuth(key_id=custom_key_id, private_key_pem=private_key_pem)
            return self._auth

        # Use bundled key
        if _BUNDLED_KEY_ID and _BUNDLED_PRIVATE_KEY:
            self._auth = CloudKitAuth(
                key_id=_BUNDLED_KEY_ID,
                private_key_pem=_BUNDLED_PRIVATE_KEY
            )
            return self._auth

        raise CloudKitNotConfigured(
            "No CloudKit credentials available. "
            "This is unexpected - please report this issue."
        )

    def _retry_with_backoff(
        self,
        operation: callable,
        operation_name: str = "operation"
    ) -> Any:
        """Execute operation with exponential backoff on failure.

        Args:
            operation: Callable to execute.
            operation_name: Name for logging.

        Returns:
            Result from operation.

        Raises:
            CloudKitUnavailable: After all retries exhausted.
        """
        last_error = None

        for attempt in range(self.MAX_RETRIES):
            try:
                return operation()
            except requests.exceptions.RequestException as e:
                last_error = e

                if attempt < self.MAX_RETRIES - 1:
                    # Exponential backoff with jitter
                    delay = min(
                        self.BASE_DELAY * (2 ** attempt) + (time.time() % 0.5),
                        self.MAX_DELAY
                    )
                    time.sleep(delay)

        raise CloudKitUnavailable(
            f"{operation_name} failed after {self.MAX_RETRIES} attempts: {last_error}"
        )

    def find_device_by_prefix(self, prefix: str) -> Tuple[str, str]:
        """Find a device by auth token prefix.

        Queries CloudKit for the WatchDevice record and validates
        that the auth token starts with the given prefix.

        Args:
            prefix: First 4 characters of the auth token (case-insensitive).

        Returns:
            Tuple of (authToken, deviceToken).

        Raises:
            CloudKitNotConfigured: If CloudKit credentials not set up.
            CloudKitUnavailable: If CloudKit service unavailable.
            DeviceNotFound: If no device matches the prefix.
        """
        # Normalize prefix to uppercase
        prefix = prefix.upper().strip()

        if len(prefix) != 4:
            raise ValueError("Prefix must be exactly 4 characters")

        if not prefix.isalnum():
            raise ValueError("Prefix must be alphanumeric")

        # Get auth handler
        auth = self._get_auth()

        # Query for the fixed record name
        url = f"{self.BASE_URL}/public/records/lookup"

        payload = {
            "records": [
                {"recordName": self.RECORD_NAME}
            ]
        }

        def do_lookup():
            response = requests.post(url, json=payload, auth=auth, timeout=10)
            response.raise_for_status()
            return response.json()

        try:
            data = self._retry_with_backoff(do_lookup, "CloudKit lookup")
        except CloudKitUnavailable:
            raise
        except requests.exceptions.RequestException as e:
            raise CloudKitUnavailable(f"CloudKit request failed: {e}")

        # Parse response
        records = data.get("records", [])

        if not records:
            raise DeviceNotFound("No WatchDevice record found in CloudKit")

        record = records[0]

        # Check for server errors
        if "serverErrorCode" in record:
            error_code = record.get("serverErrorCode")
            if error_code == "NOT_FOUND":
                raise DeviceNotFound(
                    "No Watch device registered. "
                    "Make sure WatchCode is running on your Apple Watch."
                )
            raise CloudKitUnavailable(f"CloudKit error: {error_code}")

        # Extract fields
        fields = record.get("fields", {})

        auth_token = fields.get("authToken", {}).get("value")
        device_token = fields.get("deviceToken", {}).get("value")

        if not auth_token:
            raise DeviceNotFound("WatchDevice record missing authToken")

        if not device_token:
            raise DeviceNotFound("WatchDevice record missing deviceToken")

        # Validate prefix match
        if not auth_token.upper().startswith(prefix):
            raise DeviceNotFound(
                f"No device found with prefix '{prefix}'. "
                "Check your Watch for the correct code."
            )

        return auth_token, device_token

    def get_device(self) -> Optional[Dict[str, Any]]:
        """Get the WatchDevice record without prefix validation.

        Useful for debugging or checking if any device is registered.

        Returns:
            Dict with authToken, deviceToken, timestamp if found, else None.

        Raises:
            CloudKitNotConfigured: If CloudKit credentials not set up.
            CloudKitUnavailable: If CloudKit service unavailable.
        """
        auth = self._get_auth()

        url = f"{self.BASE_URL}/public/records/lookup"

        payload = {
            "records": [
                {"recordName": self.RECORD_NAME}
            ]
        }

        def do_lookup():
            response = requests.post(url, json=payload, auth=auth, timeout=10)
            response.raise_for_status()
            return response.json()

        try:
            data = self._retry_with_backoff(do_lookup, "CloudKit lookup")
        except (CloudKitUnavailable, requests.exceptions.RequestException):
            return None

        records = data.get("records", [])
        if not records or "serverErrorCode" in records[0]:
            return None

        fields = records[0].get("fields", {})

        return {
            "authToken": fields.get("authToken", {}).get("value"),
            "deviceToken": fields.get("deviceToken", {}).get("value"),
            "timestamp": fields.get("timestamp", {}).get("value"),
        }
