"""WatchCode CLI - Send Claude Code notifications to your Apple Watch."""

import sys
import json
import shutil
from pathlib import Path
import click
from typing import Optional

from .config import Config
from .client import RelayClient
from .hooks import HooksInstaller


def install_scripts() -> dict:
    """Install hook scripts to ~/.watchcode/ directory.

    Returns dict with 'installed' and 'updated' lists.
    """
    from importlib import resources

    result = {"installed": [], "updated": [], "errors": []}

    # Ensure ~/.watchcode exists
    watchcode_dir = Path.home() / ".watchcode"
    watchcode_dir.mkdir(exist_ok=True)

    # Scripts to install
    scripts = [
        "hook_handler.py",
        "cloudkit_poller.py",
        "apns_sender.py",
        "response_listener.py"
    ]

    try:
        # Python 3.9+ way
        from importlib.resources import files
        scripts_package = files("watchcode.scripts")

        for script in scripts:
            try:
                source = scripts_package.joinpath(script)
                dest = watchcode_dir / script

                # Read source content
                content = source.read_text()

                # Check if update needed
                if dest.exists():
                    existing = dest.read_text()
                    if existing == content:
                        continue  # Already up to date
                    result["updated"].append(script)
                else:
                    result["installed"].append(script)

                # Write script
                dest.write_text(content)
                dest.chmod(0o755)  # Make executable

            except Exception as e:
                result["errors"].append(f"{script}: {e}")

    except ImportError:
        # Python 3.8 fallback
        import pkg_resources

        for script in scripts:
            try:
                source_path = pkg_resources.resource_filename("watchcode.scripts", script)
                dest = watchcode_dir / script

                with open(source_path, 'r') as f:
                    content = f.read()

                if dest.exists():
                    existing = dest.read_text()
                    if existing == content:
                        continue
                    result["updated"].append(script)
                else:
                    result["installed"].append(script)

                dest.write_text(content)
                dest.chmod(0o755)

            except Exception as e:
                result["errors"].append(f"{script}: {e}")

    return result


@click.group()
@click.version_option(package_name="watchcode-cli")
def main():
    """WatchCode CLI - Send Claude Code notifications to your Apple Watch."""
    pass


def _setup_manual(config: Config) -> str:
    """Manual setup with full 12-character code entry.

    Returns:
        The validated auth token.
    """
    click.echo("To get your setup code:")
    click.echo("1. Open WatchCode on your Apple Watch")
    click.echo("2. Go to Settings")
    click.echo("3. Your setup code will be displayed (format: XXXX-XXXX-XXXX)")
    click.echo()

    # Prompt for setup code
    while True:
        setup_code = click.prompt("Enter your setup code", type=str).strip()

        # Validate format
        if config.validate_token_format(setup_code):
            break
        else:
            click.echo("Invalid setup code format. Expected: XXXX-XXXX-XXXX (12 alphanumeric characters)")

    return setup_code.replace("-", "").upper()


@main.command()
def setup():
    """Interactive setup - configure WatchCode with your setup code."""
    config = Config()
    installer = HooksInstaller()

    click.echo("WatchCode Setup")
    click.echo("=" * 50)
    click.echo()

    # FIRST: Install/update hook scripts to ~/.watchcode/
    click.echo("Installing hook scripts...")
    scripts_result = install_scripts()
    if scripts_result["installed"]:
        click.echo(f"  Installed: {', '.join(scripts_result['installed'])}")
    if scripts_result["updated"]:
        click.echo(f"  Updated: {', '.join(scripts_result['updated'])}")
    if scripts_result["errors"]:
        for err in scripts_result["errors"]:
            click.echo(f"  Error: {err}", err=True)
    if not scripts_result["installed"] and not scripts_result["updated"]:
        click.echo("  Scripts already up to date")
    click.echo()

    # Clean up legacy spam hooks
    cleanup_result = installer.cleanup_legacy_hooks()
    if cleanup_result["removed"] or cleanup_result["fixed"]:
        click.echo("Cleaning up legacy spam hooks...")
        if cleanup_result["removed"]:
            click.echo(f"  Removed: {', '.join(cleanup_result['removed'])}")
        if cleanup_result["fixed"]:
            click.echo(f"  Fixed: {', '.join(cleanup_result['fixed'])}")
        click.echo()

    # Get setup code from user
    clean_token = _setup_manual(config)

    # Save auth token
    config.set_auth_token(clean_token)

    click.echo()
    click.echo(f"Setup code saved: {config.format_token_display(clean_token)}")
    click.echo()

    # Ask to install hooks
    if click.confirm("Install Claude Code hooks?", default=True):
        result = installer.install_hooks()

        if result["installed"]:
            click.echo(f"Installed hooks: {', '.join(result['installed'])}")
        if result["skipped"]:
            click.echo(f"Already installed: {', '.join(result['skipped'])}")

        click.echo()
        click.echo("Hooks installed successfully!")
    else:
        click.echo("Skipped hook installation. Run 'watchcode install-hooks' later.")

    click.echo()

    # Send test notification
    if click.confirm("Send test notification?", default=True):
        try:
            client = RelayClient(config)
            response = client.send_test_notification()

            if response.get("success"):
                click.echo("Test notification sent successfully!")
                click.echo("Check your Apple Watch for the notification.")
            else:
                click.echo(f"Error: {response.get('error', 'Unknown error')}")
        except Exception as e:
            click.echo(f"Error sending test notification: {str(e)}", err=True)

    click.echo()
    click.echo("Setup complete! Claude Code notifications will now be sent to your Watch.")


@main.command()
def install_hooks():
    """Install Claude Code hooks for WatchCode."""
    installer = HooksInstaller()

    # Show current status
    status = installer.get_hook_status()
    click.echo("Current hook status:")
    for hook_type, installed in status.items():
        status_text = "INSTALLED" if installed else "NOT INSTALLED"
        click.echo(f"  {hook_type}: {status_text}")

    click.echo()

    if all(status.values()):
        click.echo("All hooks are already installed.")
        return

    # Install hooks
    if click.confirm("Install WatchCode hooks?", default=True):
        result = installer.install_hooks()

        click.echo()
        if result["installed"]:
            click.echo(f"Installed: {', '.join(result['installed'])}")
        if result["skipped"]:
            click.echo(f"Already installed: {', '.join(result['skipped'])}")

        click.echo()
        click.echo("Hooks installed successfully!")
    else:
        click.echo("Installation cancelled.")


@main.command()
def uninstall_hooks():
    """Uninstall Claude Code hooks for WatchCode."""
    installer = HooksInstaller()

    if click.confirm("Remove all WatchCode hooks?", default=False):
        result = installer.uninstall_hooks()

        if result["removed"]:
            click.echo(f"Removed hooks: {', '.join(result['removed'])}")
            click.echo("Hooks uninstalled successfully!")
        else:
            click.echo("No WatchCode hooks found.")
    else:
        click.echo("Uninstall cancelled.")


@main.command()
def fix_hooks():
    """Remove legacy spam hooks that cause notification spam.

    Removes: SessionStart, SessionEnd, Notification hooks
    Fixes: PreToolUse without matcher (should only trigger for AskUserQuestion)
    """
    installer = HooksInstaller()

    click.echo("Checking for legacy spam hooks...")

    result = installer.cleanup_legacy_hooks()

    if result["removed"]:
        click.echo(f"Removed spam hooks: {', '.join(result['removed'])}")
    if result["fixed"]:
        click.echo(f"Fixed hooks: {', '.join(result['fixed'])}")

    if not result["removed"] and not result["fixed"]:
        click.echo("No spam hooks found - your configuration is clean!")
    else:
        click.echo()
        click.echo("Done! Restart Claude Code for changes to take effect.")


@main.command()
def test():
    """Send a test notification to your Apple Watch."""
    config = Config()

    if not config.is_configured():
        click.echo("WatchCode not configured. Run 'watchcode setup' first.", err=True)
        sys.exit(1)

    try:
        client = RelayClient(config)
        click.echo("Sending test notification...")

        response = client.send_test_notification()

        if response.get("success"):
            click.echo("Test notification sent successfully!")
            click.echo("Check your Apple Watch for the notification.")
        else:
            click.echo(f"Error: {response.get('error', 'Unknown error')}", err=True)
            sys.exit(1)

    except Exception as e:
        click.echo(f"Error: {str(e)}", err=True)
        sys.exit(1)


@main.command()
@click.option("--event", required=True, help="Event type (e.g., stop, permission_request)")
@click.option("--requires-action", is_flag=True, help="Notification requires user action")
@click.option("--message", "message_opt", default=None, help="Custom notification message (overrides HOOK_INPUT)")
@click.option("--notification-id", "notification_id_opt", default=None, help="Unique notification ID for response correlation")
def notify(event: str, requires_action: bool, message_opt: Optional[str], notification_id_opt: Optional[str]):
    """Send notification from Claude Code hook (reads hook data from HOOK_INPUT env or stdin)."""
    import os
    config = Config()

    if not config.is_configured():
        # Silently fail if not configured (hooks shouldn't break Claude Code)
        sys.exit(0)

    try:
        # Read hook input from HOOK_INPUT env var (Claude Code) or stdin (fallback)
        hook_data = {}
        hook_input = os.environ.get("HOOK_INPUT", "")

        if hook_input.strip():
            # Claude Code provides data via HOOK_INPUT environment variable
            try:
                hook_data = json.loads(hook_input)
            except json.JSONDecodeError:
                pass  # Ignore invalid JSON
        elif not sys.stdin.isatty():
            # Fallback to stdin for manual testing
            try:
                stdin_content = sys.stdin.read()
                if stdin_content.strip():
                    hook_data = json.loads(stdin_content)
            except json.JSONDecodeError:
                pass  # Ignore invalid JSON

        # Use CLI option if provided, otherwise use hook data
        message = message_opt if message_opt else hook_data.get("message", f"Claude Code: {event}")
        session_id = hook_data.get("session_id", "unknown")

        # Build metadata
        metadata = {
            "hook_type": event,
            "timestamp": hook_data.get("timestamp"),
        }

        # Add tool-specific metadata for pre_tool_use
        if event == "pre_tool_use":
            metadata["tool_name"] = hook_data.get("tool_name")
            metadata["tool_input"] = hook_data.get("tool_input")

        # Use CLI option if provided, otherwise use hook data
        notification_id = notification_id_opt if notification_id_opt else hook_data.get("notification_id")

        # Send notification
        client = RelayClient(config)
        response = client.send_notification(
            event=event,
            message=message,
            session_id=session_id,
            requires_action=requires_action,
            metadata=metadata,
            notification_id=notification_id
        )

        # Don't print anything on success (hooks should be silent)
        if not response.get("success"):
            # Only log errors to stderr
            click.echo(f"WatchCode error: {response.get('error')}", err=True)

    except Exception as e:
        # Log errors but don't break Claude Code
        click.echo(f"WatchCode error: {str(e)}", err=True)


@main.command()
def status():
    """Show WatchCode configuration status."""
    config = Config()
    installer = HooksInstaller()

    click.echo("WatchCode Status")
    click.echo("=" * 50)

    # Configuration status
    if config.is_configured():
        token = config.get_auth_token()
        click.echo(f"Configuration: CONFIGURED")
        click.echo(f"Setup code: {config.format_token_display(token)}")
        click.echo(f"Token storage: {config.get_storage_location()}")
        click.echo(f"Relay URL: {config.get_relay_url()}")
    else:
        click.echo("Configuration: NOT CONFIGURED")
        click.echo("Run 'watchcode setup' to configure.")

    click.echo()

    # Hook status
    hook_status = installer.get_hook_status()
    click.echo("Installed hooks:")
    for hook_type, installed in hook_status.items():
        status_icon = "✓" if installed else "✗"
        click.echo(f"  {status_icon} {hook_type}")

    if not all(hook_status.values()):
        click.echo()
        click.echo("Run 'watchcode install-hooks' to install missing hooks.")


@main.command()
@click.option("--clear", is_flag=True, help="Unregister device and clear local configuration")
@click.option("--verify", is_flag=True, help="Verify device registration on relay server")
@click.option("--yes", "-y", is_flag=True, help="Skip confirmation prompt for --clear")
def devices(clear: bool, verify: bool, yes: bool):
    """Manage device registration.

    Shows current device configuration by default.
    Use --clear to unregister and remove local config.
    Use --verify to check if device is registered on relay server.
    """
    config = Config()

    if not config.is_configured():
        click.echo("WatchCode not configured. Run 'watchcode setup' first.", err=True)
        sys.exit(1)

    token = config.get_auth_token()

    if verify:
        # Check if token is valid on relay
        click.echo(f"Verifying device {config.format_token_display(token)}...")
        client = RelayClient(config)
        is_valid = client.verify_device()
        if is_valid:
            click.echo(f"Device is registered and active on relay server.")
        else:
            click.echo(f"Device is NOT registered on relay server.", err=True)
            click.echo("Run 'watchcode setup' to re-register.")
            sys.exit(1)
        return

    if clear:
        if not yes:
            click.echo(f"This will unregister device {config.format_token_display(token)} and clear local config.")
            if not click.confirm("Continue?"):
                click.echo("Cancelled.")
                return

        # Unregister from relay
        click.echo("Unregistering from relay server...", nl=False)
        client = RelayClient(config)
        try:
            client.unregister_device()
            click.echo(" Done")
        except Exception as e:
            click.echo(f" Warning: {e}", err=True)
            click.echo("  (Continuing with local cleanup)")

        # Clear local config
        click.echo("Clearing local configuration...", nl=False)
        config.delete_auth_token()
        click.echo(" Done")

        click.echo()
        click.echo("Device cleared. Run 'watchcode setup' to configure a new device.")
        return

    # Default: show device info
    click.echo("Device Configuration")
    click.echo("=" * 50)
    click.echo(f"Setup code: {config.format_token_display(token)}")
    click.echo(f"Storage: {config.get_storage_location()}")
    click.echo(f"Relay: {config.get_relay_url()}")
    click.echo()
    click.echo("Use --verify to check relay registration status.")
    click.echo("Use --clear to unregister and remove configuration.")


@main.command("update-scripts")
def update_scripts():
    """Update hook scripts in ~/.watchcode/ to latest version."""
    click.echo("Updating hook scripts...")
    result = install_scripts()

    if result["installed"]:
        click.echo(f"Installed: {', '.join(result['installed'])}")
    if result["updated"]:
        click.echo(f"Updated: {', '.join(result['updated'])}")
    if result["errors"]:
        for err in result["errors"]:
            click.echo(f"Error: {err}", err=True)

    if not result["installed"] and not result["updated"] and not result["errors"]:
        click.echo("All scripts already up to date.")
    else:
        click.echo("Done!")


if __name__ == "__main__":
    main()
