#!/usr/bin/env python3
"""
Background listener that polls iCloud KV Store for action responses from Watch/iPhone.
Outputs JSON to stdout for Claude Code to consume.

This script runs in the background during a Claude Code session, waiting for user responses
to permission requests or other actionable notifications.
"""

import time
import json
import subprocess
import argparse
import sys
from datetime import datetime
from pathlib import Path


def log(message, level="INFO"):
    """Print log message to stderr (so stdout stays clean for JSON output)"""
    timestamp = datetime.now().strftime("%H:%M:%S")
    print(f"[{timestamp}] [{level}] {message}", file=sys.stderr)


def read_icloud_response(session_id):
    """
    Read response from iCloud KV Store.

    Returns:
        dict or None: Response data if found, None otherwise
    """
    key = f"watchcode.response.{session_id}"

    log(f"Checking iCloud for response key: {key}")

    try:
        result = subprocess.run(
            ['defaults', 'read', 'com.apple.nsubiquitouskeyvaluestore', key],
            capture_output=True,
            text=True,
            timeout=5
        )

        if result.returncode == 0 and result.stdout.strip():
            data = result.stdout.strip()
            log(f"✓ Found response in iCloud")

            # Data might be base64 encoded or plist format
            # Try to parse as JSON directly (from Data object)
            try:
                # If it's a plist dictionary, defaults will output XML
                # We need to parse it properly
                import plistlib

                # Try parsing as plist first
                try:
                    plist_data = data.encode('utf-8')
                    response_dict = plistlib.loads(plist_data)
                    log(f"✓ Parsed as plist: {response_dict}")
                    return response_dict
                except:
                    # If that fails, try as plain JSON
                    response_dict = json.loads(data)
                    log(f"✓ Parsed as JSON: {response_dict}")
                    return response_dict

            except json.JSONDecodeError as e:
                log(f"❌ Failed to parse response data: {e}", "ERROR")
                log(f"   Raw data: {data[:100]}...", "DEBUG")
                return None
        else:
            log(f"   No response found yet")
            return None

    except subprocess.TimeoutExpired:
        log("⚠️ iCloud read timeout", "WARN")
        return None
    except Exception as e:
        log(f"⚠️ Error reading iCloud: {e}", "WARN")
        return None


def clear_icloud_response(session_id):
    """Clear response from iCloud KV Store after successfully reading it"""
    key = f"watchcode.response.{session_id}"

    log(f"Clearing iCloud key: {key}")

    try:
        result = subprocess.run(
            ['defaults', 'delete', 'com.apple.nsubiquitouskeyvaluestore', key],
            capture_output=True,
            timeout=5
        )

        if result.returncode == 0:
            log("✓ Response cleared from iCloud")
        else:
            log(f"⚠️ Failed to clear response (may not exist)", "WARN")

    except Exception as e:
        log(f"⚠️ Error clearing iCloud: {e}", "WARN")


def output_response(response, session_id):
    """Output response as JSON to stdout (for Claude Code to read)"""
    output = {
        "action": response.get("action"),
        "timestamp": response.get("timestamp"),
        "sessionId": session_id,
        "notificationId": response.get("notificationId")
    }

    # Output to stdout (Claude Code reads this)
    print(json.dumps(output), flush=True)
    log(f"✓ Response sent to Claude Code: {output['action']}")


def main():
    parser = argparse.ArgumentParser(
        description='Listen for WatchCode action responses from iCloud',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
This script polls iCloud KV Store for user responses to actionable notifications.
It runs in the background during a Claude Code session.

Output:
  JSON object to stdout when response is received or timeout occurs.
  Logs to stderr for debugging.

Exit codes:
  0 - Response received and sent to Claude Code
  1 - Timeout (default action applied)
  2 - Error occurred
        """
    )

    parser.add_argument('--session-id', required=True,
                       help='Claude Code session ID to listen for')
    parser.add_argument('--timeout', type=int, default=60,
                       help='Timeout in seconds (default: 60)')
    parser.add_argument('--poll-interval', type=int, default=1,
                       help='Initial poll interval in seconds (default: 1)')
    parser.add_argument('--verbose', action='store_true',
                       help='Enable verbose logging')

    args = parser.parse_args()

    log("=" * 60)
    log("WatchCode Response Listener")
    log(f"Session ID: {args.session_id}")
    log(f"Timeout: {args.timeout}s")
    log(f"Poll interval: {args.poll_interval}s (with exponential backoff)")
    log("=" * 60)

    start_time = time.time()
    poll_interval = args.poll_interval
    max_interval = 8  # Max 8 seconds between polls
    attempts = 0

    while time.time() - start_time < args.timeout:
        attempts += 1
        elapsed = time.time() - start_time

        if args.verbose:
            log(f"Attempt {attempts} (elapsed: {elapsed:.1f}s, interval: {poll_interval}s)")

        # Check for response
        response = read_icloud_response(args.session_id)

        if response:
            log("✓ Response received!")

            # Validate response structure
            if "action" not in response:
                log("❌ Response missing 'action' field", "ERROR")
                return 2

            # Output to Claude Code
            output_response(response, args.session_id)

            # Clear from iCloud
            clear_icloud_response(args.session_id)

            log("✓ Complete")
            return 0

        # Exponential backoff (1s, 2s, 4s, 8s max)
        time.sleep(poll_interval)
        poll_interval = min(poll_interval * 2, max_interval)

    # Timeout reached - default to deny
    log(f"⏱️ Timeout reached after {args.timeout}s", "WARN")
    log("   Defaulting to 'deny' action")

    timeout_response = {
        "action": "deny",
        "timestamp": datetime.now().isoformat(),
        "sessionId": args.session_id,
        "timeout": True
    }

    print(json.dumps(timeout_response), flush=True)
    log("✓ Timeout response sent to Claude Code")

    return 1


if __name__ == "__main__":
    try:
        exit_code = main()
        sys.exit(exit_code)
    except KeyboardInterrupt:
        log("\n⚠️ Interrupted by user", "WARN")
        sys.exit(130)  # Standard exit code for SIGINT
    except Exception as e:
        log(f"❌ Unexpected error: {e}", "ERROR")
        import traceback
        traceback.print_exc(file=sys.stderr)
        sys.exit(2)
