# watchlog-python

🔗 **Website**: [https://watchlog.io](https://watchlog.io)

A lightweight, non-blocking Python client for sending custom metrics to the [Watchlog](https://watchlog.io/) monitoring platform.

## 🚀 Installation

Install the package using pip:

```bash
pip install watchlog-python
```

## 📦 Basic Usage

### 1. Import the Watchlog class

```python
from watchlog import Watchlog
```

### 2. Create an instance of Watchlog

```python
watchlog_instance = Watchlog()
```

### 3. Send metrics using simple method calls

```python
# Increment a counter
watchlog_instance.increment('page_views', 10)

# Decrement a counter
watchlog_instance.decrement('items_in_cart', 2)

# Set a gauge value
watchlog_instance.gauge('current_temperature', 22.5)

# Set a percentage value (0 to 100)
watchlog_instance.percentage('completion_rate', 85)

# Log system byte metric (e.g., memory usage in bytes)
watchlog_instance.systembyte('memory_usage', 1024)
```

All operations are performed **asynchronously and silently**, ensuring zero interruption to your main application.

## 🐳 Docker Setup

When running your Python app in Docker, you can specify the agent URL explicitly:

```python
from watchlog import Watchlog

# Create client with explicit agent URL for Docker
watchlog_instance = Watchlog(url='http://watchlog-agent:3774')

watchlog_instance.increment('page_views', 1)
```

**Docker Compose Example:**
```yaml
version: '3.8'

services:
  watchlog-agent:
    image: watchlog/agent:latest
    container_name: watchlog-agent
    ports:
      - "3774:3774"
    environment:
      - WATCHLOG_APIKEY=your-api-key
      - WATCHLOG_SERVER=https://log.watchlog.ir
    networks:
      - app-network

  python-app:
    build: .
    container_name: python-app
    ports:
      - "8000:8000"
    depends_on:
      - watchlog-agent
    networks:
      - app-network

networks:
  app-network:
    driver: bridge
```

**Docker Run Example:**
```bash
# 1. Create network
docker network create app-network

# 2. Run Watchlog Agent
docker run -d \
  --name watchlog-agent \
  --network app-network \
  -p 3774:3774 \
  -e WATCHLOG_APIKEY="your-api-key" \
  -e WATCHLOG_SERVER="https://log.watchlog.ir" \
  watchlog/agent:latest

# 3. Run Python app (make sure your code uses Watchlog(url='http://watchlog-agent:3774'))
docker run -d \
  --name python-app \
  --network app-network \
  -p 8000:8000 \
  my-python-app
```

## 🌐 Example Usage in a Django View

```python
# views.py
from django.http import HttpResponse
from watchlog import Watchlog

watchlog_instance = Watchlog()

def some_view(request):
    watchlog_instance.increment('view_hits')
    return HttpResponse("This is a view that increments a metric.")
```

## 🔍 Environment Detection

The package automatically detects the runtime environment:

* **Local / non-K8s**: `http://127.0.0.1:3774`
* **Kubernetes**: `http://watchlog-node-agent.monitoring.svc.cluster.local:3774`

**Manual Override:** You can override the endpoint by passing `url` parameter to the constructor:

```python
watchlog_instance = Watchlog(url='http://watchlog-agent:3774')  # Custom agent URL
```

**Important Notes:**
- When using Docker, use the container name as the hostname (e.g., `watchlog-agent`)
- Both containers must be on the same Docker network
- The agent must be running before your app starts
- If `url` is not provided, auto-detection will be used (local or Kubernetes)

## ✅ Features

- ⚡️ Non-blocking & thread-based HTTP request
- 🛡️ No logging, printing, or exception leaks
- 🔐 Safe and isolated from your main application flow
- 🧩 Easy to integrate with any Python web framework

## 📄 License

MIT License