"""
The latest version of this package is available at:
<http://github.com/jantman/webhook2lambda2sqs>

################################################################################
Copyright 2016 Jason Antman <jason@jasonantman.com> <http://www.jasonantman.com>

    This file is part of webhook2lambda2sqs, also known as webhook2lambda2sqs.

    webhook2lambda2sqs is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    webhook2lambda2sqs is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with webhook2lambda2sqs.  If not, see <http://www.gnu.org/licenses/>.

The Copyright and Authors attributions contained herein may not be removed or
otherwise altered, except to add the Author attribution of a contributor to
this work. (Additional Terms pursuant to Section 7b of the AGPL v3)
################################################################################
While not legally required, I sincerely request that anyone who finds
bugs please submit them at <https://github.com/jantman/webhook2lambda2sqs> or
to me via email, and that you send any contributions or improvements
either as a pull request on GitHub, or to me via email.
################################################################################

AUTHORS:
Jason Antman <jason@jasonantman.com> <http://www.jasonantman.com>
################################################################################
"""

import logging
from inspect import getsourcelines
from pprint import pformat
from webhook2lambda2sqs import lambda_func
from webhook2lambda2sqs.version import VERSION, PROJECT_URL

logger = logging.getLogger(__name__)


class LambdaFuncGenerator(object):

    def __init__(self, config):
        """
        Initialize the Lambda function code generator.

        :param config: program configuration
        :type config: :py:class:`~.Config`
        """
        self.config = config

    def _get_source(self):
        """
        Get the lambda function source template. Strip the leading docstring.
        Note that it's a real module in this project so we can test it.

        :return: function source code, with leading docstring stripped.
        :rtype: str
        """
        logger.debug('Getting module source for webhook2lambda2sqs.lambda_func')
        orig = getsourcelines(lambda_func)
        src = ''
        in_docstr = False
        have_docstr = False
        for line in orig[0]:
            if line.strip() == '"""' and not in_docstr and not have_docstr:
                in_docstr = True
                continue
            if line.strip() == '"""' and in_docstr:
                in_docstr = False
                have_docstr = True
                continue
            if not in_docstr:
                src += line
        return src

    @property
    def _docstring(self):
        """
        Generate a docstring for the generated source file.

        :return: new docstring
        :rtype: str
        """
        s = '"""' + "\n"
        s += "webhook2lambda2sqs generated function source\n"
        s += "this code was generated by webhook2lambda2sqs v%s\n" % VERSION
        s += "<%s>\n" % PROJECT_URL
        s += "this project is licensed under the AGPLv3 open source license.\n"
        s += "\n"
        s += "DO NOT MODIFY this function; modifications should be made by "
        s += "re-running\n"
        s += "the project with an updated configuration.\n"
        s += '"""' + "\n"
        return s

    @property
    def _config_src(self):
        """
        Generate the configuration source snippet to insert in the function
        source.

        :return: configuration code snippet to put in function source.
        :rtype: str
        """
        return pformat(self.config.get('endpoints')).replace("u'", "'")

    def generate(self):
        """
        Generate Lambda function source; return it as a string.

        :rtype: str
        :returns: lambda function source
        """
        s = self._docstring
        s += self._get_source().replace(
            'endpoints = {}',
            'endpoints = ' + self._config_src
        ).replace(
            'logger.setLevel(logging.INFO)',
            'logger.setLevel(logging.%s)' % self.config.logging_level
        )
        return s
