"""
The latest version of this package is available at:
<http://github.com/jantman/webhook2lambda2sqs>

################################################################################
Copyright 2016 Jason Antman <jason@jasonantman.com> <http://www.jasonantman.com>

    This file is part of webhook2lambda2sqs, also known as webhook2lambda2sqs.

    webhook2lambda2sqs is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    webhook2lambda2sqs is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with webhook2lambda2sqs.  If not, see <http://www.gnu.org/licenses/>.

The Copyright and Authors attributions contained herein may not be removed or
otherwise altered, except to add the Author attribution of a contributor to
this work. (Additional Terms pursuant to Section 7b of the AGPL v3)
################################################################################
While not legally required, I sincerely request that anyone who finds
bugs please submit them at <https://github.com/jantman/webhook2lambda2sqs> or
to me via email, and that you send any contributions or improvements
either as a pull request on GitHub, or to me via email.
################################################################################

AUTHORS:
Jason Antman <jason@jasonantman.com> <http://www.jasonantman.com>
################################################################################
"""
import sys

from webhook2lambda2sqs.func_generator import LambdaFuncGenerator
from webhook2lambda2sqs.version import VERSION, PROJECT_URL

# https://code.google.com/p/mock/issues/detail?id=249
# py>=3.4 should use unittest.mock not the mock package on pypi
if (
        sys.version_info[0] < 3 or
        sys.version_info[0] == 3 and sys.version_info[1] < 4
):
    from mock import patch, call, Mock, DEFAULT, PropertyMock  # noqa
else:
    from unittest.mock import patch, call, Mock, DEFAULT, PropertyMock  # noqa

pbm = 'webhook2lambda2sqs.func_generator'
pb = '%s.LambdaFuncGenerator' % pbm


class TestLambdaFuncGenerator(object):

    def setup(self):
        self.conf = {}

        def se_get(k):
            return self.conf.get(k, None)

        config = Mock()
        config.get.side_effect = se_get
        type(config).func_name = 'webhook2lambda2sqs'
        type(config).logging_level = 'WARN'
        self.cls = LambdaFuncGenerator(config)

    def test_init(self):
        m = Mock()
        cls = LambdaFuncGenerator(m)
        assert cls.config == m

    def test_get_source(self):
        src = [
            '"""' + "\n",
            "docstring\n",
            "\n",
            "foo\n",
            '"""' + "\n"
        ]
        expected = [
            "\n",
            "import foo\n",
            "\n",
            "def foo(bar):\n",
            '    """' + "\n",
            "    foo\n",
            "    \n",
            "    bar\n",
            '    """' + "\n",
            "    return 2\n"
        ]
        src = src + expected
        expected_lines = ''.join(expected)
        with patch('%s.getsourcelines' % pbm) as mock_gsl:
            mock_gsl.return_value = (src, 0)
            res = self.cls._get_source()
        assert res == expected_lines

    def test_docstring(self):
        s = '"""' + "\n"
        s += "webhook2lambda2sqs generated function source\n"
        s += "this code was generated by webhook2lambda2sqs v%s\n" % VERSION
        s += "<%s>\n" % PROJECT_URL
        s += "this project is licensed under the AGPLv3 open source license.\n"
        s += "\n"
        s += "DO NOT MODIFY this function; modifications should be made by "
        s += "re-running\n"
        s += "the project with an updated configuration.\n"
        s += '"""' + "\n"
        assert self.cls._docstring == s

    def test_config_src(self):
        conf = {'foo': 'bar', 'baz': {'blam': 2}}
        self.conf['endpoints'] = conf
        assert self.cls._config_src == "{'baz': {'blam': 2}, 'foo': 'bar'}"

    def test_generate(self):
        src = "import foo\n\n"
        src += "logger.setLevel(logging.INFO)\n\n"
        src += "endpoints = {}\n"
        src += "\ndef foo():\n"
        src += "    return 1\n"
        expected = "mydocstring\n"
        expected += "import foo\n\n"
        expected += "logger.setLevel(logging.WARN)\n\n"
        expected += "endpoints = myconfig\n"
        expected += "\n\ndef foo():\n"
        expected += "    return 1\n"
        with patch('%s._config_src' % pb, new_callable=PropertyMock) as mock_cs:
            with patch('%s._docstring' % pb, new_callable=PropertyMock) as m_ds:
                with patch('%s._get_source' % pb) as mock_get_src:
                    mock_cs.return_value = "myconfig\n"
                    m_ds.return_value = "mydocstring\n"
                    mock_get_src.return_value = src
                    res = self.cls.generate()
        print('### res ###')
        print(res)
        print('### expected ###')
        print(expected)
        assert res == expected
