# WeDo2 Python Library

A Python library for controlling the **LEGO WeDo 2.0 Smart Hub** using Bluetooth Low Energy (BLE).  
This project uses the `bleak` library to communicate with motors, sensors, lights and sound functions on the WeDo 2.0 hub.

---

## Features

- Connect to the WeDo 2.0 hub by name  
- Control motors (run, stop, brake)  
- Read tilt sensor  
- Read distance sensor  
- Control the LED light (preset colors + RGB)  
- Play notes, frequencies and melodies  
- Read battery level  
- Detect connected devices automatically  
- Turn off or disconnect the hub  

---

## Installation

```bash
pip install wedo2-python
```

---

## Basic Usage

```python
from wedo2 import WeDo2

hub = WeDo2('WeDo 2.0')

print('Hub name:', hub.name())
print('Battery:', hub.get_battery_level())

hub.disconnect()
```

---

# Class: WeDo2

## Connecting to the hub

```python
hub = WeDo2('WeDo 2.0')
```

## Rename the hub

```python
hub.name('MyHubName')
```

## Read the name of the hub

```python
name = hub.name()
print(name) -> "MyHubName"
```

## Disconnect the hub

```python
hub.disconnect()
```

## Turn off the hub

```python
hub.shut_off()
```

## Read battery level

```python
level = hub.get_battery_level()
print(level)
```

---

# Motor

### Run the motor

```python
hub.motor.run(50, port=1)   # Speed from -100 to 100
```

### Stop the motor

```python
hub.motor.stop(port=1)
```

### Brake the motor

```python
hub.motor.brake(port=1)
```

### Set motor port explicitly

```python
hub.motor.set_port(2)
```

---

# Distance Sensor

### Read distance (0–10 cm)

```python
d = hub.distance_sensor.distance(port=1)
print('Distance:', d)
```

### Read number of triggers

```python
t = hub.distance_sensor.times(port=1)
print('Triggered:', t)
```

### Set sensor port

```python
hub.distance_sensor.set_port(2)
```

---

# Tilt Sensor

### Read tilt direction

```python
position = hub.tilt_sensor.tilt(port=1)
print(position)
```

### Set tilt sensor port

```python
hub.tilt_sensor.set_port(2)
```

---

# Light

### Turn on a preset color

Available colors:  
` off, pink, purple, blue, sky blue, teal, green, yellow, orange, red, white `

```python
hub.light.on('red')
```

### Use integer color

```python
hub.light.on(3)   # blue
```

### RGB color

```python
hub.light.rgb(255, 120, 0)  # orange-ish
```

### Turn off

```python
hub.light.off()
```

---

# Sound

### Play a musical note

Format: ```C4```, ```D#5```, ```Gb3```

```python
hub.sound.note('C4', 500)
```

### Play a beep at frequency

```python
hub.sound.beeb(800, 300)
```

### Play a melody

```python
hub.sound.melody([
    ('C4', 300),
    ('E4', 300),
    ('G4', 500),
    '.',
    ('C5', 800)
])
```

---

# Raw Commands (Advanced)

### Raw output

```python
hub.write_raw_output(b'\x06\x04\x01\x09')
```

### Raw input

```python
hub.write_raw_input(b'\x01\x02\x06\x17\x00\x01\x00\x00\x00\x02\x01')
```

---

# License

MIT License

---

# Acknowledgements

- LEGO for the WeDo 2.0 system  
- https://ofalcao.pt/blog/series/wedo-2-0-reverse-engineering for information about WeDo2 uuids and commands
- bleak library for BLE communication  

---