from pathlib import Path
from typing import List, TYPE_CHECKING

import aiofiles

from ..response import Response

if TYPE_CHECKING:
    from ..API import GreenApi


class Groups:
    def __init__(self, api: "GreenApi"):
        self.api = api

    def createGroup(self, groupName: str, chatIds: List[str]) -> Response:
        """
        The method is aimed for creating a group chat.

        https://green-api.com/en/docs/api/groups/CreateGroup/
        """

        request_body = self.__handle_parameters(locals())

        return self.api.request(
            "POST", (
                "{{host}}/waInstance{{idInstance}}/"
                "createGroup/{{apiTokenInstance}}"
            ), request_body
        )


    async def createGroupAsync(self, groupName: str, chatIds: List[str]) -> Response:
        request_body = self.__handle_parameters(locals())

        return await self.api.requestAsync(
            "POST",
            "{{host}}/waInstance{{idInstance}}/createGroup/{{apiTokenInstance}}",
            request_body
        )

    def updateGroupName(self, groupId: str, groupName: str) -> Response:
        """
        The method changes a group chat name.

        https://green-api.com/en/docs/api/groups/UpdateGroupName/
        """

        request_body = self.__handle_parameters(locals())

        return self.api.request(
            "POST", (
                "{{host}}/waInstance{{idInstance}}/"
                "updateGroupName/{{apiTokenInstance}}"
            ), request_body
        )

    async def updateGroupNameAsync(self, groupId: str, groupName: str) -> Response:
        request_body = self.__handle_parameters(locals())

        return await self.api.requestAsync(
            "POST",
            "{{host}}/waInstance{{idInstance}}/updateGroupName/{{apiTokenInstance}}",
            request_body
        )

    def getGroupData(self, groupId: str) -> Response:
        """
        The method gets group chat data.

        https://green-api.com/en/docs/api/groups/GetGroupData/
        """

        request_body = self.__handle_parameters(locals())

        return self.api.request(
            "POST", (
                "{{host}}/waInstance{{idInstance}}/"
                "getGroupData/{{apiTokenInstance}}"
            ), request_body
        )

    async def getGroupDataAsync(self, groupId: str) -> Response:
        request_body = self.__handle_parameters(locals())

        return await self.api.requestAsync(
            "POST",
            "{{host}}/waInstance{{idInstance}}/getGroupData/{{apiTokenInstance}}",
            request_body
        )

    def addGroupParticipant(
            self, groupId: str, participantChatId: str
    ) -> Response:
        """
        The method adds a participant to a group chat.

        https://green-api.com/en/docs/api/groups/AddGroupParticipant/
        """

        request_body = self.__handle_parameters(locals())

        return self.api.request(
            "POST", (
                "{{host}}/waInstance{{idInstance}}/"
                "addGroupParticipant/{{apiTokenInstance}}"
            ), request_body
        )

    async def addGroupParticipantAsync(
            self, groupId: str, participantChatId: str
    ) -> Response:
        request_body = self.__handle_parameters(locals())

        return await self.api.requestAsync(
            "POST",
            "{{host}}/waInstance{{idInstance}}/addGroupParticipant/{{apiTokenInstance}}",
            request_body
        )

    def removeGroupParticipant(
            self, groupId: str, participantChatId: str
    ) -> Response:
        """
        The method removes a participant from a group chat.

        https://green-api.com/en/docs/api/groups/RemoveGroupParticipant/
        """

        request_body = self.__handle_parameters(locals())

        return self.api.request(
            "POST", (
                "{{host}}/waInstance{{idInstance}}/"
                "removeGroupParticipant/{{apiTokenInstance}}"
            ), request_body
        )

    async def removeGroupParticipantAsync(
            self, groupId: str, participantChatId: str
    ) -> Response:
        request_body = self.__handle_parameters(locals())

        return await self.api.requestAsync(
            "POST",
            "{{host}}/waInstance{{idInstance}}/removeGroupParticipant/{{apiTokenInstance}}",
            request_body
        )

    def setGroupAdmin(self, groupId: str, participantChatId: str) -> Response:
        """
        The method sets a group chat participant as an administrator.

        https://green-api.com/en/docs/api/groups/SetGroupAdmin/
        """

        request_body = self.__handle_parameters(locals())

        return self.api.request(
            "POST", (
                "{{host}}/waInstance{{idInstance}}/"
                "setGroupAdmin/{{apiTokenInstance}}"
            ), request_body
        )

    async def setGroupAdminAsync(self, groupId: str, participantChatId: str) -> Response:
        request_body = self.__handle_parameters(locals())

        return await self.api.requestAsync(
            "POST",
            "{{host}}/waInstance{{idInstance}}/setGroupAdmin/{{apiTokenInstance}}",
            request_body
        )

    def removeAdmin(self, groupId: str, participantChatId: str) -> Response:
        """
        The method removes a participant from group chat administration
        rights.

        https://green-api.com/en/docs/api/groups/RemoveAdmin/
        """

        request_body = self.__handle_parameters(locals())

        return self.api.request(
            "POST", (
                "{{host}}/waInstance{{idInstance}}/"
                "removeAdmin/{{apiTokenInstance}}"
            ), request_body
        )

    async def removeAdminAsync(self, groupId: str, participantChatId: str) -> Response:
        request_body = self.__handle_parameters(locals())

        return await self.api.requestAsync(
            "POST",
            "{{host}}/waInstance{{idInstance}}/removeAdmin/{{apiTokenInstance}}",
            request_body
        )

    def setGroupPicture(self, groupId: str, path: str) -> Response:
        """
        The method sets a group picture.

        https://green-api.com/en/docs/api/groups/SetGroupPicture/
        """

        request_body = self.__handle_parameters(locals())

        file_name = Path(path).name
        files = {"file": (file_name, open(path, "rb"), "image/jpeg")}

        request_body.pop("path")

        return self.api.request(
            "POST", (
                "{{host}}/waInstance{{idInstance}}/"
                "setGroupPicture/{{apiTokenInstance}}"
            ), request_body, files
        )

    async def setGroupPictureAsync(self, groupId: str, path: str) -> Response:
        request_body = self.__handle_parameters(locals())

        request_body.pop("path")
        file_name = Path(path).name

        async with aiofiles.open(path, "rb") as file:
            file_data = await file.read()
            files = {"file": (file_name, file_data, "image/jpeg")}

        return await self.api.requestAsync(
            "POST",
            "{{host}}/waInstance{{idInstance}}/setGroupPicture/{{apiTokenInstance}}",
            request_body, 
            files=files
        )

    def leaveGroup(self, groupId: str) -> Response:
        """
        The method makes the current account user leave the group chat.

        https://green-api.com/en/docs/api/groups/LeaveGroup/
        """

        request_body = self.__handle_parameters(locals())

        return self.api.request(
            "POST", (
                "{{host}}/waInstance{{idInstance}}/"
                "leaveGroup/{{apiTokenInstance}}"
            ), request_body
        )

    async def leaveGroupAsync(self, groupId: str) -> Response:
        request_body = self.__handle_parameters(locals())

        return await self.api.requestAsync(
            "POST",
            "{{host}}/waInstance{{idInstance}}/leaveGroup/{{apiTokenInstance}}",
            request_body
        )

    @classmethod
    def __handle_parameters(cls, parameters: dict) -> dict:
        handled_parameters = parameters.copy()

        handled_parameters.pop("self")

        for key, value in parameters.items():
            if value is None:
                handled_parameters.pop(key)

        return handled_parameters