import os
import shlex
from dataclasses import dataclass
from pathlib import Path
from typing import Final

from click.testing import CliRunner

from workstack.cli.commands.create import create
from workstack.cli.commands.down import down_cmd
from workstack.cli.commands.jump import jump_cmd
from workstack.cli.commands.prepare_cwd_recovery import generate_recovery_script
from workstack.cli.commands.switch import switch_cmd
from workstack.cli.commands.up import up_cmd
from workstack.cli.debug import debug_log
from workstack.cli.shell_utils import (
    STALE_SCRIPT_MAX_AGE_SECONDS,
    cleanup_stale_scripts,
    write_script_to_temp,
)
from workstack.core.context import create_context

PASSTHROUGH_MARKER: Final[str] = "__WORKSTACK_PASSTHROUGH__"
PASSTHROUGH_COMMANDS: Final[set[str]] = {"sync"}


@dataclass(frozen=True)
class ShellIntegrationResult:
    """Result returned by shell integration handlers."""

    passthrough: bool
    script: str | None
    exit_code: int


def _invoke_hidden_command(command_name: str, args: tuple[str, ...]) -> ShellIntegrationResult:
    """Invoke a command with --script flag for shell integration.

    If args contain help flags or explicit --script, passthrough to regular command.
    Otherwise, add --script flag and capture the activation script.
    """
    # Check if help flags or --script are present - these should pass through
    if "-h" in args or "--help" in args or "--script" in args:
        return ShellIntegrationResult(passthrough=True, script=None, exit_code=0)

    # Map command names to their Click commands
    command_map = {
        "switch": switch_cmd,
        "create": create,
        "jump": jump_cmd,
        "up": up_cmd,
        "down": down_cmd,
    }

    command = command_map.get(command_name)
    if command is None:
        if command_name in PASSTHROUGH_COMMANDS:
            return _build_passthrough_script(command_name, args)
        return ShellIntegrationResult(passthrough=True, script=None, exit_code=0)

    # Add --script flag to get activation script
    script_args = list(args) + ["--script"]

    debug_log(f"Handler: Invoking {command_name} with args: {script_args}")

    # Clean up stale scripts before running (opportunistic cleanup)
    cleanup_stale_scripts(max_age_seconds=STALE_SCRIPT_MAX_AGE_SECONDS)

    runner = CliRunner()
    result = runner.invoke(
        command,
        script_args,
        obj=create_context(dry_run=False),
        standalone_mode=False,
    )

    exit_code = int(result.exit_code)

    # If command failed, passthrough to show proper error
    if exit_code != 0:
        return ShellIntegrationResult(passthrough=True, script=None, exit_code=exit_code)

    # Output is now a file path, not script content
    script_path = result.output.strip() if result.output else None

    debug_log(f"Handler: Got script_path={script_path}, exit_code={exit_code}")
    if script_path:
        script_exists = Path(script_path).exists()
        debug_log(f"Handler: Script exists? {script_exists}")

    return ShellIntegrationResult(passthrough=False, script=script_path, exit_code=exit_code)


def handle_shell_request(args: tuple[str, ...]) -> ShellIntegrationResult:
    """Dispatch shell integration handling based on the original CLI invocation."""
    if not args:
        return ShellIntegrationResult(passthrough=True, script=None, exit_code=0)

    command_name = args[0]
    command_args = tuple(args[1:])

    return _invoke_hidden_command(command_name, command_args)


def _build_passthrough_script(command_name: str, args: tuple[str, ...]) -> ShellIntegrationResult:
    """Create a passthrough script tailored for the caller's shell."""
    shell_name = os.environ.get("WORKSTACK_SHELL", "bash").lower()
    ctx = create_context(dry_run=False)
    recovery_path = generate_recovery_script(ctx)

    script_content = _render_passthrough_script(shell_name, command_name, args, recovery_path)
    script_path = write_script_to_temp(
        script_content,
        command_name=f"{command_name}-passthrough",
        comment="generated by __shell passthrough handler",
    )
    return ShellIntegrationResult(passthrough=False, script=str(script_path), exit_code=0)


def _render_passthrough_script(
    shell_name: str,
    command_name: str,
    args: tuple[str, ...],
    recovery_path: Path | None,
) -> str:
    """Render shell-specific script that runs the command and performs recovery."""
    if shell_name == "fish":
        return _render_fish_passthrough(command_name, args, recovery_path)
    return _render_posix_passthrough(command_name, args, recovery_path)


def _render_posix_passthrough(
    command_name: str,
    args: tuple[str, ...],
    recovery_path: Path | None,
) -> str:
    quoted_args = " ".join(shlex.quote(part) for part in (command_name, *args))
    recovery_literal = shlex.quote(str(recovery_path)) if recovery_path is not None else "''"
    lines = [
        f"command workstack {quoted_args}",
        "__workstack_exit=$?",
        f"__workstack_recovery={recovery_literal}",
        'if [ -n "$__workstack_recovery" ] && [ -f "$__workstack_recovery" ]; then',
        '  if [ ! -d "$PWD" ]; then',
        '    . "$__workstack_recovery"',
        "  fi",
        '  if [ -z "$WORKSTACK_KEEP_SCRIPTS" ]; then',
        '    rm -f "$__workstack_recovery"',
        "  fi",
        "fi",
        "return $__workstack_exit",
    ]
    return "\n".join(lines) + "\n"


def _quote_fish(arg: str) -> str:
    if not arg:
        return '""'

    escape_map = {
        "\\": "\\\\",
        '"': '\\"',
        "$": "\\$",
        "`": "\\`",
        "~": "\\~",
        "*": "\\*",
        "?": "\\?",
        "{": "\\{",
        "}": "\\}",
        "[": "\\[",
        "]": "\\]",
        "(": "\\(",
        ")": "\\)",
        "<": "\\<",
        ">": "\\>",
        "|": "\\|",
        ";": "\\;",
        "&": "\\&",
    }
    escaped_parts: list[str] = []
    for char in arg:
        if char == "\n":
            escaped_parts.append("\\n")
            continue
        if char == "\t":
            escaped_parts.append("\\t")
            continue
        escaped_parts.append(escape_map.get(char, char))

    escaped = "".join(escaped_parts)
    return f'"{escaped}"'


def _render_fish_passthrough(
    command_name: str,
    args: tuple[str, ...],
    recovery_path: Path | None,
) -> str:
    command_parts = " ".join(_quote_fish(part) for part in (command_name, *args))
    recovery_literal = _quote_fish(str(recovery_path)) if recovery_path is not None else '""'
    lines = [
        f"command workstack {command_parts}",
        "set __workstack_exit $status",
        f"set __workstack_recovery {recovery_literal}",
        'if test -n "$__workstack_recovery"',
        '    if test -f "$__workstack_recovery"',
        '        if not test -d "$PWD"',
        '            source "$__workstack_recovery"',
        "        end",
        "        if not set -q WORKSTACK_KEEP_SCRIPTS",
        '            rm -f "$__workstack_recovery"',
        "        end",
        "    end",
        "end",
        "return $__workstack_exit",
    ]
    return "\n".join(lines) + "\n"
