from .exceptions import ConnectionClosed

import websockets.client
import websockets.exceptions
import socket
import asyncio
import json

class Connection():
    def __init__(self):
        self.safe = True
        self._conn = None
        self._lock = asyncio.Lock()

    async def connect(self, url):
        try:
            self._conn = await websockets.client.connect(url, close_timeout=0, max_size=None)

        except socket.gaierror:
            raise ConnectionClosed("Hostname cannot be resolved")

        except asyncio.exceptions.TimeoutError:
            raise ConnectionClosed("Connection timed out")

        except ConnectionRefusedError:
            raise ConnectionClosed("Connection refused")

    async def disconnect(self):
        try:
            if self._conn:
                await self._conn.close()

        except websockets.exceptions.ConnectionClosed:
            pass

        self._conn = None

    async def listen(self):
        try:
            if self._conn:
                async for message in self._conn:
                    yield json.loads(message)
            else:
                raise ConnectionClosed("Not connected")

        except websockets.exceptions.ConnectionClosed:
            self._conn = None
            raise ConnectionClosed("Connection unexpectedly closed")

    async def _request(self, command):
        try:
            if self._conn:
                await self._conn.send(json.dumps(command))
            else:
                raise ConnectionClosed("Not connected")

        except websockets.exceptions.ConnectionClosed:
            self._conn = None
            raise ConnectionClosed("Connection unexpectedly closed")

    async def _transaction(self, command):
        async with self._lock:
            try:
                if self._conn:
                    await self._conn.send(json.dumps(command))
                    response = await self._conn.recv()
                    return json.loads(response)
                else:
                    raise ConnectionClosed("Not connected")

            except websockets.exceptions.ConnectionClosed:
                self._conn = None
                raise ConnectionClosed("Connection unexpectedly closed")