import unittest
from unittest.mock import AsyncMock

from xapi import Socket, TradeCmd, TradeType, PeriodCode

class TestSocket(unittest.IsolatedAsyncioTestCase):

    def setUp(self):
        self.socket = Socket()
        self.socket._transaction = AsyncMock()

    async def test_login(self):
        await self.socket.login("my_account_id", "my_password")
        self.socket._transaction.assert_called_once_with({
            "command": "login",
            "arguments": {
                "userId": "my_account_id",
                "password": "my_password"
            }
        })

    async def test_logout(self):
        await self.socket.logout()
        self.socket._transaction.assert_awaited_once_with({"command": "logout"})

    async def test_getAllSymbols(self):
        await self.socket.getAllSymbols()
        self.socket._transaction.assert_awaited_once_with({"command": "getAllSymbols"})

    async def test_getCalendar(self):
        await self.socket.getCalendar()
        self.socket._transaction.assert_awaited_once_with({"command": "getCalendar"})

    async def test_getChartLastRequest(self):
        await self.socket.getChartLastRequest("symbol", 123, PeriodCode.PERIOD_H4)
        self.socket._transaction.assert_awaited_once_with({
            "command": "getChartLastRequest",
            "arguments": {
                "info": {
                    "period": PeriodCode.PERIOD_H4.value,
                    "start": 123,
                    "symbol": "symbol"
                }
            }
        })

    async def test_getChartRangeRequest(self):
        await self.socket.getChartRangeRequest("symbol", 123, 456, PeriodCode.PERIOD_M1, 10)
        self.socket._transaction.assert_awaited_once_with({
            "command": "getChartRangeRequest",
            "arguments": {
                "info": {
                    "end": 456,
                    "period": PeriodCode.PERIOD_M1.value,
                    "start": 123,
                    "symbol": "symbol",
                    "ticks": 10
                }
            }
        })

    async def test_getCommissionDef(self):
        await self.socket.getCommissionDef("symbol", 123)
        self.socket._transaction.assert_awaited_once_with({
            "command": "getCommissionDef",
            "arguments": {
                "symbol": "symbol",
                "volume": 123
            }
        })

    async def test_getCurrentUserData(self):
        await self.socket.getCurrentUserData()
        self.socket._transaction.assert_awaited_once_with({
            "command": "getCurrentUserData"
        })

    async def test_getIbsHistory(self):
        await self.socket.getIbsHistory(123, 456)
        self.socket._transaction.assert_awaited_once_with({
            "command": "getIbsHistory",
            "arguments": {
                "end": 456,
                "start": 123
            }
        })

    async def test_getMarginLevel(self):
        await self.socket.getMarginLevel()
        self.socket._transaction.assert_awaited_once_with({
            "command": "getMarginLevel"
        })

    async def test_getMarginTrade(self):
        await self.socket.getMarginTrade("symbol", 123)
        self.socket._transaction.assert_awaited_once_with({
            "command": "getMarginTrade",
            "arguments": {
                "symbol": "symbol",
                "volume": 123
            }
        })

    async def test_getNews(self):
        await self.socket.getNews(123, 456)
        self.socket._transaction.assert_awaited_once_with({
            "command": "getNews",
            "arguments": {
                "end": 456,
                "start": 123
            }
        })

    async def test_getProfitCalculation(self):
        await self.socket.getProfitCalculation("symbol", TradeCmd.SELL, 1.23, 4.56, 10)
        self.socket._transaction.assert_awaited_once_with({
            "command": "getProfitCalculation",
            "arguments": {
                "closePrice": 4.56,
                "cmd": TradeCmd.SELL.value,
                "openPrice": 1.23,
                "symbol": "symbol",
                "volume": 10
            }
        })

    async def test_getServerTime(self):
        await self.socket.getServerTime()
        self.socket._transaction.assert_awaited_once_with({
            "command": "getServerTime"
        })

    async def test_getStepRules(self):
        await self.socket.getStepRules()
        self.socket._transaction.assert_awaited_once_with({
            "command": "getStepRules"
        })

    async def test_getSymbol(self):
        await self.socket.getSymbol("symbol")
        self.socket._transaction.assert_awaited_once_with({
            "command": "getSymbol",
            "arguments": {
                "symbol": "symbol"
            }
        })

    async def test_getTickPrices(self):
        await self.socket.getTickPrices(["symbol_a", "symbol_b"], 123)
        self.socket._transaction.assert_awaited_once_with({
            "command": "getTickPrices",
            "arguments": {
                "level": 0,
                "symbols": ["symbol_a", "symbol_b"],
                "timestamp": 123
            }
        })

    async def test_getTradeRecords(self):
        await self.socket.getTradeRecords([123, 456])
        self.socket._transaction.assert_awaited_once_with({
            "command": "getTradeRecords",
            "arguments": {
                "orders": [123, 456]
            }
        })

    async def test_getTrades(self):
        await self.socket.getTrades()
        self.socket._transaction.assert_awaited_once_with({
            "command": "getTrades",
            "arguments": {
                "openedOnly": True
            }
        })

    async def test_getTradesHistory(self):
        await self.socket.getTradesHistory(123)
        self.socket._transaction.assert_awaited_once_with({
            "command": "getTradesHistory",
            "arguments": {
                "end": 0,
                "start": 123
            }
        })

    async def test_getTradingHours(self):
        await self.socket.getTradingHours(["symbol_a", "symbol_b"])
        self.socket._transaction.assert_awaited_once_with({
            "command": "getTradingHours",
            "arguments": {
                "symbols": ["symbol_a", "symbol_b"]
            }
        })

    async def test_getVersion(self):
        await self.socket.getVersion()
        self.socket._transaction.assert_awaited_once_with({
            "command": "getVersion"
        })

    async def test_ping(self):
        await self.socket.ping()
        self.socket._transaction.assert_awaited_once_with({
            "command": "ping"
        })

    async def test_tradeTransaction(self):
        self.socket.safe = True
        await self.socket.tradeTransaction("symbol", TradeCmd.BUY, TradeType.OPEN, 1.23, 4.56)
        self.socket._transaction.assert_not_awaited()

        self.socket.safe = False
        await self.socket.tradeTransaction("symbol", TradeCmd.BUY, TradeType.OPEN, 1.23, 4.56)
        self.socket._transaction.assert_awaited_once_with({
            "command": "tradeTransaction",
            "arguments": {
                "tradeTransInfo": {
                    "cmd": TradeCmd.BUY.value,
                    "customComment": str(),
                    "expiration": 0,
                    "offset": 0,
                    "order": 0,
                    "price": 1.23,
                    "sl": 0,
                    "symbol": "symbol",
                    "tp": 0,
                    "type": TradeType.OPEN.value,
                    "volume": 4.56
                }
            }
        })

    async def test_tradeTransactionStatus(self):
        await self.socket.tradeTransactionStatus(123)
        self.socket._transaction.assert_awaited_once_with({
            "command": "tradeTransactionStatus",
            "arguments": {
                "order": 123
            }
        })
