import numpy as np
import dask.array as da
import xarray
from xarray import DataArray
from typing import Any, List, Optional, Tuple, Union, Sequence, Callable, Dict
from scipy.interpolate import interp1d
from dask.array import coarsen


def multiscale(
    array: Any,
    reduction: Callable[[Any], Any],
    scale_factors: Union[Sequence[int], int],
    pad_mode: Optional[str] = None,
    preserve_dtype: bool = True,
    chunks: Optional[Union[Sequence[int], Dict[str, int]]] = None,
    chained: bool = True,
) -> List[DataArray]:
    """
    Generate a lazy, coordinate-aware multiscale representation of an array.

    Parameters
    ----------
    array : numpy array, dask array, or xarray DataArray
        The array to be downscaled

    reduction : callable
        A function that aggregates chunks of data over windows. See the documentation of `dask.array.coarsen` for the expected
        signature of this callable.

    scale_factors : iterable of ints
        The desired downscaling factors, one for each axis.

    pad_mode : string or None, default=None
        How arrays should be padded prior to downscaling in order to ensure that each array dimension
        is evenly divisible by the respective scale factor. When set to `None` (default), the input will be sliced before downscaling
        if its dimensions are not divisible by `scale_factors`.

    preserve_dtype : bool, default=True
        Determines whether the multiresolution arrays are all cast to the same dtype as the input.

    chunks : sequence or dict of ints, or None, default=None.
        If `chunks` is supplied, all output arrays are returned with this chunking. If not None, this
        argument is passed directly to the `xarray.DataArray.chunk` method of each output array.

    chained : bool, default=True
        If True (default), the nth downscaled array is generated by applying the reduction function on the n-1th
        downscaled array with the user-supplied `scale_factors`. This means that the nth downscaled array directly depends on the n-1th
        downscaled array. Note that nonlinear reductions like the windowed mode may give inaccurate results with `chained` set to True.

        If False, the nth downscaled array is generated by applying the reduction function on the 0th downscaled array
        (i.e., the input array) with the `scale_factors` raised to the nth power. This means that the nth downscaled array directly
        depends on the input array.

    Returns
    -------
    result : list of DataArrays
        The `coords` attribute of these DataArrays properties that track the changing offset (if any)
        induced by the downsampling operation. Additionally, the scale factors are stored each DataArray's attrs propery under the key `scale_factors`


    """
    needs_padding = pad_mode != None
    if isinstance(scale_factors, int):
        scale_factors = (scale_factors,) * array.ndim
    else:
        assert len(scale_factors) == array.ndim

    if pad_mode is None:
        # with pad_mode set to "none", dask will trim the data such that it can be tiled
        # by the scale factors
        padded_shape = np.subtract(array.shape, np.mod(array.shape, scale_factors))
    else:
        padded_shape = prepad(array, scale_factors, pad_mode=pad_mode).shape

    levels = range(
        0, 1 + get_downscale_depth(padded_shape, scale_factors, pad=needs_padding)
    )
    scales = tuple(tuple(s ** l for s in scale_factors) for l in levels)
    result = [_ingest_array(array, scales[0])]

    for level in levels[1:]:
        if chained:
            scale = scale_factors
            downscaled = downscale(result[-1], reduction, scale, pad_mode=pad_mode)
        else:
            scale = scales[level]
            downscaled = downscale(result[0], reduction, scale, pad_mode=pad_mode)
        result.append(downscaled)

    if preserve_dtype:
        result = [r.astype(array.dtype) for r in result]

    if chunks is not None:
        if isinstance(chunks, Sequence):
            _chunks = {k: v for k, v in zip(result[0].dims, chunks)}
        elif isinstance(chunks, dict):
            _chunks = chunks
        else:
            raise ValueError(
                f"Chunks must be an instance or a dict, not {type(chunks)}"
            )
        result = [r.chunk(_chunks) for r in result]

    return result


def _ingest_array(array: Any, scales: Sequence[int]):
    """
    Ingest an array in preparation for downscaling
    """
    if hasattr(array, "coords"):
        # if the input is a xarray.DataArray, assign a new variable to the DataArray and use the variable
        # `array` to refer to the data property of that array
        data = da.asarray(array.data)
        dims = array.dims
        # ensure that key order matches dimension order
        coords = {d: array.coords[d] for d in dims}
        attrs = array.attrs
        name = array.name
    else:
        data = da.asarray(array)
        dims = tuple(f"dim_{d}" for d in range(data.ndim))
        coords = {
            dim: DataArray(offset + np.arange(s, dtype="float"), dims=dim)
            for dim, s, offset in zip(dims, array.shape, get_downsampled_offset(scales))
        }
        name = None
        attrs = {}

    result = DataArray(data=data, coords=coords, dims=dims, attrs=attrs, name=name)
    return result


def even_padding(length: int, window: int) -> int:
    """
    Compute how much to add to `length` such that the resulting value is evenly divisible by `window`.

    Parameters
    ----------
    length : int

    window : int

    Returns
    -------
    int
        Value that, when added to `length`, results in a sum that is evenly divided by `window`
    """
    return (window - (length % window)) % window


def logn(x: float, n: float) -> float:
    """
    Compute the logarithm of x base n.

    Parameters
    ----------
    x : float or int.
    n: float or int.

    Returns
    -------
    float
        np.log(x) / np.log(n)

    """
    result: float = np.log(x) / np.log(n)
    return result


def prepad(
    array: Any,
    scale_factors: Sequence[int],
    pad_mode: Optional[str] = "reflect",
    rechunk: bool = True,
) -> da.array:
    """
    Lazily pad an array such that its new dimensions are evenly divisible by some integer.

    Parameters
    ----------
    array : ndarray
        Array that will be padded.

    scale_factors : Sequence of ints
        The output array is guaranteed to have dimensions that are each evenly divisible
        by the corresponding scale factor, and chunks that are smaller than or equal
        to the scale factor (if the array has chunks)

    pad_mode : str
        The edge mode used by the padding routine. This parameter will be passed to
        `dask.array.pad` as the `mode` keyword.

    Returns
    -------
    dask array
    """

    if pad_mode == None:
        # no op
        return array

    pw = tuple(
        (0, even_padding(ax, scale)) for ax, scale in zip(array.shape, scale_factors)
    )

    result = da.pad(array, pw, mode=pad_mode)

    # rechunk so that small extra chunks added by padding are fused into larger chunks, but only if we had to add chunks after padding
    if rechunk and np.any(pw):
        new_chunks = tuple(
            np.multiply(
                scale_factors, np.ceil(np.divide(result.chunksize, scale_factors))
            ).astype("int")
        )
        result = result.rechunk(new_chunks)

    if hasattr(array, "coords"):
        new_coords = {}
        for p, k in zip(pw, array.coords):
            old_coord = array.coords[k]
            if np.diff(p) == 0:
                new_coords[k] = old_coord
            else:
                extended_coords = interp1d(
                    np.arange(len(old_coord.values)),
                    old_coord.values,
                    fill_value="extrapolate",
                )(np.arange(len(old_coord.values) + p[-1])).astype(old_coord.dtype)
                new_coords[k] = DataArray(
                    extended_coords, dims=k, attrs=old_coord.attrs
                )
        result = DataArray(
            result, coords=new_coords, dims=array.dims, attrs=array.attrs
        )
    return result


def downscale(
    array: Union[np.array, da.array, xarray.DataArray],
    reduction: Callable,
    scale_factors: Sequence[int],
    pad_mode: Optional[str] = None,
    **kwargs,
) -> DataArray:
    """
    Downscale an array using windowed aggregation. This function is a light wrapper for `dask.array.coarsen`.

    Parameters
    ----------
    array : numpy array, dask array, xarray DataArray
        The array to be downscaled.

    reduction : callable
        A function that aggregates chunks of data over windows. See the documentation of `dask.array.coarsen` for the expected
        signature of this callable.

    scale_factors : iterable of ints
        The desired downscaling factors, one for each axis.

    trim_excess : bool, default=False
        Whether the size of the input array should be increased or decreased such that
        each scale factor tiles its respective array axis. Defaults to False, which will result in the input being padded.

    **kwargs
        extra kwargs passed to dask.array.coarsen

    Returns the downscaled version of the input as a dask array.
    -------
    """

    trim_excess = False
    if pad_mode == None:
        trim_excess = True

    to_coarsen = prepad(da.asarray(array), scale_factors, pad_mode=pad_mode)

    coarsened = coarsen(
        reduction,
        to_coarsen,
        {d: s for d, s in enumerate(scale_factors)},
        trim_excess=trim_excess,
        **kwargs,
    )

    if isinstance(array, xarray.DataArray):
        base_coords = array.coords
        new_coords = base_coords
        if len(base_coords) > 0:
            new_coords = tuple(
                DataArray(
                    (offset * abs(base_coords[bc][1] - base_coords[bc][0]))
                    + (base_coords[bc][:s] * sc)
                    - base_coords[bc][0],
                    name=base_coords[bc].name,
                    attrs=base_coords[bc].attrs,
                )
                for s, bc, offset, sc in zip(
                    coarsened.shape,
                    base_coords,
                    get_downsampled_offset(scale_factors),
                    scale_factors,
                )
            )
        coarsened = DataArray(
            coarsened,
            dims=array.dims,
            coords=new_coords,
            attrs=array.attrs,
            name=array.name,
        )

    return coarsened


def get_downscale_depth(
    shape: Tuple[int, ...], scale_factors: Sequence[int], pad=False
) -> int:
    """
    For an array and a sequence of scale factors, calculate the maximum possible number of downscaling operations.
    If any element of `scale_factors` is greater than the corresponding shape, this function returns 0.
    If all `scale factors` are 1, this function returns 0.
    """
    if len(shape) != len(scale_factors):
        raise ValueError(
            f"Shape (length == {len(shape)} ) and scale factors (length == {len(scale_factors)}) do not align."
        )

    _scale_factors = np.array(scale_factors).astype("int")
    _shape = np.array(shape).astype("int")
    if np.all(_scale_factors == 1):
        result = 0
    elif np.any(_scale_factors > _shape):
        result = 0
    else:
        if pad:
            depths = np.ceil(logn(shape, scale_factors)).astype("int")
        else:
            lg = logn(shape, scale_factors)
            depths = np.floor(logn(shape, scale_factors)).astype("int")
        result = min(depths)
    return result


def get_downsampled_offset(scale_factors: Sequence[int]) -> Any:
    """
    For a given number of dimensions and a sequence of downscale factors, calculate the starting offset of the downscaled
    array in the units of the full-resolution data.
    """
    return np.array([np.arange(s).mean() for s in scale_factors])


def downscale_slice(sl: slice, scale: int) -> slice:
    """
    Downscale the start, stop, and step of a slice by an integer factor. Ceiling division is used, i.e.
    downscale_slice(Slice(0, 10, None), 3) returns Slice(0, 4, None).
    """

    start, stop, step = sl.start, sl.stop, sl.step
    if start:
        start = int(np.ceil(sl.start / scale))
    if stop:
        stop = int(np.ceil(sl.stop / scale))
    if step:
        step = int(np.ceil(sl.step / scale))
    result = slice(start, stop, step)

    return result


def slice_span(sl: slice) -> int:
    """
    Measure the length of a slice
    """
    return sl.stop - sl.start
