# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['xendit',
 'xendit.models',
 'xendit.models.balance',
 'xendit.models.creditcard',
 'xendit.models.creditcard.charge',
 'xendit.models.directdebit',
 'xendit.models.directdebit.customer',
 'xendit.models.directdebit.payment',
 'xendit.models.directdebit.paymentmethod',
 'xendit.models.directdebit.token',
 'xendit.models.disbursement',
 'xendit.models.ewallet',
 'xendit.models.ewallet.dana',
 'xendit.models.ewallet.linkaja',
 'xendit.models.ewallet.ovo',
 'xendit.models.invoice',
 'xendit.models.recurringpayment',
 'xendit.models.retailoutlet',
 'xendit.models.virtualaccount',
 'xendit.network']

package_data = \
{'': ['*']}

install_requires = \
['pytest-cov>=2.10.0,<3.0.0', 'requests>=2.23.0,<3.0.0']

setup_kwargs = {
    'name': 'xendit-python',
    'version': '0.0.1',
    'description': 'Xendit REST API Client for Python - Card, Virtual Account, Invoice, Disbursement, Recurring Payments, Payout, EWallet, Balance, Retail Outlets Services https://xendit.github.io/apireference/',
    'long_description': '# Xendit Python Library\n\nThis library is the abstraction of Xendit API for access from applications written with Python.\n\n## Table of Contents\n\n<!-- START doctoc generated TOC please keep comment here to allow auto update -->\n<!-- DON\'T EDIT THIS SECTION, INSTEAD RE-RUN doctoc TO UPDATE -->\n\n- [Xendit Python Library](#xendit-python-library)\n  - [Table of Contents](#table-of-contents)\n  - [API Documentation](#api-documentation)\n  - [Requirements](#requirements)\n  - [Installation](#installation)\n  - [Usage](#usage)\n    - [API Key](#api-key)\n      - [Global Variable](#global-variable)\n      - [Use Xendit Instance](#use-xendit-instance)\n    - [Headers](#headers)\n    - [Object Creation](#object-creation)\n    - [Balance Service](#balance-service)\n      - [Get Balance](#get-balance)\n    - [Credit Card](#credit-card)\n      - [Create Authorization](#create-authorization)\n      - [Reverse Authorization](#reverse-authorization)\n      - [Create Charge](#create-charge)\n      - [Capture Charge](#capture-charge)\n      - [Get Charge](#get-charge)\n      - [Create Refund](#create-refund)\n      - [Create Promotion](#create-promotion)\n    - [eWallets](#ewallets)\n      - [Create OVO Payment](#create-ovo-payment)\n      - [Create DANA Payment](#create-dana-payment)\n      - [Create LinkAja Payment](#create-linkaja-payment)\n      - [Get Payment Status](#get-payment-status)\n    - [Direct Debit](#direct-debit)\n      - [Create Customer](#create-customer)\n      - [Get Customer by Reference ID](#get-customer-by-reference-id)\n      - [Initialize Linked Account Tokenization](#initialize-linked-account-tokenization)\n      - [Validate OTP for Linked Account Token](#validate-otp-for-linked-account-token)\n      - [Retrieve Accessible Accounts by Linked Account Token](#retrieve-accessible-accounts-by-linked-account-token)\n      - [Create Payment Method](#create-payment-method)\n      - [Get Payment Methods by Customer ID](#get-payment-methods-by-customer-id)\n      - [Create Direct Debit Payment](#create-direct-debit-payment)\n      - [Create Recurring Payment with Direct Debit](#create-recurring-payment-with-direct-debit)\n      - [Validate OTP for Direct Debit Payment](#validate-otp-for-direct-debit-payment)\n      - [Get Direct Debit Payment Status by ID](#get-direct-debit-payment-status-by-id)\n      - [Get Direct Debit Payment Status by Reference ID](#get-direct-debit-payment-status-by-reference-id)\n    - [Virtual Account Service](#virtual-account-service)\n      - [Create Virtual Account](#create-virtual-account)\n      - [Get Virtual Account Banks](#get-virtual-account-banks)\n      - [Get Virtual Account](#get-virtual-account)\n      - [Update Virtual Account](#update-virtual-account)\n      - [Get Virtual Account Payment](#get-virtual-account-payment)\n    - [Retail Outlet Service](#retail-outlet-service)\n      - [Create Fixed Payment Code](#create-fixed-payment-code)\n      - [Update Fixed Payment Code](#update-fixed-payment-code)\n      - [Get Fixed Payment Code](#get-fixed-payment-code)\n    - [Invoice Service](#invoice-service)\n      - [Create Invoice](#create-invoice)\n      - [Get Invoice](#get-invoice)\n      - [Expire Invoice](#expire-invoice)\n      - [List All Invoice](#list-all-invoice)\n    - [Recurring Payment](#recurring-payment)\n      - [Create Recurring Payment](#create-recurring-payment)\n      - [Get Recurring Payment](#get-recurring-payment)\n      - [Edit Recurring Payment](#edit-recurring-payment)\n      - [Stop Recurring Payment](#stop-recurring-payment)\n      - [Pause Recurring Payment](#pause-recurring-payment)\n      - [Resume Recurring Payment](#resume-recurring-payment)\n    - [Disbursement Service](#disbursement-service)\n      - [Create Disbursement](#create-disbursement)\n      - [Get Disbursement by ID](#get-disbursement-by-id)\n      - [Get Disbursement by External ID](#get-disbursement-by-external-id)\n      - [Get Available Banks](#get-available-banks)\n  - [Contributing](#contributing)\n    - [Tests](#tests)\n      - [Running the Test](#running-the-test)\n      - [Creating Custom HTTP Client](#creating-custom-http-client)\n\n<!-- END doctoc generated TOC please keep comment here to allow auto update -->\n\n## API Documentation\nPlease check [Xendit API Reference](https://xendit.github.io/apireference/).\n\n## Requirements\n\nPython 3.7 or later\n\n## Installation\n\nTo use the package, run ```pip install xendit-python```\n\n## Usage\n\n### API Key\n\nTo add API Key, you have 2 option: Use global variable or use Xendit instance\n\n#### Global Variable\n\n```python\nimport xendit\nxendit.api_key = "test-key123"\n\n# Then just run each class as static\nfrom xendit import Balance\nBalance.get()\n```\n\n#### Use Xendit Instance\n```python\nimport xendit\nx = xendit.Xendit(api_key="test-key123")\n\n# Then access each class from x attribute\nBalance = x.Balance\nBalance.get()\n```\n\n### Headers\n\nYou can add headers by using the following keyword parameters\n- X-IDEMPOTENCY-KEY: `x_idempotency_key`\n\n```\nVirtualAccount.create(x_idempotency_key="your-idemp-key")\n```\n\n- for-user-id: `for_user_id`\n\n```\nBalance.get(for_user_id=\'subaccount-user-id\')\n```\n\n- X-API-VERSION: `x_api_version`\n\n```\nBalance.get(x_api_version=\'2020-01-01\')\n```\n\n### Object Creation\n\nIf an API need an object as its parameter, you can use either dictionary for that class or a helper method e.g:\n\n```python\nitems = []\nitem = {\n    id: "123123",\n    name: "Phone Case",\n    price: 100000,\n    quantity: 1\n}\nitems.append(item)\nEWallet.create_linkaja_payment(\n    external_id="linkaja-ewallet-test-1593663498",\n    phone="089911111111",\n    items=items,\n    amount=300000,\n    callback_url="https://my-shop.com/callbacks",\n    redirect_url="https://xendit.co/",\n)\n```\n\nis equivalent with\n\n```python\nitems = []\nitem = xendit.EWallet.helper_create_linkaja_item(\n    id="123123", name="Phone Case", price=100000, quantity=1\n)\nitems.append(item)\nEWallet.create_linkaja_payment(\n    external_id="linkaja-ewallet-test-1593663498",\n    phone="089911111111",\n    items=items,\n    amount=300000,\n    callback_url="https://my-shop.com/callbacks",\n    redirect_url="https://xendit.co/",\n)\n```\n\n### Balance Service\n\n#### Get Balance\n\nThe `account_type` parameter is optional.\n\n```python\nfrom xendit import Balance\nBalance.get()\n\nBalance.AccountType(\n    account_type=BalanceAccountType.CASH,\n)\n```\n\nUsage example:\n\n```python\nfrom xendit import Balance, BalanceAccountType\nBalance balance = Balance.get(\n    account_type=BalanceAccountType.CASH,\n)\n\n# To get the JSON view\nprint(balance)\n\n# To get only the value\nprint(balance.balance)\n``` \n\nWill return\n\n```\n{\'balance\': 1000000000}\n1000000000\n```\n\n### Credit Card\n\n#### Create Authorization\n\n```python\nfrom xendit import CreditCard\n\ncharge = CreditCard.create_authorization(\n    token_id="5f0410898bcf7a001a00879d",\n    external_id="card_preAuth-1594106356",\n    amount=75000,\n    card_cvn="123",\n)\nprint(charge)\n```\n\nWill return\n\n```\n{\n    "status": "AUTHORIZED",\n    "authorized_amount": 75000,\n    "capture_amount": 0,\n    "currency": "IDR",\n    "business_id": "5ed75086a883856178afc12e",\n    "merchant_id": "xendit_ctv_agg",\n    "merchant_reference_code": "5f0421faa98815a4f4c92a0d",\n    "external_id": "card_preAuth-1594106356",\n    "eci": "07",\n    "charge_type": "MULTIPLE_USE_TOKEN",\n    "masked_card_number": "400000XXXXXX0002",\n    "card_brand": "VISA",\n    "card_type": "CREDIT",\n    "descriptor": "XENDIT*XENDIT&AMP;#X27;S INTERN",\n    "bank_reconciliation_id": "5941063625146828103011",\n    "approval_code": "831000",\n    "created": "2020-07-07T07:19:22.921Z",\n    "id": "5f0421fa8cc1e8001973a1d6"\n}\n```\n\n#### Reverse Authorization\n\n```python\nfrom xendit import CreditCard\n\nreverse_authorization = CreditCard.reverse_authorizatiton(\n    credit_card_charge_id="5f0421fa8cc1e8001973a1d6",\n    external_id="reverse-authorization-1594106387",\n)\nprint(reverse_authorization)\n```\n\nWill return\n\n```\n{\n    "status": "SUCCEEDED",\n    "currency": "IDR",\n    "credit_card_charge_id": "5f0421fa8cc1e8001973a1d6",\n    "business_id": "5ed75086a883856178afc12e",\n    "external_id": "card_preAuth-1594106356",\n    "amount": 75000,\n    "created": "2020-07-07T07:19:48.896Z",\n    "id": "5f0422148cc1e8001973a1dc"\n}\n```\n\n#### Create Charge\n\n```python\nfrom xendit import CreditCard\n\ncharge = CreditCard.create_charge(\n    token_id="5f0410898bcf7a001a00879d",\n    external_id="card_charge-1594106478",\n    amount=75000,\n    card_cvn="123",\n)\nprint(charge)\n```\n\nWill return\n\n```\n{\n    "status": "CAPTURED",\n    "authorized_amount": 75000,\n    "capture_amount": 75000,\n    "currency": "IDR",\n    "business_id": "5ed75086a883856178afc12e",\n    "merchant_id": "xendit_ctv_agg",\n    "merchant_reference_code": "5f0422746fc1d25bd222df2e",\n    "external_id": "card_charge-1594106478",\n    "eci": "07",\n    "charge_type": "MULTIPLE_USE_TOKEN",\n    "masked_card_number": "400000XXXXXX0002",\n    "card_brand": "VISA",\n    "card_type": "CREDIT",\n    "descriptor": "XENDIT*XENDIT&AMP;#X27;S INTERN",\n    "bank_reconciliation_id": "5941064846646923303008",\n    "approval_code": "831000",\n    "created": "2020-07-07T07:21:25.027Z",\n    "id": "5f0422752bbbe50019a368b5"\n}\n```\n\n#### Capture Charge\n\n```python\nfrom xendit import CreditCard\n\ncharge = CreditCard.capture_charge(\n    credit_card_charge_id="5f0422aa2bbbe50019a368c2",\n    amount=75000,\n)\nprint(charge)\n```\n\nWill return\n\n```\n{\n    "status": "CAPTURED",\n    "authorized_amount": 75000,\n    "capture_amount": 75000,\n    "currency": "IDR",\n    "created": "2020-07-07T07:22:18.719Z",\n    "business_id": "5ed75086a883856178afc12e",\n    "merchant_id": "xendit_ctv_agg",\n    "merchant_reference_code": "5f0422aa6fc1d25bd222df32",\n    "external_id": "card_preAuth-1594106532",\n    "eci": "07",\n    "charge_type": "MULTIPLE_USE_TOKEN",\n    "masked_card_number": "400000XXXXXX0002",\n    "card_brand": "VISA",\n    "card_type": "CREDIT",\n    "descriptor": "XENDIT*XENDIT&AMP;#X27;S INTERN",\n    "bank_reconciliation_id": "5941065383296525603007",\n    "approval_code": "831000",\n    "mid_label": "CTV_TEST",\n    "id": "5f0422aa2bbbe50019a368c2"\n}\n```\n\n#### Get Charge\n\n```python\nfrom xendit import CreditCard\n\ncharge = CreditCard.get_charge(\n    credit_card_charge_id="5f0422aa2bbbe50019a368c2",\n)\nprint(charge)\n```\n\nWill return\n\n```\n{\n    "status": "CAPTURED",\n    "authorized_amount": 75000,\n    "capture_amount": 75000,\n    "currency": "IDR",\n    "created": "2020-07-07T07:22:18.719Z",\n    "business_id": "5ed75086a883856178afc12e",\n    "merchant_id": "xendit_ctv_agg",\n    "merchant_reference_code": "5f0422aa6fc1d25bd222df32",\n    "external_id": "card_preAuth-1594106532",\n    "eci": "07",\n    "charge_type": "MULTIPLE_USE_TOKEN",\n    "masked_card_number": "400000XXXXXX0002",\n    "card_brand": "VISA",\n    "card_type": "CREDIT",\n    "descriptor": "XENDIT*XENDIT&AMP;#X27;S INTERN",\n    "bank_reconciliation_id": "5941065383296525603007",\n    "approval_code": "831000",\n    "mid_label": "CTV_TEST",\n    "metadata": {},\n    "id": "5f0422aa2bbbe50019a368c2"\n}\n```\n\n#### Create Refund\n\n```python\nfrom xendit import CreditCard\n\nrefund = CreditCard.create_refund(\n    credit_card_charge_id="5f0422aa2bbbe50019a368c2",\n    amount=10000,\n    external_id="card_refund-1594106755",\n)\nprint(refund)\n```\n\nWill return\n\n```\n{\n    "status": "REQUESTED",\n    "currency": "IDR",\n    "credit_card_charge_id": "5f0422aa2bbbe50019a368c2",\n    "user_id": "5ed75086a883856178afc12e",\n    "amount": 10000,\n    "external_id": "card_refund-1594106755",\n    "created": "2020-07-07T07:25:56.872Z",\n    "updated": "2020-07-07T07:25:57.740Z",\n    "id": "5f0423848bb8da600c57c44f",\n    "fee_refund_amount": 290\n}\n```\n\n#### Create Promotion\n\n```python\nfrom xendit import CreditCard\n\npromotion = CreditCard.create_promotion(\n    reference_id="BRI_20_JAN-1594176600",\n    description="20% discount applied for all BRI cards",\n    discount_amount=10000,\n    bin_list=[\'400000\', \'460000\'],\n    start_time="2020-01-01T00:00:00.000Z",\n    end_time="2021-01-01T00:00:00.000Z",\n)\nprint(promotion)\n```\n\nWill return\n\n```\n{\n    "business_id": "5ed75086a883856178afc12e",\n    "reference_id": "BRI_20_JAN-1594176600",\n    "description": "20% discount applied for all BRI cards",\n    "start_time": "2020-01-01T00:00:00.000Z",\n    "end_time": "2021-01-01T00:00:00.000Z",\n    "type": "CARD_BIN",\n    "discount_amount": 10000,\n    "bin_list": [\n        "400000",\n        "460000"\n    ],\n    "currency": "IDR",\n    "id": "c65a2ae7-ce75-4a15-bbec-55d076f46bd0",\n    "created": "2020-07-08T02:50:02.296Z",\n    "status": "ACTIVE"\n}\n```\n\n### eWallets\n\n#### Create OVO Payment\n\n```python\nfrom xendit import EWallet\n\novo_payment = EWallet.create_ovo_payment(\n    external_id="ovo-ewallet-testing-id-1593663430",\n    amount="80001",\n    phone="08123123123",\n)\nprint(ovo_payment)\n```\n\nWill return\n\n```\n{\n    "amount": 80001,\n    "business_id": "5ed75086a883856178afc12e",\n    "external_id": "ovo-ewallet-testing-id-1593663430",\n    "ewallet_type": "OVO",\n    "phone": "08123123123",\n    "created": "2020-07-02T04:17:12.979Z",\n    "status": "PENDING"\n}\n```\n\n#### Create DANA Payment\n\n```python\nfrom xendit import EWallet\n\ndana_payment = EWallet.create_dana_payment(\n    external_id="dana-ewallet-test-1593663447",\n    amount="1001",\n    callback_url="https://my-shop.com/callbacks",\n    redirect_url="https://my-shop.com/home",\n)\nprint(dana_payment)\n```\n\nWill return\n\n```\n{\n    "external_id": "dana-ewallet-test-1593663447",\n    "amount": 1001,\n    "checkout_url": "https://sandbox.m.dana.id/m/portal/cashier/checkout?bizNo=20200702111212800110166820100550620&timestamp=1593663450389&mid=216620000000261692328&sign=XS3FMKj1oZHkTWu0EXk8PBwzjR1VtwSedqbKX%2BgMF6CyZvbA5xhAmMUR%2FlhD4QkBODbbTPcju1YDFnHmSdzmjbqPfGcQGtkCPgLwVOZo1ERPmoUhhGJIbQXkfZ1Z8eA1w1RSqDzdmDB%2B%2FlvHaTbYPiUlvjzs%2BfgkM33YFFEl0BG1kUFz0%2FKb9OoT1QKyoHxw6ge4SWPF3Po6BwNtjqUZe2n43s7y0CvSrcNiNLHT3k2XHSlIdguwCGjNHh2zClgtv9XbSCecnD96nuIuohYARX8Ai%2BaYo%2FEDO1VEch4XditfIXvyBhL0TocxhYxda7yKNNjkZj56Rl9ds8u7Wyv1eQ%3D%3D",\n    "ewallet_type": "DANA"\n}\n```\n\n#### Create LinkAja Payment\n\n```python\nfrom xendit import EWallet, LinkAjaItem\n\nitems = []\nitems.append(LinkAjaItem(id="123123", name="Phone Case", price=100000, quantity=1))\nlinkaja_payment = EWallet.create_linkaja_payment(\n    external_id="linkaja-ewallet-test-1593663498",\n    phone="089911111111",\n    items=items,\n    amount=300000,\n    callback_url="https://my-shop.com/callbacks",\n    redirect_url="https://xendit.co/",\n)\nprint(linkaja_payment)\n```\n\nWill return\n\n```\n{\n    "checkout_url": "https://ewallet-linkaja-dev.xendit.co/checkouts/c627cf1f-0470-420f-a0f4-3931ef384bf4",\n    "transaction_date": "2020-07-02T04:18:21.729Z",\n    "amount": 300000,\n    "external_id": "linkaja-ewallet-test-1593663498",\n    "ewallet_type": "LINKAJA"\n}\n```\n\n#### Get Payment Status\n\n```python\nfrom xendit import EWallet\n\novo_payment_status = EWallet.get_payment_status(\n    ewallet_type=EWalletType.OVO,\n    external_id="ovo-ewallet-testing-id-1234",\n)\nprint(ovo_payment_status)\n```\n\nWill return\n\n```\n{\n    "amount": "8888",\n    "business_id": "5ed75086a883856178afc12e",\n    "ewallet_type": "OVO",\n    "external_id": "ovo-ewallet-testing-id-1234",\n    "status": "COMPLETED",\n    "transaction_date": "2020-06-30T01:32:28.267Z"\n}\n```\n\n### Direct Debit\n\n#### Create Customer\n\n```python\nfrom xendit import DirectDebit\n\ncustomer = DirectDebit.create_customer(\n    reference_id="merc-1594279037",\n    email="t@x.co",\n    given_names="Adyaksa",\n)\nprint(customer)\n```\n\nWill return\n\n```\n{\n    "id": "ed20b5db-df04-41fc-8018-8ea4ac4d1030",\n    "reference_id": "merc-1594279037",\n    "description": null,\n    "given_names": "Adyaksa",\n    "middle_name": null,\n    "surname": null,\n    "mobile_number": null,\n    "phone_number": null,\n    "email": "t@x.co",\n    "nationality": null,\n    "addresses": null,\n    "date_of_birth": null,\n    "employment": null,\n    "source_of_wealth": null,\n    "metadata": null\n}\n```\n\n#### Get Customer by Reference ID\n\n```python\nfrom xendit import DirectDebit\n\ncustomer = DirectDebit.get_customer_by_ref_id(\n    reference_id="merc-1594279037",\n)\nprint(customer)\n```\n\nWill return\n\n```\n[{\n    "id": "ed20b5db-df04-41fc-8018-8ea4ac4d1030",\n    "reference_id": "merc-1594279037",\n    "description": null,\n    "given_names": "Adyaksa",\n    "middle_name": null,\n    "surname": null,\n    "mobile_number": null,\n    "phone_number": null,\n    "email": "t@x.co",\n    "nationality": null,\n    "addresses": null,\n    "date_of_birth": null,\n    "employment": null,\n    "source_of_wealth": null,\n    "metadata": null\n}]\n```\n\n#### Initialize Linked Account Tokenization\n\n```python\nfrom xendit import DirectDebit\n\ncard_linking = DirectDebit.helper_create_card_link(\n    account_mobile_number="+62818555988",\n    card_last_four="8888",\n    card_expiry="06/24",\n    account_email="test.email@xendit.co",\n)\nlinked_account = DirectDebit.initialize_tokenization(\n    customer_id="ed20b5db-df04-41fc-8018-8ea4ac4d1030",\n    channel_code="DC_BRI",\n    properties=card_linking,   \n)\nprint(linked_account)\n```\n\nWill return\n\n```\n{\n    "id": "lat-f325b757-0aae-4c24-92c5-3661e299e154",\n    "customer_id": "ed20b5db-df04-41fc-8018-8ea4ac4d1030",\n    "channel_code": "DC_BRI",\n    "authorizer_url": null,\n    "status": "PENDING",\n    "metadata": null\n}\n```\n\n#### Validate OTP for Linked Account Token\n\n```python\nfrom xendit import DirectDebit\n\nlinked_account = DirectDebit.validate_token_otp(\n    linked_account_token_id="lat-f325b757-0aae-4c24-92c5-3661e299e154",\n    otp_code="333000",\n)\nprint(linked_account)\n```\n\nWill return\n\n```\n{\n    "id": "lat-f325b757-0aae-4c24-92c5-3661e299e154",\n    "customer_id": "ed20b5db-df04-41fc-8018-8ea4ac4d1030",\n    "channel_code": "DC_BRI",\n    "status": "SUCCESS",\n    "metadata": null\n}\n```\n\n#### Retrieve Accessible Accounts by Linked Account Token\n\n```python\nfrom xendit import DirectDebit\n\naccessible_accounts = DirectDebit.get_accessible_account_by_token(\n    linked_account_token_id="lat-f325b757-0aae-4c24-92c5-3661e299e154",\n)\nprint(accessible_accounts)\n```\n\nWill return\n\n```\n[{\n    "channel_code": "DC_BRI",\n    "id": "la-08b089e8-7035-4f5f-bdd9-94edd9dc9480",\n    "properties": {\n        "card_expiry": "06/24",\n        "card_last_four": "8888",\n        "currency": "IDR",\n        "description": ""\n    },\n    "type": "DEBIT_CARD"\n}]\n```\n\n#### Create Payment Method\n\n```python\nfrom xendit import DirectDebit\n\npayment_method = DirectDebit.create_payment_method(\n    customer_id="ed20b5db-df04-41fc-8018-8ea4ac4d1030",\n    type=DirectDebitPaymentMethodType.DEBIT_CARD,\n    properties={\'id\': \'la-fac7e744-ab40-4100-a447-cbbb16f29ded\'},\n)\n\nprint(payment_method)\n```\n\nWill return\n\n```\n{\n    "customer_id": "ed20b5db-df04-41fc-8018-8ea4ac4d1030",\n    "type": "DEBIT_CARD",\n    "properties": {\n        "id": "la-fac7e744-ab40-4100-a447-cbbb16f29ded",\n        "currency": "IDR",\n        "card_expiry": "06/24",\n        "description": "",\n        "channel_code": "DC_BRI",\n        "card_last_four": "8888"\n    },\n    "status": "ACTIVE",\n    "metadata": {},\n    "id": "pm-b6116aea-8c23-42d0-a1e6-33227b52fccd",\n    "created": "2020-07-13T07:28:57.716Z",\n    "updated": "2020-07-13T07:28:57.716Z"\n}\n```\n\n#### Get Payment Methods by Customer ID\n\n```python\nfrom xendit import DirectDebit\n\npayment_methods = DirectDebit.get_payment_methods_by_customer_id(\n    customer_id="ed20b5db-df04-41fc-8018-8ea4ac4d1030",\n)\n\nprint(payment_methods)\n```\n\nWill return\n\n```\n[{\n    "id": "pm-b6116aea-8c23-42d0-a1e6-33227b52fccd",\n    "customer_id": "ed20b5db-df04-41fc-8018-8ea4ac4d1030",\n    "status": "ACTIVE",\n    "type": "DEBIT_CARD",\n    "properties": {\n        "id": "la-fac7e744-ab40-4100-a447-cbbb16f29ded",\n        "currency": "IDR",\n        "card_expiry": "06/24",\n        "description": "",\n        "channel_code": "DC_BRI",\n        "card_last_four": "8888"\n    },\n    "metadata": {},\n    "created": "2020-07-13T07:28:57.716Z",\n    "updated": "2020-07-13T07:28:57.716Z"\n}]\n```\n\n#### Create Direct Debit Payment\n\n```python\nfrom xendit import DirectDebit\n\npayment = DirectDebit.create_payment(\n    reference_id="direct-debit-ref-1594718940",\n    payment_method_id="pm-b6116aea-8c23-42d0-a1e6-33227b52fccd",\n    currency="IDR",\n    amount="60000",\n    callback_url="http://webhook.site/",\n    enable_otp=True,\n    idempotency_key="idemp_key-1594718940",\n)\n\nprint(payment)\n```\n\nWill return\n\n```\n{\n    "failure_code": null,\n    "otp_mobile_number": null,\n    "otp_expiration_timestamp": null,\n    "id": "ddpy-eaa093b6-b669-401a-ba2e-61ac644b2aff",\n    "reference_id": "direct-debit-ref-1594718940",\n    "payment_method_id": "pm-b6116aea-8c23-42d0-a1e6-33227b52fccd",\n    "channel_code": "DC_BRI",\n    "currency": "IDR",\n    "amount": 60000,\n    "is_otp_required": true,\n    "basket": null,\n    "description": "",\n    "status": "PENDING",\n    "metadata": null,\n    "created": "2020-07-14T09:29:02.614443Z",\n    "updated": "2020-07-14T09:29:02.614443Z"\n}\n```\n\n#### Create Recurring Payment with Direct Debit\n\nYou can use [Create Recurring Payment](#create-recurring-payment) to use this feature.\n\n#### Validate OTP for Direct Debit Payment\n\n```python\nfrom xendit import DirectDebit\n\npayment = DirectDebit.validate_payment_otp(\n    direct_debit_id="ddpy-eaa093b6-b669-401a-ba2e-61ac644b2aff",\n    otp_code="222000",\n)\n\nprint(payment)\n```\n\nWill return\n\n```\n{\n    "failure_code": null,\n    "otp_mobile_number": null,\n    "otp_expiration_timestamp": null,\n    "id": "ddpy-eaa093b6-b669-401a-ba2e-61ac644b2aff",\n    "reference_id": "direct-debit-ref-1594718940",\n    "payment_method_id": "pm-b6116aea-8c23-42d0-a1e6-33227b52fccd",\n    "channel_code": "DC_BRI",\n    "currency": "IDR",\n    "amount": 60000,\n    "is_otp_required": true,\n    "basket": null,\n    "description": "",\n    "status": "PENDING",\n    "metadata": null,\n    "created": "2020-07-14T09:29:02.614443Z",\n    "updated": "2020-07-14T09:29:02.614443Z"\n}\n```\n\n#### Get Direct Debit Payment Status by ID\n\n```python\nfrom xendit import DirectDebit\n\npayment = DirectDebit.get_payment_status(\n    direct_debit_id="ddpy-38ef50a8-00f0-4019-8b28-9bca81f2cbf1",\n)\n\nprint(payment)\n```\n\nWill return\n\n```\n{\n    "failure_code": null,\n    "otp_mobile_number": null,\n    "otp_expiration_timestamp": null,\n    "id": "ddpy-38ef50a8-00f0-4019-8b28-9bca81f2cbf1",\n    "reference_id": "direct-debit-ref-1594717458",\n    "payment_method_id": "pm-b6116aea-8c23-42d0-a1e6-33227b52fccd",\n    "channel_code": "DC_BRI",\n    "currency": "IDR",\n    "amount": 60000,\n    "is_otp_required": false,\n    "basket": null,\n    "description": "",\n    "status": "PENDING",\n    "metadata": null,\n    "created": "2020-07-14T09:04:20.031451Z",\n    "updated": "2020-07-14T09:04:20.031451Z"\n}\n```\n\n#### Get Direct Debit Payment Status by Reference ID\n\n```python\nfrom xendit import DirectDebit\n\npayments = DirectDebit.get_payment_status_by_ref_id(\n    reference_id="direct-debit-ref-1594717458",\n)\n\nprint(payments)\n```\n\nWill return\n\n```\n[{\n    "amount": 60000,\n    "basket": null,\n    "channel_code": "DC_BRI",\n    "created": "2020-07-14T09:04:20.031451Z",\n    "currency": "IDR",\n    "description": "",\n    "failure_code": null,\n    "id": "ddpy-38ef50a8-00f0-4019-8b28-9bca81f2cbf1",\n    "is_otp_required": false,\n    "metadata": null,\n    "otp_expiration_timestamp": null,\n    "otp_mobile_number": null,\n    "payment_method_id": "pm-b6116aea-8c23-42d0-a1e6-33227b52fccd",\n    "reference_id": "direct-debit-ref-1594717458",\n    "status": "PENDING",\n    "updated": "2020-07-14T09:04:20.031451Z"\n}]\n```\n\n### Virtual Account Service\n\n#### Create Virtual Account\n\n```python\nfrom xendit import VirtualAccount\n\nvirtual_account = VirtualAccount.create(\n    external_id="demo_1475459775872",\n    bank_code="BNI",\n    name="Rika Sutanto",\n)\nprint(virtual_account)\n```\n\nWill return\n\n```\n{\n    "owner_id": "5ed75086a883856178afc12e",\n    "external_id": "demo_1475459775872",\n    "bank_code": "BNI",\n    "merchant_code": "8808",\n    "name": "Rika Sutanto",\n    "account_number": "8808999956275653",\n    "is_single_use": false,\n    "status": "PENDING",\n    "expiration_date": "2051-06-22T17:00:00.000Z",\n    "is_closed": false,\n    "id": "5ef174c48dd9ea2fc97d6a1e"\n}\n```\n\n#### Get Virtual Account Banks\n```python\nfrom xendit import VirtualAccount\n\nvirtual_account_banks = VirtualAccount.get_banks()\nprint(virtual_account_banks)\n```\n\nWill return\n\n```\n[{\n    "name": "Bank Mandiri",\n    "code": "MANDIRI"\n}, {\n    "name": "Bank Negara Indonesia",\n    "code": "BNI"\n}, {\n    "name": "Bank Rakyat Indonesia",\n    "code": "BRI"\n}, {\n    "name": "Bank Permata",\n    "code": "PERMATA"\n}, {\n    "name": "Bank Central Asia",\n    "code": "BCA"\n}]\n```\n#### Get Virtual Account\n\n```python\nfrom xendit import VirtualAccount\n\nvirtual_account = VirtualAccount.get(\n    id="5eec3a3e8dd9ea2fc97d6728",\n)\nprint(virtual_account)\n```\n\nWill return\n\n```\n{\n    "owner_id": "5ed75086a883856178afc12e",\n    "external_id": "demo_1475459775872",\n    "bank_code": "BNI",\n    "merchant_code": "8808",\n    "name": "Rika Sutanto",\n    "account_number": "8808999917965673",\n    "is_single_use": true,\n    "status": "ACTIVE",\n    "expiration_date": "2051-06-18T17:00:00.000Z",\n    "is_closed": false,\n    "id": "5eec3a3e8dd9ea2fc97d6728"\n}\n```\n\n#### Update Virtual Account\n\n```python\nfrom xendit import VirtualAccount\n\nvirtual_account = VirtualAccount.update(\n    id="5eec3a3e8dd9ea2fc97d6728",\n    is_single_use=True,\n)\nprint(virtual_account)\n```\n\nWill return\n\n```\n{\n    "owner_id": "5ed75086a883856178afc12e",\n    "external_id": "demo_1475459775872",\n    "bank_code": "BNI",\n    "merchant_code": "8808",\n    "name": "Rika Sutanto",\n    "account_number": "8808999917965673",\n    "is_single_use": true,\n    "status": "PENDING",\n    "expiration_date": "2051-06-18T17:00:00.000Z",\n    "is_closed": false,\n    "id": "5eec3a3e8dd9ea2fc97d6728"\n}\n```\n\n#### Get Virtual Account Payment\n\n```python\nfrom xendit import VirtualAccount\n\nvirtual_account_payment = VirtualAccount.get(\n    payment_id="5ef18efca7d10d1b4d61fb52",\n)\nprint(virtual_account)\n```\n\nWill return\n\n```\n{\n    "id": "5ef18efcf9ce3b5f8e188ee4",\n    "payment_id": "5ef18efca7d10d1b4d61fb52",\n    "callback_virtual_account_id": "5ef18ece8dd9ea2fc97d6a84",\n    "external_id": "VA_fixed-1592889038",\n    "merchant_code": "88608",\n    "account_number": "9999317837",\n    "bank_code": "MANDIRI",\n    "amount": 50000,\n    "transaction_timestamp": "2020-06-23T05:11:24.000Z"\n}\n```\n\n### Retail Outlet Service\n\n#### Create Fixed Payment Code\n\n```python\nfrom xendit import RetailOutlet\n\nretail_outlet = RetailOutlet.create_fixed_payment_code(\n    external_id="demo_fixed_payment_code_123",\n    retail_outlet_name="ALFAMART",\n    name="Rika Sutanto",\n    expected_amount=10000,\n)\nprint(retail_outlet)\n```\n\nWill return\n\n```\n{\n    "owner_id": "5ed75086a883856178afc12e",\n    "external_id": "demo_fixed_payment_code_123",\n    "retail_outlet_name": "ALFAMART",\n    "prefix": "TEST",\n    "name": "Rika Sutanto",\n    "payment_code": "TEST56147",\n    "expected_amount": 10000,\n    "is_single_use": False,\n    "expiration_date": "2051-06-23T17:00:00.000Z",\n    "id": "5ef2f0f8e7f5c14077275493",\n}\n```\n\n#### Update Fixed Payment Code\n\n```python\nfrom xendit import RetailOutlet\n\nretail_outlet = RetailOutlet.update_fixed_payment_code(\n    fixed_payment_code_id="5ef2f0f8e7f5c14077275493",\n    name="Joe Contini",\n)\nprint(retail_outlet)\n```\n\nWill return\n\n```\n{\n    "owner_id": "5ed75086a883856178afc12e",\n    "external_id": "demo_fixed_payment_code_123",\n    "retail_outlet_name": "ALFAMART",\n    "prefix": "TEST",\n    "name": "Joe Contini",\n    "payment_code": "TEST56147",\n    "expected_amount": 10000,\n    "is_single_use": False,\n    "expiration_date": "2051-06-23T17:00:00.000Z",\n    "id": "5ef2f0f8e7f5c14077275493",\n}\n```\n\n#### Get Fixed Payment Code\n\n```python\nfrom xendit import RetailOutlet\n\nretail_outlet = RetailOutlet.get_fixed_payment_code(\n    fixed_payment_code_id="5ef2f0f8e7f5c14077275493",\n)\nprint(retail_outlet)\n```\n\nWill return\n\n```\n{\n    "owner_id": "5ed75086a883856178afc12e",\n    "external_id": "demo_fixed_payment_code_123",\n    "retail_outlet_name": "ALFAMART",\n    "prefix": "TEST",\n    "name": "Rika Sutanto",\n    "payment_code": "TEST56147",\n    "expected_amount": 10000,\n    "is_single_use": False,\n    "expiration_date": "2051-06-23T17:00:00.000Z",\n    "id": "5ef2f0f8e7f5c14077275493",\n}\n```\n\n### Invoice Service\n\n#### Create Invoice\n\n```python\nfrom xendit import Invoice\n\ninvoice = Invoice.create(\n    external_id="invoice-1593684000",\n    amount=20000,\n    payer_email="customer@domain.com",\n    description="Invoice Demo #123",\n)\nprint(invoice)\n```\n\nWill return\n\n```\n{\n    "id": "5efdb0210425db620ec35fb3",\n    "external_id": "invoice-1593684000",\n    "user_id": "5ed75086a883856178afc12e",\n    "status": "PENDING",\n    "merchant_name": "Xendit&amp;#x27;s Intern",\n    "merchant_profile_picture_url": "https://xnd-companies.s3.amazonaws.com/prod/1591169469152_279.png",\n    "amount": 20000,\n    "payer_email": "customer@domain.com",\n    "description": "Invoice Demo #123",\n    "expiry_date": "2020-07-03T10:00:01.148Z",\n    "invoice_url": "https://invoice-staging.xendit.co/web/invoices/5efdb0210425db620ec35fb3",\n    "available_banks": [\n        {\n            "bank_code": "MANDIRI",\n            "collection_type": "POOL",\n            "bank_account_number": "8860846854335",\n            "transfer_amount": 20000,\n            "bank_branch": "Virtual Account",\n            "account_holder_name": "XENDIT&AMP;#X27;S INTERN",\n            "identity_amount": 0\n        },\n        {\n            "bank_code": "BRI",\n            "collection_type": "POOL",\n            "bank_account_number": "2621554807492",\n            "transfer_amount": 20000,\n            "bank_branch": "Virtual Account",\n            "account_holder_name": "XENDIT&AMP;#X27;S INTERN",\n            "identity_amount": 0\n        },\n        {\n            "bank_code": "BNI",\n            "collection_type": "POOL",\n            "bank_account_number": "880854597383",\n            "transfer_amount": 20000,\n            "bank_branch": "Virtual Account",\n            "account_holder_name": "XENDIT&AMP;#X27;S INTERN",\n            "identity_amount": 0\n        },\n        {\n            "bank_code": "PERMATA",\n            "collection_type": "POOL",\n            "bank_account_number": "821456659745",\n            "transfer_amount": 20000,\n            "bank_branch": "Virtual Account",\n            "account_holder_name": "XENDIT&AMP;#X27;S INTERN",\n            "identity_amount": 0\n        },\n        {\n            "bank_code": "BCA",\n            "collection_type": "POOL",\n            "bank_account_number": "1076619844859",\n            "transfer_amount": 20000,\n            "bank_branch": "Virtual Account",\n            "account_holder_name": "XENDIT&AMP;#X27;S INTERN",\n            "identity_amount": 0\n        }\n    ],\n    "available_retail_outlets": [\n        {\n            "retail_outlet_name": "ALFAMART",\n            "payment_code": "TEST34956",\n            "transfer_amount": 20000\n        }\n    ],\n    "available_ewallets": [],\n    "should_exclude_credit_card": false,\n    "should_send_email": false,\n    "created": "2020-07-02T10:00:01.285Z",\n    "updated": "2020-07-02T10:00:01.285Z",\n    "currency": "IDR"\n}\n```\n\n#### Get Invoice\n\n```python\nfrom xendit import Invoice\n\ninvoice = Invoice.get(\n    invoice_id="5efda8a20425db620ec35f43",\n)\nprint(invoice)\n```\n\nWill return\n\n```\n{\n    "id": "5efda8a20425db620ec35f43",\n    "external_id": "invoice-1593682080",\n    "user_id": "5ed75086a883856178afc12e",\n    "status": "EXPIRED",\n    "merchant_name": "Xendit&amp;#x27;s Intern",\n    "merchant_profile_picture_url": "https://xnd-companies.s3.amazonaws.com/prod/1591169469152_279.png",\n    "amount": 20000,\n    "payer_email": "customer@domain.com",\n    "description": "Invoice Demo #123",\n    "expiry_date": "2020-07-02T09:55:47.794Z",\n    "invoice_url": "https://invoice-staging.xendit.co/web/invoices/5efda8a20425db620ec35f43",\n    "available_banks": [\n        {\n            "bank_code": "MANDIRI",\n            "collection_type": "POOL",\n            "bank_account_number": "8860846853111",\n            "transfer_amount": 20000,\n            "bank_branch": "Virtual Account",\n            "account_holder_name": "XENDIT&AMP;#X27;S INTERN",\n            "identity_amount": 0\n        },\n        {\n            "bank_code": "BRI",\n            "collection_type": "POOL",\n            "bank_account_number": "2621554806292",\n            "transfer_amount": 20000,\n            "bank_branch": "Virtual Account",\n            "account_holder_name": "XENDIT&AMP;#X27;S INTERN",\n            "identity_amount": 0\n        }\n    ],\n    "available_retail_outlets": [\n        {\n            "retail_outlet_name": "ALFAMART",\n            "payment_code": "TEST34950",\n            "transfer_amount": 20000\n        }\n    ],\n    "available_ewallets": [],\n    "should_exclude_credit_card": false,\n    "should_send_email": false,\n    "created": "2020-07-02T09:28:02.191Z",\n    "updated": "2020-07-02T09:55:47.794Z",\n    "currency": "IDR"\n}\n```\n\n#### Expire Invoice\n\n```python\nfrom xendit import Invoice\n\ninvoice = Invoice.expire(\n    invoice_id="5efda8a20425db620ec35f43",\n)\nprint(invoice)\n```\n\nWill return\n\n```\n{\n    "id": "5efda8a20425db620ec35f43",\n    "external_id": "invoice-1593682080",\n    "user_id": "5ed75086a883856178afc12e",\n    "status": "EXPIRED",\n    "merchant_name": "Xendit&amp;#x27;s Intern",\n    "merchant_profile_picture_url": "https://xnd-companies.s3.amazonaws.com/prod/1591169469152_279.png",\n    "amount": 20000,\n    "payer_email": "customer@domain.com",\n    "description": "Invoice Demo #123",\n    "expiry_date": "2020-07-02T09:55:47.794Z",\n    "invoice_url": "https://invoice-staging.xendit.co/web/invoices/5efda8a20425db620ec35f43",\n    "available_banks": [\n        {\n            "bank_code": "MANDIRI",\n            "collection_type": "POOL",\n            "bank_account_number": "8860846853111",\n            "transfer_amount": 20000,\n            "bank_branch": "Virtual Account",\n            "account_holder_name": "XENDIT&AMP;#X27;S INTERN",\n            "identity_amount": 0\n        },\n        {\n            "bank_code": "BRI",\n            "collection_type": "POOL",\n            "bank_account_number": "2621554806292",\n            "transfer_amount": 20000,\n            "bank_branch": "Virtual Account",\n            "account_holder_name": "XENDIT&AMP;#X27;S INTERN",\n            "identity_amount": 0\n        }\n    "available_retail_outlets": [\n        {\n            "retail_outlet_name": "ALFAMART",\n            "payment_code": "TEST34950",\n            "transfer_amount": 20000\n        }\n    ],\n    "available_ewallets": [],\n    "should_exclude_credit_card": false,\n    "should_send_email": false,\n    "created": "2020-07-02T09:28:02.191Z",\n    "updated": "2020-07-02T09:55:47.794Z",\n    "currency": "IDR"\n}\n```\n\n#### List All Invoice\n\n```python\nfrom xendit import Invoice\n\ninvoices = Invoice.list_all(\n    limit=3,\n)\nprint(invoices)\n```\n\nWill return\n\n```\n[\n    ...\n    {\n        "id": "5efda8a20425db620ec35f43",\n        "external_id": "invoice-1593682080",\n        "user_id": "5ed75086a883856178afc12e",\n        "status": "EXPIRED",\n        "merchant_name": "Xendit&amp;#x27;s Intern",\n        "merchant_profile_picture_url": "https://xnd-companies.s3.amazonaws.com/prod/1591169469152_279.png",\n        "amount": 20000,\n        "payer_email": "customer@domain.com",\n        "description": "Invoice Demo #123",\n        "expiry_date": "2020-07-02T09:55:47.794Z",\n        "invoice_url": "https://invoice-staging.xendit.co/web/invoices/5efda8a20425db620ec35f43",\n        "available_banks": [\n            {\n                "bank_code": "MANDIRI",\n                "collection_type": "POOL",\n                "bank_account_number": "8860846853111",\n                "transfer_amount": 20000,\n                "bank_branch": "Virtual Account",\n                "account_holder_name": "XENDIT&AMP;#X27;S INTERN",\n                "identity_amount": 0\n            },\n            {\n                "bank_code": "BRI",\n                "collection_type": "POOL",\n                "bank_account_number": "2621554806292",\n                "transfer_amount": 20000,\n                "bank_branch": "Virtual Account",\n                "account_holder_name": "XENDIT&AMP;#X27;S INTERN",\n                "identity_amount": 0\n            }\n        "available_retail_outlets": [\n            {\n                "retail_outlet_name": "ALFAMART",\n                "payment_code": "TEST34950",\n                "transfer_amount": 20000\n            }\n        ],\n        "available_ewallets": [],\n        "should_exclude_credit_card": false,\n        "should_send_email": false,\n        "created": "2020-07-02T09:28:02.191Z",\n        "updated": "2020-07-02T09:55:47.794Z",\n        "currency": "IDR"\n    }\n    ...\n]\n```\n\n### Recurring Payment\n\n#### Create Recurring Payment\n\n```python\nfrom xendit import RecurringPayment\n\nrecurring_payment = RecurringPayment.create_recurring_payment(\n    external_id="recurring_12345",\n    payer_email="test@x.co",\n    description="Test Curring Payment",\n    amount=100000,\n    interval="MONTH",\n    interval_count=1,\n)\n```\n\nWill return\n\n```\n{\n    "status": "ACTIVE",\n    "should_send_email": false,\n    "missed_payment_action": "IGNORE",\n    "recurrence_progress": 1,\n    "recharge": true,\n    "user_id": "5ed75086a883856178afc12e",\n    "external_id": "recurring_12345",\n    "payer_email": "test@x.co",\n    "description": "Test Curring Payment",\n    "amount": 100000,\n    "interval": "MONTH",\n    "interval_count": 1,\n    "start_date": "2020-07-08T08:22:55.815Z",\n    "currency": "IDR",\n    "created": "2020-07-08T08:22:55.817Z",\n    "updated": "2020-07-08T08:22:55.994Z",\n    "id": "5f05825ff9f52d3ed204c687",\n    "last_created_invoice_url": "https://invoice-staging.xendit.co/web/invoices/5f05825ff9f52d3ed204c688"\n}\n```\n\n#### Get Recurring Payment\n\n```python\nfrom xendit import RecurringPayment\n\nrecurring_payment = RecurringPayment.get_recurring_payment(\n    id="5f05825ff9f52d3ed204c687",\n)\n```\n\nWill return\n\n```\n{\n    "status": "ACTIVE",\n    "should_send_email": false,\n    "missed_payment_action": "IGNORE",\n    "recurrence_progress": 1,\n    "recharge": true,\n    "user_id": "5ed75086a883856178afc12e",\n    "external_id": "recurring_12345",\n    "payer_email": "test@x.co",\n    "description": "Test Curring Payment",\n    "amount": 100000,\n    "interval": "MONTH",\n    "interval_count": 1,\n    "start_date": "2020-07-08T08:22:55.815Z",\n    "currency": "IDR",\n    "created": "2020-07-08T08:22:55.817Z",\n    "updated": "2020-07-08T08:22:55.994Z",\n    "id": "5f05825ff9f52d3ed204c687",\n    "last_created_invoice_url": "https://invoice-staging.xendit.co/web/invoices/5f05825ff9f52d3ed204c688"\n}\n```\n\n#### Edit Recurring Payment\n\n```python\nfrom xendit import RecurringPayment\n\nrecurring_payment = RecurringPayment.edit_recurring_payment(\n    id="5f05825ff9f52d3ed204c687",\n    interval_count=2,\n)\n```\n\nWill return\n\n```\n{\n    "status": "ACTIVE",\n    "should_send_email": false,\n    "missed_payment_action": "IGNORE",\n    "recurrence_progress": 1,\n    "recharge": true,\n    "user_id": "5ed75086a883856178afc12e",\n    "external_id": "recurring_12345",\n    "payer_email": "test@x.co",\n    "description": "Test Curring Payment",\n    "amount": 100000,\n    "interval": "MONTH",\n    "interval_count": 2,\n    "start_date": "2020-07-08T08:22:55.815Z",\n    "currency": "IDR",\n    "created": "2020-07-08T08:22:55.817Z",\n    "updated": "2020-07-08T08:24:58.295Z",\n    "id": "5f05825ff9f52d3ed204c687"\n}\n```\n\n#### Stop Recurring Payment\n\n```python\nfrom xendit import RecurringPayment\n\nrecurring_payment = RecurringPayment.stop_recurring_payment(\n    id="5f05825ff9f52d3ed204c687",\n)\n```\n\nWill return\n\n```\n{\n    "status": "STOPPED",\n    "should_send_email": false,\n    "missed_payment_action": "IGNORE",\n    "recurrence_progress": 1,\n    "recharge": true,\n    "user_id": "5ed75086a883856178afc12e",\n    "external_id": "recurring_12345",\n    "payer_email": "test@x.co",\n    "description": "Test Curring Payment",\n    "amount": 100000,\n    "interval": "MONTH",\n    "interval_count": 2,\n    "start_date": "2020-07-08T08:22:55.815Z",\n    "currency": "IDR",\n    "created": "2020-07-08T08:22:55.817Z",\n    "updated": "2020-07-08T08:26:32.464Z",\n    "id": "5f05825ff9f52d3ed204c687"\n}\n```\n\n#### Pause Recurring Payment\n\n```python\nfrom xendit import RecurringPayment\n\nrecurring_payment = RecurringPayment.pause_recurring_payment(\n    id="5f05825ff9f52d3ed204c687",\n)\n```\n\nWill return\n\n```\n{\n    "status": "PAUSED",\n    "should_send_email": false,\n    "missed_payment_action": "IGNORE",\n    "recurrence_progress": 1,\n    "recharge": true,\n    "user_id": "5ed75086a883856178afc12e",\n    "external_id": "recurring_12345",\n    "payer_email": "test@x.co",\n    "description": "Test Curring Payment",\n    "amount": 100000,\n    "interval": "MONTH",\n    "interval_count": 2,\n    "start_date": "2020-07-08T08:22:55.815Z",\n    "currency": "IDR",\n    "created": "2020-07-08T08:22:55.817Z",\n    "updated": "2020-07-08T08:25:44.580Z",\n    "id": "5f05825ff9f52d3ed204c687"\n}\n```\n\n#### Resume Recurring Payment\n\n```python\nfrom xendit import RecurringPayment\n\nrecurring_payment = RecurringPayment.resume_recurring_payment(\n    id="5f05825ff9f52d3ed204c687",\n)\n```\n\nWill return\n\n```\n{\n    "status": "ACTIVE",\n    "should_send_email": false,\n    "missed_payment_action": "IGNORE",\n    "recurrence_progress": 1,\n    "recharge": true,\n    "user_id": "5ed75086a883856178afc12e",\n    "external_id": "recurring_12345",\n    "payer_email": "test@x.co",\n    "description": "Test Curring Payment",\n    "amount": 100000,\n    "interval": "MONTH",\n    "interval_count": 2,\n    "start_date": "2020-07-08T08:22:55.815Z",\n    "currency": "IDR",\n    "created": "2020-07-08T08:22:55.817Z",\n    "updated": "2020-07-08T08:26:03.082Z",\n    "id": "5f05825ff9f52d3ed204c687"\n}\n```\n\n### Disbursement Service\n\n#### Create Disbursement\n\n```python\nfrom xendit import Disbursement\n\ndisbursement = Disbursement.create(\n    external_id="demo_1475459775872",\n    bank_code="BCA",\n    account_holder_name="Bob Jones",\n    account_number="1231242311",\n    description="Reimbursement for shoes",\n    amount=17000,\n)\nprint(disbursement)\n```\n\nWill return\n\n```\n{\n    "user_id": "5ed75086a883856178afc12e",\n    "external_id": "demo_1475459775872",\n    "amount": 17000,\n    "bank_code": "BCA",\n    "account_holder_name": "Bob Jones",\n    "disbursement_description": "Reimbursement for shoes",\n    "status": "PENDING",\n    "id": "5ef1c4f40c2e150017ce3b96",\n}\n```\n\n#### Get Disbursement by ID\n\n```python\nfrom xendit import Disbursement\n\ndisbursement = Disbursement.get(\n    id="5ef1befeecb16100179e1d05",\n)\nprint(disbursement)\n```\n\nWill return\n\n```\n{\n    "user_id": "5ed75086a883856178afc12e",\n    "external_id": "demo_1475459775872",\n    "amount": 17000,\n    "bank_code": "BCA",\n    "account_holder_name": "Bob Jones",\n    "disbursement_description": "Disbursement from Postman",\n    "status": "PENDING",\n    "id": "5ef1befeecb16100179e1d05"\n}\n```\n\n#### Get Disbursement by External ID\n\n```python\nfrom xendit import Disbursement\n\ndisbursement_list = Disbursement.get_by_ext_id(\n    external_id="demo_1475459775872",\n)\nprint(disbursement_list)\n\n```\n\nWill return\n\n```\n[\n    {\n        "user_id": "5ed75086a883856178afc12e",\n        "external_id": "demo_1475459775872",\n        "amount": 17000,\n        "bank_code": "BCA",\n        "account_holder_name": "Bob Jones",\n        "disbursement_description": "Reimbursement for shoes",\n        "status": "PENDING",\n        "id": "5ef1c4f40c2e150017ce3b96",\n    },\n    {\n        "user_id": "5ed75086a883856178afc12e",\n        "external_id": "demo_1475459775872",\n        "amount": 17000,\n        "bank_code": "BCA",\n        "account_holder_name": "Bob Jones",\n        "disbursement_description": "Disbursement from Postman",\n        "status": "PENDING",\n        "id": "5ef1befeecb16100179e1d05",\n    },\n    ...\n]\n```\n#### Get Available Banks\n\n```python\nfrom xendit import Disbursement\n\ndisbursement_banks = Disbursement.get_available_banks()\nprint(disbursement_banks)\n```\n\nWill return\n\n```\n[\n    ...\n    {\n        "name": "Mandiri Taspen Pos (formerly Bank Sinar Harapan Bali)",\n        "code": "MANDIRI_TASPEN",\n        "can_disburse": True,\n        "can_name_validate": True,\n    },\n    {\n        "name": "Bank QNB Indonesia (formerly Bank QNB Kesawan)",\n        "code": "QNB_INDONESIA",\n        "can_disburse": True,\n        "can_name_validate": True,\n    }\n]\n```\n## Contributing\n\nFor any requests, bugs, or comments, please open an [issue](https://github.com/xendit/xendit-python/issues) or [submit a pull request](https://github.com/xendit/xendit-python/pulls).\n\nTo start developing on this repository, you need to have Poetry installed for package dependency. After that, you can run ```poetry install``` to install the dependency. To enter the environment, run ```poetry shell```\n\n### Tests\n\n#### Running the Test\n\nMake sure the the code passes all tests.\n\nRun the test:\n\n```\npython -m pytest tests/\n```\n\nRun with coverage:\n\n```\npython -m pytest tests/ --cov=xendit/\n```\n\n#### Creating Custom HTTP Client\n\nTo create your own HTTP Client, you can do it by implementing interface at `xendit/network/http_client_interface.py`. Our default HTTP Client are wrapper of [requests](https://github.com/psf/requests), which can be found at `xendit/network/_xendit_http_client.py`. To attach it to your instance, add it to your xendit parameter.\n\n```python\nimport xendit\n\nxendit_instance =  xendit.Xendit(api_key=\'\', http_client=YourHTTPClientClass)\n```\n',
    'author': 'Adyaksa Wisanggeni',
    'author_email': 'adyaksa@xendit.co',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/xendit/xendit-python',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
