# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['xendit',
 'xendit.models',
 'xendit.models.balance',
 'xendit.models.batchdisbursement',
 'xendit.models.cardlesscredit',
 'xendit.models.cardlesscredit.payment',
 'xendit.models.creditcard',
 'xendit.models.creditcard.charge',
 'xendit.models.directdebit',
 'xendit.models.directdebit.customer',
 'xendit.models.directdebit.payment',
 'xendit.models.directdebit.paymentmethod',
 'xendit.models.directdebit.token',
 'xendit.models.disbursement',
 'xendit.models.ewallet',
 'xendit.models.ewallet.dana',
 'xendit.models.ewallet.linkaja',
 'xendit.models.ewallet.ovo',
 'xendit.models.invoice',
 'xendit.models.payout',
 'xendit.models.qrcode',
 'xendit.models.recurringpayment',
 'xendit.models.retailoutlet',
 'xendit.models.virtualaccount',
 'xendit.models.xenplatform',
 'xendit.models.xenplatform.response',
 'xendit.network']

package_data = \
{'': ['*']}

install_requires = \
['requests>=2.23.0,<3.0.0']

setup_kwargs = {
    'name': 'xendit-python',
    'version': '0.1.6',
    'description': 'Xendit REST API Client for Python - Card, Virtual Account, Invoice, Disbursement, Recurring Payments, Payout, EWallet, Balance, Retail Outlets Services https://xendit.github.io/apireference/',
    'long_description': '<!-- omit in toc -->\n# Xendit Python Library\n\nThis library is the abstraction of Xendit API for access from applications written with Python.\n\n<!-- omit in toc -->\n## Table of Contents\n\n<!-- START doctoc generated TOC please keep comment here to allow auto update -->\n<!-- DON\'T EDIT THIS SECTION, INSTEAD RE-RUN doctoc TO UPDATE -->\n\n\n- [API Documentation](#api-documentation)\n- [Requirements](#requirements)\n- [Installation](#installation)\n- [Usage](#usage)\n  - [API Key](#api-key)\n    - [Global Variable](#global-variable)\n    - [Use Xendit Instance](#use-xendit-instance)\n  - [Headers](#headers)\n  - [Object Creation](#object-creation)\n  - [Using Custom HTTP Client](#using-custom-http-client)\n  - [Balance Service](#balance-service)\n    - [Get Balance](#get-balance)\n  - [Credit Card Service](#credit-card-service)\n    - [Create Authorization](#create-authorization)\n    - [Reverse Authorization](#reverse-authorization)\n    - [Create Charge](#create-charge)\n    - [Capture Charge](#capture-charge)\n    - [Get Charge](#get-charge)\n    - [Create Refund](#create-refund)\n    - [Create Promotion](#create-promotion)\n  - [eWallets Service](#ewallets-service)\n    - [Create OVO Payment](#create-ovo-payment)\n    - [Create DANA Payment](#create-dana-payment)\n    - [Create LinkAja Payment](#create-linkaja-payment)\n    - [Get Payment Status](#get-payment-status)\n  - [Cardless Credit Service](#cardless-credit-service)\n    - [Create Payment / Generate Checkout URL](#create-payment--generate-checkout-url)\n    - [Calculate Payment Types](#calculate-payment-types)\n  - [QR Codes Service](#qr-codes-service)\n    - [Create QR Code](#create-qr-code)\n    - [Get QR Code by External ID](#get-qr-code-by-external-id)\n  - [Direct Debit Service](#direct-debit-service)\n    - [Create Customer](#create-customer)\n    - [Get Customer by Reference ID](#get-customer-by-reference-id)\n    - [Initialize Linked Account Tokenization](#initialize-linked-account-tokenization)\n    - [Validate OTP for Linked Account Token](#validate-otp-for-linked-account-token)\n    - [Retrieve Accessible Accounts by Linked Account Token](#retrieve-accessible-accounts-by-linked-account-token)\n    - [Create Payment Method](#create-payment-method)\n    - [Get Payment Methods by Customer ID](#get-payment-methods-by-customer-id)\n    - [Create Direct Debit Payment](#create-direct-debit-payment)\n    - [Create Recurring Payment with Direct Debit](#create-recurring-payment-with-direct-debit)\n    - [Validate OTP for Direct Debit Payment](#validate-otp-for-direct-debit-payment)\n    - [Get Direct Debit Payment Status by ID](#get-direct-debit-payment-status-by-id)\n    - [Get Direct Debit Payment Status by Reference ID](#get-direct-debit-payment-status-by-reference-id)\n  - [Virtual Account Service](#virtual-account-service)\n    - [Create Virtual Account](#create-virtual-account)\n    - [Get Virtual Account Banks](#get-virtual-account-banks)\n    - [Get Virtual Account](#get-virtual-account)\n    - [Update Virtual Account](#update-virtual-account)\n    - [Get Virtual Account Payment](#get-virtual-account-payment)\n  - [Retail Outlet Service](#retail-outlet-service)\n    - [Create Fixed Payment Code](#create-fixed-payment-code)\n    - [Update Fixed Payment Code](#update-fixed-payment-code)\n    - [Get Fixed Payment Code](#get-fixed-payment-code)\n  - [Invoice Service](#invoice-service)\n    - [Create Invoice](#create-invoice)\n    - [Get Invoice](#get-invoice)\n    - [Expire Invoice](#expire-invoice)\n    - [List All Invoice](#list-all-invoice)\n  - [Recurring Payment Service](#recurring-payment-service)\n    - [Create Recurring Payment](#create-recurring-payment)\n    - [Get Recurring Payment](#get-recurring-payment)\n    - [Edit Recurring Payment](#edit-recurring-payment)\n    - [Stop Recurring Payment](#stop-recurring-payment)\n    - [Pause Recurring Payment](#pause-recurring-payment)\n    - [Resume Recurring Payment](#resume-recurring-payment)\n  - [Payout Service](#payout-service)\n    - [Create Payout](#create-payout)\n    - [Get Payout](#get-payout)\n    - [Void a Payout](#void-a-payout)\n  - [Disbursement Service](#disbursement-service)\n    - [Create Disbursement](#create-disbursement)\n    - [Get Disbursement by ID](#get-disbursement-by-id)\n    - [Get Disbursement by External ID](#get-disbursement-by-external-id)\n    - [Get Available Banks](#get-available-banks)\n  - [Batch Disbursement Service](#batch-disbursement-service)\n    - [Create Batch Disbursement](#create-batch-disbursement)\n    - [Get Batch Disbursement Available Banks](#get-batch-disbursement-available-banks)\n  - [xenPlatform Service](#xenplatform-service)\n    - [Create Account](#create-account)\n    - [Set Callback URLs](#set-callback-urls)\n    - [Transfers](#transfers)\n- [Contributing](#contributing)\n  - [Tests](#tests)\n    - [Running the Test](#running-the-test)\n\n<!-- END doctoc generated TOC please keep comment here to allow auto update -->\n\n## API Documentation\nPlease check [Xendit API Reference](https://xendit.github.io/apireference/).\n\n## Requirements\n\nPython 3.7 or later\n\n## Installation\n\nTo use the package, run ```pip install xendit-python```\n\n## Usage\n\n### API Key\n\nTo add API Key, you have 2 option: Use global variable or use Xendit instance\n\n#### Global Variable\n\n```python\nimport xendit\nxendit.api_key = "test-key123"\n\n# Then just run each class as static\nfrom xendit import Balance\nBalance.get()\n```\n\n#### Use Xendit Instance\n```python\nimport xendit\nx = xendit.Xendit(api_key="test-key123")\n\n# Then access each class from x attribute\nBalance = x.Balance\nBalance.get()\n```\n\n### Headers\n\nYou can add headers by using the following keyword parameters\n- X-IDEMPOTENCY-KEY: `x_idempotency_key`\n\n```\nVirtualAccount.create(x_idempotency_key="your-idemp-key")\n```\n\n- for-user-id: `for_user_id`\n\n```\nBalance.get(for_user_id=\'subaccount-user-id\')\n```\n\n- X-API-VERSION: `x_api_version`\n\n```\nBalance.get(x_api_version=\'2020-01-01\')\n```\n\n### Object Creation\n\nIf an API need an object as its parameter, you can use either dictionary for that class or a helper method e.g:\n\n```python\nitems = []\nitem = {\n    id: "123123",\n    name: "Phone Case",\n    price: 100000,\n    quantity: 1\n}\nitems.append(item)\nEWallet.create_linkaja_payment(\n    external_id="linkaja-ewallet-test-1593663498",\n    phone="089911111111",\n    items=items,\n    amount=300000,\n    callback_url="https://my-shop.com/callbacks",\n    redirect_url="https://xendit.co/",\n)\n```\n\nis equivalent with\n\n```python\nitems = []\nitem = EWallet.helper_create_linkaja_item(\n    id="123123", name="Phone Case", price=100000, quantity=1\n)\nitems.append(item)\nEWallet.create_linkaja_payment(\n    external_id="linkaja-ewallet-test-1593663498",\n    phone="089911111111",\n    items=items,\n    amount=300000,\n    callback_url="https://my-shop.com/callbacks",\n    redirect_url="https://xendit.co/",\n)\n```\n\n### Using Custom HTTP Client\n\nTo use your own HTTP Client, you can do it as long as your http client adhere to HTTP client interface at `xendit/network/http_client_interface.py`. For example, [requests](https://github.com/psf/requests) library are compatible with that interface, so we can freely use it in our library. To attach it to your instance, add it to your xendit parameter.\n\n```python\nimport xendit\n\nxendit_instance =  xendit.Xendit(api_key=\'\', http_client=YourHTTPClientClass)\n```\n\n### Balance Service\n\n#### Get Balance\n\nThe `account_type` parameter is optional.\n\n```python\nfrom xendit import Balance\nBalance.get()\n\nBalance.AccountType(\n    account_type=BalanceAccountType.CASH,\n)\n```\n\nUsage example:\n\n```python\nfrom xendit import Balance, BalanceAccountType\nBalance balance = Balance.get(\n    account_type=BalanceAccountType.CASH,\n)\n\n# To get the JSON view\nprint(balance)\n\n# To get only the value\nprint(balance.balance)\n``` \n\nWill return\n\n```\n{\'balance\': 1000000000}\n1000000000\n```\n\n### Credit Card Service\n\n#### Create Authorization\n\n```python\nfrom xendit import CreditCard\n\ncharge = CreditCard.create_authorization(\n    token_id="5f0410898bcf7a001a00879d",\n    external_id="card_preAuth-1594106356",\n    amount=75000,\n    card_cvn="123",\n)\nprint(charge)\n```\n\nWill return\n\n```\n{\n    "status": "AUTHORIZED",\n    "authorized_amount": 75000,\n    "capture_amount": 0,\n    "currency": "IDR",\n    "business_id": "5ed75086a883856178afc12e",\n    "merchant_id": "xendit_ctv_agg",\n    "merchant_reference_code": "5f0421faa98815a4f4c92a0d",\n    "external_id": "card_preAuth-1594106356",\n    "eci": "07",\n    "charge_type": "MULTIPLE_USE_TOKEN",\n    "masked_card_number": "400000XXXXXX0002",\n    "card_brand": "VISA",\n    "card_type": "CREDIT",\n    "descriptor": "XENDIT*XENDIT&AMP;#X27;S INTERN",\n    "bank_reconciliation_id": "5941063625146828103011",\n    "approval_code": "831000",\n    "created": "2020-07-07T07:19:22.921Z",\n    "id": "5f0421fa8cc1e8001973a1d6"\n}\n```\n\n#### Reverse Authorization\n\n```python\nfrom xendit import CreditCard\n\nreverse_authorization = CreditCard.reverse_authorizatiton(\n    credit_card_charge_id="5f0421fa8cc1e8001973a1d6",\n    external_id="reverse-authorization-1594106387",\n)\nprint(reverse_authorization)\n```\n\nWill return\n\n```\n{\n    "status": "SUCCEEDED",\n    "currency": "IDR",\n    "credit_card_charge_id": "5f0421fa8cc1e8001973a1d6",\n    "business_id": "5ed75086a883856178afc12e",\n    "external_id": "card_preAuth-1594106356",\n    "amount": 75000,\n    "created": "2020-07-07T07:19:48.896Z",\n    "id": "5f0422148cc1e8001973a1dc"\n}\n```\n\n#### Create Charge\n\n```python\nfrom xendit import CreditCard\n\ncharge = CreditCard.create_charge(\n    token_id="5f0410898bcf7a001a00879d",\n    external_id="card_charge-1594106478",\n    amount=75000,\n    card_cvn="123",\n)\nprint(charge)\n```\n\nWill return\n\n```\n{\n    "status": "CAPTURED",\n    "authorized_amount": 75000,\n    "capture_amount": 75000,\n    "currency": "IDR",\n    "business_id": "5ed75086a883856178afc12e",\n    "merchant_id": "xendit_ctv_agg",\n    "merchant_reference_code": "5f0422746fc1d25bd222df2e",\n    "external_id": "card_charge-1594106478",\n    "eci": "07",\n    "charge_type": "MULTIPLE_USE_TOKEN",\n    "masked_card_number": "400000XXXXXX0002",\n    "card_brand": "VISA",\n    "card_type": "CREDIT",\n    "descriptor": "XENDIT*XENDIT&AMP;#X27;S INTERN",\n    "bank_reconciliation_id": "5941064846646923303008",\n    "approval_code": "831000",\n    "created": "2020-07-07T07:21:25.027Z",\n    "id": "5f0422752bbbe50019a368b5"\n}\n```\n\n#### Capture Charge\n\n```python\nfrom xendit import CreditCard\n\ncharge = CreditCard.capture_charge(\n    credit_card_charge_id="5f0422aa2bbbe50019a368c2",\n    amount=75000,\n)\nprint(charge)\n```\n\nWill return\n\n```\n{\n    "status": "CAPTURED",\n    "authorized_amount": 75000,\n    "capture_amount": 75000,\n    "currency": "IDR",\n    "created": "2020-07-07T07:22:18.719Z",\n    "business_id": "5ed75086a883856178afc12e",\n    "merchant_id": "xendit_ctv_agg",\n    "merchant_reference_code": "5f0422aa6fc1d25bd222df32",\n    "external_id": "card_preAuth-1594106532",\n    "eci": "07",\n    "charge_type": "MULTIPLE_USE_TOKEN",\n    "masked_card_number": "400000XXXXXX0002",\n    "card_brand": "VISA",\n    "card_type": "CREDIT",\n    "descriptor": "XENDIT*XENDIT&AMP;#X27;S INTERN",\n    "bank_reconciliation_id": "5941065383296525603007",\n    "approval_code": "831000",\n    "mid_label": "CTV_TEST",\n    "id": "5f0422aa2bbbe50019a368c2"\n}\n```\n\n#### Get Charge\n\n```python\nfrom xendit import CreditCard\n\ncharge = CreditCard.get_charge(\n    credit_card_charge_id="5f0422aa2bbbe50019a368c2",\n)\nprint(charge)\n```\n\nWill return\n\n```\n{\n    "status": "CAPTURED",\n    "authorized_amount": 75000,\n    "capture_amount": 75000,\n    "currency": "IDR",\n    "created": "2020-07-07T07:22:18.719Z",\n    "business_id": "5ed75086a883856178afc12e",\n    "merchant_id": "xendit_ctv_agg",\n    "merchant_reference_code": "5f0422aa6fc1d25bd222df32",\n    "external_id": "card_preAuth-1594106532",\n    "eci": "07",\n    "charge_type": "MULTIPLE_USE_TOKEN",\n    "masked_card_number": "400000XXXXXX0002",\n    "card_brand": "VISA",\n    "card_type": "CREDIT",\n    "descriptor": "XENDIT*XENDIT&AMP;#X27;S INTERN",\n    "bank_reconciliation_id": "5941065383296525603007",\n    "approval_code": "831000",\n    "mid_label": "CTV_TEST",\n    "metadata": {},\n    "id": "5f0422aa2bbbe50019a368c2"\n}\n```\n\n#### Create Refund\n\n```python\nfrom xendit import CreditCard\n\nrefund = CreditCard.create_refund(\n    credit_card_charge_id="5f0422aa2bbbe50019a368c2",\n    amount=10000,\n    external_id="card_refund-1594106755",\n)\nprint(refund)\n```\n\nWill return\n\n```\n{\n    "status": "REQUESTED",\n    "currency": "IDR",\n    "credit_card_charge_id": "5f0422aa2bbbe50019a368c2",\n    "user_id": "5ed75086a883856178afc12e",\n    "amount": 10000,\n    "external_id": "card_refund-1594106755",\n    "created": "2020-07-07T07:25:56.872Z",\n    "updated": "2020-07-07T07:25:57.740Z",\n    "id": "5f0423848bb8da600c57c44f",\n    "fee_refund_amount": 290\n}\n```\n\n#### Create Promotion\n\n```python\nfrom xendit import CreditCard\n\npromotion = CreditCard.create_promotion(\n    reference_id="BRI_20_JAN-1594176600",\n    description="20% discount applied for all BRI cards",\n    discount_amount=10000,\n    bin_list=[\'400000\', \'460000\'],\n    start_time="2020-01-01T00:00:00.000Z",\n    end_time="2021-01-01T00:00:00.000Z",\n)\nprint(promotion)\n```\n\nWill return\n\n```\n{\n    "business_id": "5ed75086a883856178afc12e",\n    "reference_id": "BRI_20_JAN-1594176600",\n    "description": "20% discount applied for all BRI cards",\n    "start_time": "2020-01-01T00:00:00.000Z",\n    "end_time": "2021-01-01T00:00:00.000Z",\n    "type": "CARD_BIN",\n    "discount_amount": 10000,\n    "bin_list": [\n        "400000",\n        "460000"\n    ],\n    "currency": "IDR",\n    "id": "c65a2ae7-ce75-4a15-bbec-55d076f46bd0",\n    "created": "2020-07-08T02:50:02.296Z",\n    "status": "ACTIVE"\n}\n```\n\n### eWallets Service\n\n#### Create E-Wallet Charge\n```python\nfrom xendit import EWallet\n\nbasket = []\nbasket_item = EWallet.helper_create_basket_item(\n    reference_id = "basket-product-ref-id",\n    name = "product_name",\n    category = "product_category",\n    currency = "IDR",\n    price = 50000,\n    quantity = 5,\n    type = "product_type",\n    sub_category = "product_sub_category",\n    metadata = {\n        "meta": "data"\n    }\n)\nbasket.append(basket_item)\n\newallet_charge = EWallet.create_ewallet_charge(\n    reference_id="basket-product-ref-id",\n    currency="IDR",\n    amount=10000,\n    checkout_method="ONE_TIME_PAYMENT",\n    channel_code="ID_SHOPEEPAY",\n    channel_properties={\n        "success_redirect_url": "https://yourwebsite.com/order/123",\n    },\n    basket=basket,\n)\n```\n\nWill return\n\n```json\n{\n    "id": "ewc_f3925450-5c54-4777-98c1-fcf22b0d1e1c",\n    "business_id": "5ed75086a883856178afc12e",\n    "reference_id": "basket-product-ref-id",\n    "status": "PENDING",\n    "currency": "IDR",\n    "charge_amount": 10000,\n    "capture_amount": 10000,\n    "checkout_method": "ONE_TIME_PAYMENT",\n    "channel_code": "ID_SHOPEEPAY",\n    "channel_properties": {\n        "success_redirect_url": "https://yourwebsite.com/order/123",\n    },\n    "actions": {\n        "desktop_web_checkout_url": null,\n        "mobile_web_checkout_url": null,\n        "mobile_deeplink_checkout_url": "https://ewallet-mock-connector.xendit.co/v1/ewallet_connector/checkouts?token=ZjQyOGIzMGVlNGFjOWJhNTE2YWQ3NGQyN2RiMTIwODg6ZTY2YjA2YjQ1ZjJlZWI0NDA4OGNjODg2NGFlYzQ2N2U5YTI5MjM3ODUzODViYzljNjQyYWYwOGExMjU4MzdmMTc3NDFlMWZmYjcxN2MzOWZiYmMyNjY4N2ViNmMxM2ZkMjg1ZmIzZDM5ZmZiZDYzM2ViNGMyMDRkOWM3ZTUzNWUyMDBlOWUzMzdhZTkwZjllZjQwZjQyMjExOTkyNWQ2MTg2YzgzZTQ3N2JhYWZkNDFhN2U0MWM1ZDMzMGJmMmNhNzhiMjhmMmY5ZDBjZDQ4MjlkODA3MjE5YWQzYTlhNTE0YmM1NjUzYjljMmZlOWU1YjMwM2FmNTZiNmViNGVlZDIxODQzNzdjNDJmYjRmNzBmZDZlZDhlM2MyMGM4YmExY2RmNTVkOTdjZmU3MWYxMWVmMDYzMmQzNGE1ZTFmMzE=",\n        "qr_checkout_string": "test-qr-string",\n    },\n    "is_redirect_required": true,\n    "callback_url": "https://yourwebsite.com/order/123",\n    "created": "2021-02-09T06:22:35.064408Z",\n    "updated": "2021-02-09T06:22:35.064408Z",\n    "voided_at": null,\n    "capture_now": true,\n    "customer_id": null,\n    "payment_method_id": null,\n    "failure_code": null,\n    "basket": [\n        {\n            "reference_id": "basket-product-ref-id",\n            "name": "product_name",\n            "category": "product_category",\n            "currency": "IDR",\n            "price": 50000,\n            "quantity": 5,\n            "type": "product_type",\n            "url": "",\n            "description": "",\n            "sub_category": "product_sub_category",\n            "metadata": {\n                "meta": "data"\n            }\n        }\n    ],\n    "metadata": null,\n}\n```\n\n#### Get E-Wallet Charge Status\n```python\nfrom xendit import EWallet\n\newallet_charge = EWallet.get_ewallet_charge_status(\n    charge_id="ewc_f3925450-5c54-4777-98c1-fcf22b0d1e1c",\n)\n```\n\nWill return\n\n```json\n{\n    "id": "ewc_f3925450-5c54-4777-98c1-fcf22b0d1e1c",\n    "business_id": "5ed75086a883856178afc12e",\n    "reference_id": "basket-product-ref-id",\n    "status": "PENDING",\n    "currency": "IDR",\n    "charge_amount": 10000,\n    "capture_amount": 10000,\n    "checkout_method": "ONE_TIME_PAYMENT",\n    "channel_code": "ID_SHOPEEPAY",\n    "channel_properties": {\n        "success_redirect_url": "https://yourwebsite.com/order/123",\n    },\n    "actions": {\n        "desktop_web_checkout_url": null,\n        "mobile_web_checkout_url": null,\n        "mobile_deeplink_checkout_url": "https://ewallet-mock-connector.xendit.co/v1/ewallet_connector/checkouts?token=ZjQyOGIzMGVlNGFjOWJhNTE2YWQ3NGQyN2RiMTIwODg6ZTY2YjA2YjQ1ZjJlZWI0NDA4OGNjODg2NGFlYzQ2N2U5YTI5MjM3ODUzODViYzljNjQyYWYwOGExMjU4MzdmMTc3NDFlMWZmYjcxN2MzOWZiYmMyNjY4N2ViNmMxM2ZkMjg1ZmIzZDM5ZmZiZDYzM2ViNGMyMDRkOWM3ZTUzNWUyMDBlOWUzMzdhZTkwZjllZjQwZjQyMjExOTkyNWQ2MTg2YzgzZTQ3N2JhYWZkNDFhN2U0MWM1ZDMzMGJmMmNhNzhiMjhmMmY5ZDBjZDQ4MjlkODA3MjE5YWQzYTlhNTE0YmM1NjUzYjljMmZlOWU1YjMwM2FmNTZiNmViNGVlZDIxODQzNzdjNDJmYjRmNzBmZDZlZDhlM2MyMGM4YmExY2RmNTVkOTdjZmU3MWYxMWVmMDYzMmQzNGE1ZTFmMzE=",\n        "qr_checkout_string": "test-qr-string",\n    },\n    "is_redirect_required": true,\n    "callback_url": "https://yourwebsite.com/order/123",\n    "created": "2021-02-09T06:22:35.064408Z",\n    "updated": "2021-02-09T06:22:35.064408Z",\n    "voided_at": null,\n    "capture_now": true,\n    "customer_id": null,\n    "payment_method_id": null,\n    "failure_code": null,\n    "basket": [\n        {\n            "reference_id": "basket-product-ref-id",\n            "name": "product_name",\n            "category": "product_category",\n            "currency": "IDR",\n            "price": 50000,\n            "quantity": 5,\n            "type": "product_type",\n            "url": "",\n            "description": "",\n            "sub_category": "product_sub_category",\n            "metadata": {\n                "meta": "data"\n            }\n        }\n    ],\n    "metadata": null,\n}\n```\n\n### Cardless Credit Service\n\n#### Create Payment / Generate Checkout URL\n\n```python\nfrom xendit import CardlessCredit, CardlessCreditType\n\ncardless_credit_items = []\ncardless_credit_items.append(\n    CardlessCredit.helper_create_item(\n        id="item-123",\n        name="Phone Case",\n        price=200000,\n        type="Smartphone",\n        url="http://example.com/phone/phone_case",\n        quantity=2,\n    )\n)\ncustomer_details = CardlessCredit.helper_create_customer_details(\n    first_name="customer first name",\n    last_name="customer last name",\n    email="customer@email.com",\n    phone="0812332145",\n)\nshipping_address = CardlessCredit.helper_create_shipping_address(\n    first_name="first name",\n    last_name="last name",\n    address="Jl Teknologi No. 12",\n    city="Jakarta",\n    postal_code="12345",\n    phone="081513114262",\n    country_code="IDN",\n)\ncardless_credit_payment = CardlessCredit.create_payment(\n    cardless_credit_type=CardlessCreditType.KREDIVO,\n    external_id="id-1595923113",\n    amount=10000,\n    payment_type="3_months",\n    items=cardless_credit_items,\n    customer_details=customer_details,\n    shipping_address=shipping_address,\n    redirect_url="https://my-shop.com/home",\n    callback_url="https://my-shop.com/callback",\n)\nprint(cardless_credit_payment)\n```\n\nWill return\n\n```\n{\n    "redirect_url": "https://pay-sandbox.kredivo.com/signIn?tk=26458cdf-660c-4491-a1de-bb6e63312d8a",\n    "order_id": "e8ae4066-7980-499f-b92c-eb3a587782c1",\n    "external_id": "id-1595923113",\n    "cardless_credit_type": "KREDIVO"\n}\n```\n\n#### Calculate Payment Types\n\n```python\nfrom xendit import CardlessCredit, CardlessCreditType\n\ncardless_credit_items = []\ncardless_credit_items.append(\n    CardlessCredit.helper_create_item(\n        id="item-123",\n        name="Phone Case",\n        price=200000,\n        type="Smartphone",\n        url="http://example.com/phone/phone_case",\n        quantity=2,\n    )\n)\ncardless_credit_payment_types = CardlessCredit.calculate_payment_type(\n    cardless_credit_type=CardlessCreditType.KREDIVO,\n    amount=10000,\n    items=cardless_credit_items,\n)\nprint(cardless_credit_payment_types)\n```\n\nWill return\n\n```\n{\n    "message": "Available payment types are listed.",\n    "payments": [\n        {\n            "raw_monthly_installment": 401000,\n            "name": "Bayar dalam 30 hari",\n            "amount": 401000,\n            "installment_amount": 401000,\n            "raw_amount": 401000,\n            "rate": 0,\n            "down_payment": 0,\n            "monthly_installment": 401000,\n            "discounted_monthly_installment": 0,\n            "tenure": 1,\n            "id": "30_days"\n        }\n    ]\n}\n```\n\n### QR Codes Service\n\n#### Create QR Code\n\n```python\nfrom xendit import QRCode, QRCodeType\n\nqrcode = QRCode.create(\n    external_id="qrcode-id-1594794038",\n    type=QRCodeType.DYNAMIC,\n    callback_url="https://webhook.site",\n    amount=4000,\n)\nprint(qrcode)\n```\n\nWill return\n\n```\n{\n    "id": "qr_13c31ddd-9d58-449b-9f52-1bf5123a45b5",\n    "external_id": "qrcode-id-1594794038",\n    "amount": 4000,\n    "qr_string": "00020101021226660014ID.LINKAJA.WWW011893600911002411480002152004230411480010303UME51450015ID.OR.GPNQR.WWW02150000000000000000303UME520454995802ID5920Placeholder merchant6007Jakarta610612345662380115wLoc6DRGwAOgSkZ0715wLoc6DRGwAOgSkZ53033605404400063047668",\n    "callback_url": "https://webhook.site",\n    "type": "DYNAMIC",\n    "status": "ACTIVE",\n    "created": "2020-07-15T06:20:40.636Z",\n    "updated": "2020-07-15T06:20:40.636Z"\n}\n```\n\n#### Get QR Code by External ID\n\n```python\nfrom xendit import QRCode\n\nqrcode = QRCode.get_by_ext_id(\n    external_id="qrcode-id-1594794038",\n)\nprint(qrcode)\n```\n\nWill return\n\n```\n{\n    "id": "qr_13c31ddd-9d58-449b-9f52-1bf5123a45b5",\n    "external_id": "qrcode-id-1594794038",\n    "amount": 4000,\n    "qr_string": "00020101021226660014ID.LINKAJA.WWW011893600911002411480002152004230411480010303UME51450015ID.OR.GPNQR.WWW02150000000000000000303UME520454995802ID5920Placeholder merchant6007Jakarta610612345662380115wLoc6DRGwAOgSkZ0715wLoc6DRGwAOgSkZ53033605404400063047668",\n    "callback_url": "https://webhook.site",\n    "type": "DYNAMIC",\n    "status": "ACTIVE",\n    "created": "2020-07-15T06:20:40.636Z",\n    "updated": "2020-07-15T06:20:40.636Z"\n}\n```\n\n### Direct Debit Service\n\n#### Create Customer\n\n```python\nfrom xendit import DirectDebit\n\ncustomer = DirectDebit.create_customer(\n    reference_id="merc-1594279037",\n    email="t@x.co",\n    given_names="Adyaksa",\n)\nprint(customer)\n```\n\nWill return\n\n```\n{\n    "id": "ed20b5db-df04-41fc-8018-8ea4ac4d1030",\n    "reference_id": "merc-1594279037",\n    "description": null,\n    "given_names": "Adyaksa",\n    "middle_name": null,\n    "surname": null,\n    "mobile_number": null,\n    "phone_number": null,\n    "email": "t@x.co",\n    "nationality": null,\n    "addresses": null,\n    "date_of_birth": null,\n    "employment": null,\n    "source_of_wealth": null,\n    "metadata": null\n}\n```\n\n#### Get Customer by Reference ID\n\n```python\nfrom xendit import DirectDebit\n\ncustomer = DirectDebit.get_customer_by_ref_id(\n    reference_id="merc-1594279037",\n)\nprint(customer)\n```\n\nWill return\n\n```\n[{\n    "id": "ed20b5db-df04-41fc-8018-8ea4ac4d1030",\n    "reference_id": "merc-1594279037",\n    "description": null,\n    "given_names": "Adyaksa",\n    "middle_name": null,\n    "surname": null,\n    "mobile_number": null,\n    "phone_number": null,\n    "email": "t@x.co",\n    "nationality": null,\n    "addresses": null,\n    "date_of_birth": null,\n    "employment": null,\n    "source_of_wealth": null,\n    "metadata": null\n}]\n```\n\n#### Initialize Linked Account Tokenization\n\n```python\nfrom xendit import DirectDebit\n\ncard_linking = DirectDebit.helper_create_card_link(\n    account_mobile_number="+62818555988",\n    card_last_four="8888",\n    card_expiry="06/24",\n    account_email="test.email@xendit.co",\n)\nlinked_account = DirectDebit.initialize_tokenization(\n    customer_id="ed20b5db-df04-41fc-8018-8ea4ac4d1030",\n    channel_code="DC_BRI",\n    properties=card_linking,   \n)\nprint(linked_account)\n```\n\nWill return\n\n```\n{\n    "id": "lat-f325b757-0aae-4c24-92c5-3661e299e154",\n    "customer_id": "ed20b5db-df04-41fc-8018-8ea4ac4d1030",\n    "channel_code": "DC_BRI",\n    "authorizer_url": null,\n    "status": "PENDING",\n    "metadata": null\n}\n```\n\n#### Validate OTP for Linked Account Token\n\n```python\nfrom xendit import DirectDebit\n\nlinked_account = DirectDebit.validate_token_otp(\n    linked_account_token_id="lat-f325b757-0aae-4c24-92c5-3661e299e154",\n    otp_code="333000",\n)\nprint(linked_account)\n```\n\nWill return\n\n```\n{\n    "id": "lat-f325b757-0aae-4c24-92c5-3661e299e154",\n    "customer_id": "ed20b5db-df04-41fc-8018-8ea4ac4d1030",\n    "channel_code": "DC_BRI",\n    "status": "SUCCESS",\n    "metadata": null\n}\n```\n\n#### Retrieve Accessible Accounts by Linked Account Token\n\n```python\nfrom xendit import DirectDebit\n\naccessible_accounts = DirectDebit.get_accessible_account_by_token(\n    linked_account_token_id="lat-f325b757-0aae-4c24-92c5-3661e299e154",\n)\nprint(accessible_accounts)\n```\n\nWill return\n\n```\n[{\n    "channel_code": "DC_BRI",\n    "id": "la-08b089e8-7035-4f5f-bdd9-94edd9dc9480",\n    "properties": {\n        "card_expiry": "06/24",\n        "card_last_four": "8888",\n        "currency": "IDR",\n        "description": ""\n    },\n    "type": "DEBIT_CARD"\n}]\n```\n\n#### Create Payment Method\n\n```python\nfrom xendit import DirectDebit, DirectDebitPaymentMethodType\n\npayment_method = DirectDebit.create_payment_method(\n    customer_id="ed20b5db-df04-41fc-8018-8ea4ac4d1030",\n    type=DirectDebitPaymentMethodType.DEBIT_CARD,\n    properties={\'id\': \'la-fac7e744-ab40-4100-a447-cbbb16f29ded\'},\n)\n\nprint(payment_method)\n```\n\nWill return\n\n```\n{\n    "customer_id": "ed20b5db-df04-41fc-8018-8ea4ac4d1030",\n    "type": "DEBIT_CARD",\n    "properties": {\n        "id": "la-fac7e744-ab40-4100-a447-cbbb16f29ded",\n        "currency": "IDR",\n        "card_expiry": "06/24",\n        "description": "",\n        "channel_code": "DC_BRI",\n        "card_last_four": "8888"\n    },\n    "status": "ACTIVE",\n    "metadata": {},\n    "id": "pm-b6116aea-8c23-42d0-a1e6-33227b52fccd",\n    "created": "2020-07-13T07:28:57.716Z",\n    "updated": "2020-07-13T07:28:57.716Z"\n}\n```\n\n#### Get Payment Methods by Customer ID\n\n```python\nfrom xendit import DirectDebit\n\npayment_methods = DirectDebit.get_payment_methods_by_customer_id(\n    customer_id="ed20b5db-df04-41fc-8018-8ea4ac4d1030",\n)\n\nprint(payment_methods)\n```\n\nWill return\n\n```\n[{\n    "id": "pm-b6116aea-8c23-42d0-a1e6-33227b52fccd",\n    "customer_id": "ed20b5db-df04-41fc-8018-8ea4ac4d1030",\n    "status": "ACTIVE",\n    "type": "DEBIT_CARD",\n    "properties": {\n        "id": "la-fac7e744-ab40-4100-a447-cbbb16f29ded",\n        "currency": "IDR",\n        "card_expiry": "06/24",\n        "description": "",\n        "channel_code": "DC_BRI",\n        "card_last_four": "8888"\n    },\n    "metadata": {},\n    "created": "2020-07-13T07:28:57.716Z",\n    "updated": "2020-07-13T07:28:57.716Z"\n}]\n```\n\n#### Create Direct Debit Payment\n\n```python\nfrom xendit import DirectDebit\n\npayment = DirectDebit.create_payment(\n    reference_id="direct-debit-ref-1594718940",\n    payment_method_id="pm-b6116aea-8c23-42d0-a1e6-33227b52fccd",\n    currency="IDR",\n    amount="60000",\n    callback_url="http://webhook.site/",\n    enable_otp=True,\n    idempotency_key="idemp_key-1594718940",\n)\n\nprint(payment)\n```\n\nWill return\n\n```\n{\n    "failure_code": null,\n    "otp_mobile_number": null,\n    "otp_expiration_timestamp": null,\n    "id": "ddpy-eaa093b6-b669-401a-ba2e-61ac644b2aff",\n    "reference_id": "direct-debit-ref-1594718940",\n    "payment_method_id": "pm-b6116aea-8c23-42d0-a1e6-33227b52fccd",\n    "channel_code": "DC_BRI",\n    "currency": "IDR",\n    "amount": 60000,\n    "is_otp_required": true,\n    "basket": null,\n    "description": "",\n    "status": "PENDING",\n    "metadata": null,\n    "created": "2020-07-14T09:29:02.614443Z",\n    "updated": "2020-07-14T09:29:02.614443Z"\n}\n```\n\n#### Create Recurring Payment with Direct Debit\n\nYou can use [Create Recurring Payment](#create-recurring-payment) to use this feature.\n\n#### Validate OTP for Direct Debit Payment\n\n```python\nfrom xendit import DirectDebit\n\npayment = DirectDebit.validate_payment_otp(\n    direct_debit_id="ddpy-eaa093b6-b669-401a-ba2e-61ac644b2aff",\n    otp_code="222000",\n)\n\nprint(payment)\n```\n\nWill return\n\n```\n{\n    "failure_code": null,\n    "otp_mobile_number": null,\n    "otp_expiration_timestamp": null,\n    "id": "ddpy-eaa093b6-b669-401a-ba2e-61ac644b2aff",\n    "reference_id": "direct-debit-ref-1594718940",\n    "payment_method_id": "pm-b6116aea-8c23-42d0-a1e6-33227b52fccd",\n    "channel_code": "DC_BRI",\n    "currency": "IDR",\n    "amount": 60000,\n    "is_otp_required": true,\n    "basket": null,\n    "description": "",\n    "status": "PENDING",\n    "metadata": null,\n    "created": "2020-07-14T09:29:02.614443Z",\n    "updated": "2020-07-14T09:29:02.614443Z"\n}\n```\n\n#### Get Direct Debit Payment Status by ID\n\n```python\nfrom xendit import DirectDebit\n\npayment = DirectDebit.get_payment_status(\n    direct_debit_id="ddpy-38ef50a8-00f0-4019-8b28-9bca81f2cbf1",\n)\n\nprint(payment)\n```\n\nWill return\n\n```\n{\n    "failure_code": null,\n    "otp_mobile_number": null,\n    "otp_expiration_timestamp": null,\n    "id": "ddpy-38ef50a8-00f0-4019-8b28-9bca81f2cbf1",\n    "reference_id": "direct-debit-ref-1594717458",\n    "payment_method_id": "pm-b6116aea-8c23-42d0-a1e6-33227b52fccd",\n    "channel_code": "DC_BRI",\n    "currency": "IDR",\n    "amount": 60000,\n    "is_otp_required": false,\n    "basket": null,\n    "description": "",\n    "status": "PENDING",\n    "metadata": null,\n    "created": "2020-07-14T09:04:20.031451Z",\n    "updated": "2020-07-14T09:04:20.031451Z"\n}\n```\n\n#### Get Direct Debit Payment Status by Reference ID\n\n```python\nfrom xendit import DirectDebit\n\npayments = DirectDebit.get_payment_status_by_ref_id(\n    reference_id="direct-debit-ref-1594717458",\n)\n\nprint(payments)\n```\n\nWill return\n\n```\n[{\n    "amount": 60000,\n    "basket": null,\n    "channel_code": "DC_BRI",\n    "created": "2020-07-14T09:04:20.031451Z",\n    "currency": "IDR",\n    "description": "",\n    "failure_code": null,\n    "id": "ddpy-38ef50a8-00f0-4019-8b28-9bca81f2cbf1",\n    "is_otp_required": false,\n    "metadata": null,\n    "otp_expiration_timestamp": null,\n    "otp_mobile_number": null,\n    "payment_method_id": "pm-b6116aea-8c23-42d0-a1e6-33227b52fccd",\n    "reference_id": "direct-debit-ref-1594717458",\n    "status": "PENDING",\n    "updated": "2020-07-14T09:04:20.031451Z"\n}]\n```\n\n### Virtual Account Service\n\n#### Create Virtual Account\n\n```python\nfrom xendit import VirtualAccount\n\nvirtual_account = VirtualAccount.create(\n    external_id="demo_1475459775872",\n    bank_code="BNI",\n    name="Rika Sutanto",\n)\nprint(virtual_account)\n```\n\nWill return\n\n```\n{\n    "owner_id": "5ed75086a883856178afc12e",\n    "external_id": "demo_1475459775872",\n    "bank_code": "BNI",\n    "merchant_code": "8808",\n    "name": "Rika Sutanto",\n    "account_number": "8808999956275653",\n    "is_single_use": false,\n    "status": "PENDING",\n    "expiration_date": "2051-06-22T17:00:00.000Z",\n    "is_closed": false,\n    "id": "5ef174c48dd9ea2fc97d6a1e"\n}\n```\n\n#### Get Virtual Account Banks\n```python\nfrom xendit import VirtualAccount\n\nvirtual_account_banks = VirtualAccount.get_banks()\nprint(virtual_account_banks)\n```\n\nWill return\n\n```\n[{\n    "name": "Bank Mandiri",\n    "code": "MANDIRI"\n}, {\n    "name": "Bank Negara Indonesia",\n    "code": "BNI"\n}, {\n    "name": "Bank Rakyat Indonesia",\n    "code": "BRI"\n}, {\n    "name": "Bank Permata",\n    "code": "PERMATA"\n}, {\n    "name": "Bank Central Asia",\n    "code": "BCA"\n}]\n```\n#### Get Virtual Account\n\n```python\nfrom xendit import VirtualAccount\n\nvirtual_account = VirtualAccount.get(\n    id="5eec3a3e8dd9ea2fc97d6728",\n)\nprint(virtual_account)\n```\n\nWill return\n\n```\n{\n    "owner_id": "5ed75086a883856178afc12e",\n    "external_id": "demo_1475459775872",\n    "bank_code": "BNI",\n    "merchant_code": "8808",\n    "name": "Rika Sutanto",\n    "account_number": "8808999917965673",\n    "is_single_use": true,\n    "status": "ACTIVE",\n    "expiration_date": "2051-06-18T17:00:00.000Z",\n    "is_closed": false,\n    "id": "5eec3a3e8dd9ea2fc97d6728"\n}\n```\n\n#### Update Virtual Account\n\n```python\nfrom xendit import VirtualAccount\n\nvirtual_account = VirtualAccount.update(\n    id="5eec3a3e8dd9ea2fc97d6728",\n    is_single_use=True,\n)\nprint(virtual_account)\n```\n\nWill return\n\n```\n{\n    "owner_id": "5ed75086a883856178afc12e",\n    "external_id": "demo_1475459775872",\n    "bank_code": "BNI",\n    "merchant_code": "8808",\n    "name": "Rika Sutanto",\n    "account_number": "8808999917965673",\n    "is_single_use": true,\n    "status": "PENDING",\n    "expiration_date": "2051-06-18T17:00:00.000Z",\n    "is_closed": false,\n    "id": "5eec3a3e8dd9ea2fc97d6728"\n}\n```\n\n#### Get Virtual Account Payment\n\n```python\nfrom xendit import VirtualAccount\n\nvirtual_account_payment = VirtualAccount.get_payment(\n    payment_id="5ef18efca7d10d1b4d61fb52",\n)\nprint(virtual_account_payment)\n```\n\nWill return\n\n```\n{\n    "id": "5ef18efcf9ce3b5f8e188ee4",\n    "payment_id": "5ef18efca7d10d1b4d61fb52",\n    "callback_virtual_account_id": "5ef18ece8dd9ea2fc97d6a84",\n    "external_id": "VA_fixed-1592889038",\n    "merchant_code": "88608",\n    "account_number": "9999317837",\n    "bank_code": "MANDIRI",\n    "amount": 50000,\n    "transaction_timestamp": "2020-06-23T05:11:24.000Z"\n}\n```\n\n### Retail Outlet Service\n\n#### Create Fixed Payment Code\n\n```python\nfrom xendit import RetailOutlet\n\nretail_outlet = RetailOutlet.create_fixed_payment_code(\n    external_id="demo_fixed_payment_code_123",\n    retail_outlet_name="ALFAMART",\n    name="Rika Sutanto",\n    expected_amount=10000,\n)\nprint(retail_outlet)\n```\n\nWill return\n\n```\n{\n    "owner_id": "5ed75086a883856178afc12e",\n    "external_id": "demo_fixed_payment_code_123",\n    "retail_outlet_name": "ALFAMART",\n    "prefix": "TEST",\n    "name": "Rika Sutanto",\n    "payment_code": "TEST56147",\n    "expected_amount": 10000,\n    "is_single_use": False,\n    "expiration_date": "2051-06-23T17:00:00.000Z",\n    "id": "5ef2f0f8e7f5c14077275493",\n}\n```\n\n#### Update Fixed Payment Code\n\n```python\nfrom xendit import RetailOutlet\n\nretail_outlet = RetailOutlet.update_fixed_payment_code(\n    fixed_payment_code_id="5ef2f0f8e7f5c14077275493",\n    name="Joe Contini",\n)\nprint(retail_outlet)\n```\n\nWill return\n\n```\n{\n    "owner_id": "5ed75086a883856178afc12e",\n    "external_id": "demo_fixed_payment_code_123",\n    "retail_outlet_name": "ALFAMART",\n    "prefix": "TEST",\n    "name": "Joe Contini",\n    "payment_code": "TEST56147",\n    "expected_amount": 10000,\n    "is_single_use": False,\n    "expiration_date": "2051-06-23T17:00:00.000Z",\n    "id": "5ef2f0f8e7f5c14077275493",\n}\n```\n\n#### Get Fixed Payment Code\n\n```python\nfrom xendit import RetailOutlet\n\nretail_outlet = RetailOutlet.get_fixed_payment_code(\n    fixed_payment_code_id="5ef2f0f8e7f5c14077275493",\n)\nprint(retail_outlet)\n```\n\nWill return\n\n```\n{\n    "owner_id": "5ed75086a883856178afc12e",\n    "external_id": "demo_fixed_payment_code_123",\n    "retail_outlet_name": "ALFAMART",\n    "prefix": "TEST",\n    "name": "Rika Sutanto",\n    "payment_code": "TEST56147",\n    "expected_amount": 10000,\n    "is_single_use": False,\n    "expiration_date": "2051-06-23T17:00:00.000Z",\n    "id": "5ef2f0f8e7f5c14077275493",\n}\n```\n\n### Invoice Service\n\n#### Create Invoice\n\n```python\nfrom xendit import Invoice\n\ninvoice = Invoice.create(\n    external_id="invoice-1593684000",\n    amount=20000,\n    payer_email="customer@domain.com",\n    description="Invoice Demo #123",\n)\nprint(invoice)\n```\n\nWill return\n\n```\n{\n    "id": "5efdb0210425db620ec35fb3",\n    "external_id": "invoice-1593684000",\n    "user_id": "5ed75086a883856178afc12e",\n    "status": "PENDING",\n    "merchant_name": "Xendit&amp;#x27;s Intern",\n    "merchant_profile_picture_url": "https://xnd-companies.s3.amazonaws.com/prod/1591169469152_279.png",\n    "amount": 20000,\n    "payer_email": "customer@domain.com",\n    "description": "Invoice Demo #123",\n    "expiry_date": "2020-07-03T10:00:01.148Z",\n    "invoice_url": "https://invoice-staging.xendit.co/web/invoices/5efdb0210425db620ec35fb3",\n    "available_banks": [\n        {\n            "bank_code": "MANDIRI",\n            "collection_type": "POOL",\n            "bank_account_number": "8860846854335",\n            "transfer_amount": 20000,\n            "bank_branch": "Virtual Account",\n            "account_holder_name": "XENDIT&AMP;#X27;S INTERN",\n            "identity_amount": 0\n        },\n        {\n            "bank_code": "BRI",\n            "collection_type": "POOL",\n            "bank_account_number": "2621554807492",\n            "transfer_amount": 20000,\n            "bank_branch": "Virtual Account",\n            "account_holder_name": "XENDIT&AMP;#X27;S INTERN",\n            "identity_amount": 0\n        },\n        {\n            "bank_code": "BNI",\n            "collection_type": "POOL",\n            "bank_account_number": "880854597383",\n            "transfer_amount": 20000,\n            "bank_branch": "Virtual Account",\n            "account_holder_name": "XENDIT&AMP;#X27;S INTERN",\n            "identity_amount": 0\n        },\n        {\n            "bank_code": "PERMATA",\n            "collection_type": "POOL",\n            "bank_account_number": "821456659745",\n            "transfer_amount": 20000,\n            "bank_branch": "Virtual Account",\n            "account_holder_name": "XENDIT&AMP;#X27;S INTERN",\n            "identity_amount": 0\n        },\n        {\n            "bank_code": "BCA",\n            "collection_type": "POOL",\n            "bank_account_number": "1076619844859",\n            "transfer_amount": 20000,\n            "bank_branch": "Virtual Account",\n            "account_holder_name": "XENDIT&AMP;#X27;S INTERN",\n            "identity_amount": 0\n        }\n    ],\n    "available_retail_outlets": [\n        {\n            "retail_outlet_name": "ALFAMART",\n            "payment_code": "TEST34956",\n            "transfer_amount": 20000\n        }\n    ],\n    "available_ewallets": [],\n    "should_exclude_credit_card": false,\n    "should_send_email": false,\n    "created": "2020-07-02T10:00:01.285Z",\n    "updated": "2020-07-02T10:00:01.285Z",\n    "currency": "IDR"\n}\n```\n\n#### Get Invoice\n\n```python\nfrom xendit import Invoice\n\ninvoice = Invoice.get(\n    invoice_id="5efda8a20425db620ec35f43",\n)\nprint(invoice)\n```\n\nWill return\n\n```\n{\n    "id": "5efda8a20425db620ec35f43",\n    "external_id": "invoice-1593682080",\n    "user_id": "5ed75086a883856178afc12e",\n    "status": "EXPIRED",\n    "merchant_name": "Xendit&amp;#x27;s Intern",\n    "merchant_profile_picture_url": "https://xnd-companies.s3.amazonaws.com/prod/1591169469152_279.png",\n    "amount": 20000,\n    "payer_email": "customer@domain.com",\n    "description": "Invoice Demo #123",\n    "expiry_date": "2020-07-02T09:55:47.794Z",\n    "invoice_url": "https://invoice-staging.xendit.co/web/invoices/5efda8a20425db620ec35f43",\n    "available_banks": [\n        {\n            "bank_code": "MANDIRI",\n            "collection_type": "POOL",\n            "bank_account_number": "8860846853111",\n            "transfer_amount": 20000,\n            "bank_branch": "Virtual Account",\n            "account_holder_name": "XENDIT&AMP;#X27;S INTERN",\n            "identity_amount": 0\n        },\n        {\n            "bank_code": "BRI",\n            "collection_type": "POOL",\n            "bank_account_number": "2621554806292",\n            "transfer_amount": 20000,\n            "bank_branch": "Virtual Account",\n            "account_holder_name": "XENDIT&AMP;#X27;S INTERN",\n            "identity_amount": 0\n        }\n    ],\n    "available_retail_outlets": [\n        {\n            "retail_outlet_name": "ALFAMART",\n            "payment_code": "TEST34950",\n            "transfer_amount": 20000\n        }\n    ],\n    "available_ewallets": [],\n    "should_exclude_credit_card": false,\n    "should_send_email": false,\n    "created": "2020-07-02T09:28:02.191Z",\n    "updated": "2020-07-02T09:55:47.794Z",\n    "currency": "IDR"\n}\n```\n\n#### Expire Invoice\n\n```python\nfrom xendit import Invoice\n\ninvoice = Invoice.expire(\n    invoice_id="5efda8a20425db620ec35f43",\n)\nprint(invoice)\n```\n\nWill return\n\n```\n{\n    "id": "5efda8a20425db620ec35f43",\n    "external_id": "invoice-1593682080",\n    "user_id": "5ed75086a883856178afc12e",\n    "status": "EXPIRED",\n    "merchant_name": "Xendit&amp;#x27;s Intern",\n    "merchant_profile_picture_url": "https://xnd-companies.s3.amazonaws.com/prod/1591169469152_279.png",\n    "amount": 20000,\n    "payer_email": "customer@domain.com",\n    "description": "Invoice Demo #123",\n    "expiry_date": "2020-07-02T09:55:47.794Z",\n    "invoice_url": "https://invoice-staging.xendit.co/web/invoices/5efda8a20425db620ec35f43",\n    "available_banks": [\n        {\n            "bank_code": "MANDIRI",\n            "collection_type": "POOL",\n            "bank_account_number": "8860846853111",\n            "transfer_amount": 20000,\n            "bank_branch": "Virtual Account",\n            "account_holder_name": "XENDIT&AMP;#X27;S INTERN",\n            "identity_amount": 0\n        },\n        {\n            "bank_code": "BRI",\n            "collection_type": "POOL",\n            "bank_account_number": "2621554806292",\n            "transfer_amount": 20000,\n            "bank_branch": "Virtual Account",\n            "account_holder_name": "XENDIT&AMP;#X27;S INTERN",\n            "identity_amount": 0\n        }\n    "available_retail_outlets": [\n        {\n            "retail_outlet_name": "ALFAMART",\n            "payment_code": "TEST34950",\n            "transfer_amount": 20000\n        }\n    ],\n    "available_ewallets": [],\n    "should_exclude_credit_card": false,\n    "should_send_email": false,\n    "created": "2020-07-02T09:28:02.191Z",\n    "updated": "2020-07-02T09:55:47.794Z",\n    "currency": "IDR"\n}\n```\n\n#### List All Invoice\n\n```python\nfrom xendit import Invoice\n\ninvoices = Invoice.list_all(\n    limit=3,\n)\nprint(invoices)\n```\n\nWill return\n\n```\n[\n    ...\n    {\n        "id": "5efda8a20425db620ec35f43",\n        "external_id": "invoice-1593682080",\n        "user_id": "5ed75086a883856178afc12e",\n        "status": "EXPIRED",\n        "merchant_name": "Xendit&amp;#x27;s Intern",\n        "merchant_profile_picture_url": "https://xnd-companies.s3.amazonaws.com/prod/1591169469152_279.png",\n        "amount": 20000,\n        "payer_email": "customer@domain.com",\n        "description": "Invoice Demo #123",\n        "expiry_date": "2020-07-02T09:55:47.794Z",\n        "invoice_url": "https://invoice-staging.xendit.co/web/invoices/5efda8a20425db620ec35f43",\n        "available_banks": [\n            {\n                "bank_code": "MANDIRI",\n                "collection_type": "POOL",\n                "bank_account_number": "8860846853111",\n                "transfer_amount": 20000,\n                "bank_branch": "Virtual Account",\n                "account_holder_name": "XENDIT&AMP;#X27;S INTERN",\n                "identity_amount": 0\n            },\n            {\n                "bank_code": "BRI",\n                "collection_type": "POOL",\n                "bank_account_number": "2621554806292",\n                "transfer_amount": 20000,\n                "bank_branch": "Virtual Account",\n                "account_holder_name": "XENDIT&AMP;#X27;S INTERN",\n                "identity_amount": 0\n            }\n        "available_retail_outlets": [\n            {\n                "retail_outlet_name": "ALFAMART",\n                "payment_code": "TEST34950",\n                "transfer_amount": 20000\n            }\n        ],\n        "available_ewallets": [],\n        "should_exclude_credit_card": false,\n        "should_send_email": false,\n        "created": "2020-07-02T09:28:02.191Z",\n        "updated": "2020-07-02T09:55:47.794Z",\n        "currency": "IDR"\n    }\n    ...\n]\n```\n\n### Recurring Payment Service\n\n#### Create Recurring Payment\n\n```python\nfrom xendit import RecurringPayment\n\nrecurring_payment = RecurringPayment.create(\n    external_id="recurring_12345",\n    payer_email="test@x.co",\n    description="Test Curring Payment",\n    amount=100000,\n    interval="MONTH",\n    interval_count=1,\n)\nprint(recurring_payment)\n```\n\nWill return\n\n```\n{\n    "status": "ACTIVE",\n    "should_send_email": false,\n    "missed_payment_action": "IGNORE",\n    "recurrence_progress": 1,\n    "recharge": true,\n    "user_id": "5ed75086a883856178afc12e",\n    "external_id": "recurring_12345",\n    "payer_email": "test@x.co",\n    "description": "Test Curring Payment",\n    "amount": 100000,\n    "interval": "MONTH",\n    "interval_count": 1,\n    "start_date": "2020-07-08T08:22:55.815Z",\n    "currency": "IDR",\n    "created": "2020-07-08T08:22:55.817Z",\n    "updated": "2020-07-08T08:22:55.994Z",\n    "id": "5f05825ff9f52d3ed204c687",\n    "last_created_invoice_url": "https://invoice-staging.xendit.co/web/invoices/5f05825ff9f52d3ed204c688"\n}\n```\n\n#### Get Recurring Payment\n\n```python\nfrom xendit import RecurringPayment\n\nrecurring_payment = RecurringPayment.get(\n    id="5f05825ff9f52d3ed204c687",\n)\nprint(recurring_payment)\n```\n\nWill return\n\n```\n{\n    "status": "ACTIVE",\n    "should_send_email": false,\n    "missed_payment_action": "IGNORE",\n    "recurrence_progress": 1,\n    "recharge": true,\n    "user_id": "5ed75086a883856178afc12e",\n    "external_id": "recurring_12345",\n    "payer_email": "test@x.co",\n    "description": "Test Curring Payment",\n    "amount": 100000,\n    "interval": "MONTH",\n    "interval_count": 1,\n    "start_date": "2020-07-08T08:22:55.815Z",\n    "currency": "IDR",\n    "created": "2020-07-08T08:22:55.817Z",\n    "updated": "2020-07-08T08:22:55.994Z",\n    "id": "5f05825ff9f52d3ed204c687",\n    "last_created_invoice_url": "https://invoice-staging.xendit.co/web/invoices/5f05825ff9f52d3ed204c688"\n}\n```\n\n#### Edit Recurring Payment\n\n```python\nfrom xendit import RecurringPayment\n\nrecurring_payment = RecurringPayment.edit(\n    id="5f05825ff9f52d3ed204c687",\n    interval_count=2,\n)\nprint(recurring_payment)\n```\n\nWill return\n\n```\n{\n    "status": "ACTIVE",\n    "should_send_email": false,\n    "missed_payment_action": "IGNORE",\n    "recurrence_progress": 1,\n    "recharge": true,\n    "user_id": "5ed75086a883856178afc12e",\n    "external_id": "recurring_12345",\n    "payer_email": "test@x.co",\n    "description": "Test Curring Payment",\n    "amount": 100000,\n    "interval": "MONTH",\n    "interval_count": 2,\n    "start_date": "2020-07-08T08:22:55.815Z",\n    "currency": "IDR",\n    "created": "2020-07-08T08:22:55.817Z",\n    "updated": "2020-07-08T08:24:58.295Z",\n    "id": "5f05825ff9f52d3ed204c687"\n}\n```\n\n#### Stop Recurring Payment\n\n```python\nfrom xendit import RecurringPayment\n\nrecurring_payment = RecurringPayment.stop(\n    id="5f05825ff9f52d3ed204c687",\n)\nprint(recurring_payment)\n```\n\nWill return\n\n```\n{\n    "status": "STOPPED",\n    "should_send_email": false,\n    "missed_payment_action": "IGNORE",\n    "recurrence_progress": 1,\n    "recharge": true,\n    "user_id": "5ed75086a883856178afc12e",\n    "external_id": "recurring_12345",\n    "payer_email": "test@x.co",\n    "description": "Test Curring Payment",\n    "amount": 100000,\n    "interval": "MONTH",\n    "interval_count": 2,\n    "start_date": "2020-07-08T08:22:55.815Z",\n    "currency": "IDR",\n    "created": "2020-07-08T08:22:55.817Z",\n    "updated": "2020-07-08T08:26:32.464Z",\n    "id": "5f05825ff9f52d3ed204c687"\n}\n```\n\n#### Pause Recurring Payment\n\n```python\nfrom xendit import RecurringPayment\n\nrecurring_payment = RecurringPayment.pause(\n    id="5f05825ff9f52d3ed204c687",\n)\nprint(recurring_payment)\n```\n\nWill return\n\n```\n{\n    "status": "PAUSED",\n    "should_send_email": false,\n    "missed_payment_action": "IGNORE",\n    "recurrence_progress": 1,\n    "recharge": true,\n    "user_id": "5ed75086a883856178afc12e",\n    "external_id": "recurring_12345",\n    "payer_email": "test@x.co",\n    "description": "Test Curring Payment",\n    "amount": 100000,\n    "interval": "MONTH",\n    "interval_count": 2,\n    "start_date": "2020-07-08T08:22:55.815Z",\n    "currency": "IDR",\n    "created": "2020-07-08T08:22:55.817Z",\n    "updated": "2020-07-08T08:25:44.580Z",\n    "id": "5f05825ff9f52d3ed204c687"\n}\n```\n\n#### Resume Recurring Payment\n\n```python\nfrom xendit import RecurringPayment\n\nrecurring_payment = RecurringPayment.resume(\n    id="5f05825ff9f52d3ed204c687",\n)\nprint(recurring_payment)\n```\n\nWill return\n\n```\n{\n    "status": "ACTIVE",\n    "should_send_email": false,\n    "missed_payment_action": "IGNORE",\n    "recurrence_progress": 1,\n    "recharge": true,\n    "user_id": "5ed75086a883856178afc12e",\n    "external_id": "recurring_12345",\n    "payer_email": "test@x.co",\n    "description": "Test Curring Payment",\n    "amount": 100000,\n    "interval": "MONTH",\n    "interval_count": 2,\n    "start_date": "2020-07-08T08:22:55.815Z",\n    "currency": "IDR",\n    "created": "2020-07-08T08:22:55.817Z",\n    "updated": "2020-07-08T08:26:03.082Z",\n    "id": "5f05825ff9f52d3ed204c687"\n}\n```\n\n### Payout Service\n\n#### Create Payout\n\n```python\nfrom xendit import Payout\n\npayout = Payout.create(\n    external_id="payout-1595405117",\n    amount=50000,\n    email="test@email.co",\n)\nprint(payout)\n```\n\nWill return\n\n```\n{\n    "id": "a6ee1bf1-ffcd-4bda-a7ab-99c1d5cd0472",\n    "external_id": "payout-1595405117",\n    "amount": 50000,\n    "merchant_name": "Xendit&amp;#x27;s Intern",\n    "status": "PENDING",\n    "expiration_timestamp": "2020-07-23T08:05:19.815Z",\n    "created": "2020-07-22T08:05:18.421Z",\n    "email": "test@email.co",\n    "payout_url": "https://payout-staging.xendit.co/web/a6ee1bf1-ffcd-4bda-a7ab-99c1d5cd0472"\n}\n```\n\n#### Get Payout\n\n```python\nfrom xendit import Payout\n\npayout = Payout.get(\n    id="a6ee1bf1-ffcd-4bda-a7ab-99c1d5cd0472",\n)\nprint(payout)\n```\n\nWill return\n\n```\n{\n    "id": "a6ee1bf1-ffcd-4bda-a7ab-99c1d5cd0472",\n    "external_id": "payout-1595405117",\n    "amount": 50000,\n    "merchant_name": "Xendit&amp;#x27;s Intern",\n    "status": "PENDING",\n    "expiration_timestamp": "2020-07-23T08:05:19.815Z",\n    "created": "2020-07-22T08:05:18.421Z",\n    "email": "test@email.co",\n    "payout_url": "https://payout-staging.xendit.co/web/a6ee1bf1-ffcd-4bda-a7ab-99c1d5cd0472"\n}\n```\n\n#### Void a Payout\n\n```python\nfrom xendit import Payout\n\npayout = Payout.void(\n    id="a6ee1bf1-ffcd-4bda-a7ab-99c1d5cd0472",\n)\nprint(payout)\n```\n\nWill return\n\n```\n{\n    "id": "a6ee1bf1-ffcd-4bda-a7ab-99c1d5cd0472",\n    "external_id": "payout-1595405117",\n    "amount": 50000,\n    "merchant_name": "Xendit&amp;#x27;s Intern",\n    "status": "VOIDED",\n    "expiration_timestamp": "2020-07-23T08:05:19.815Z",\n    "created": "2020-07-22T08:05:18.421Z",\n    "email": "test@email.co"\n}\n```\n\n### Disbursement Service\n\n#### Create Disbursement\n\n```python\nfrom xendit import Disbursement\n\ndisbursement = Disbursement.create(\n    external_id="demo_1475459775872",\n    bank_code="BCA",\n    account_holder_name="Bob Jones",\n    account_number="1231242311",\n    description="Reimbursement for shoes",\n    amount=17000,\n)\nprint(disbursement)\n```\n\nWill return\n\n```\n{\n    "user_id": "5ed75086a883856178afc12e",\n    "external_id": "demo_1475459775872",\n    "amount": 17000,\n    "bank_code": "BCA",\n    "account_holder_name": "Bob Jones",\n    "disbursement_description": "Reimbursement for shoes",\n    "status": "PENDING",\n    "id": "5ef1c4f40c2e150017ce3b96",\n}\n```\n\n#### Get Disbursement by ID\n\n```python\nfrom xendit import Disbursement\n\ndisbursement = Disbursement.get(\n    id="5ef1befeecb16100179e1d05",\n)\nprint(disbursement)\n```\n\nWill return\n\n```\n{\n    "user_id": "5ed75086a883856178afc12e",\n    "external_id": "demo_1475459775872",\n    "amount": 17000,\n    "bank_code": "BCA",\n    "account_holder_name": "Bob Jones",\n    "disbursement_description": "Disbursement from Postman",\n    "status": "PENDING",\n    "id": "5ef1befeecb16100179e1d05"\n}\n```\n\n#### Get Disbursement by External ID\n\n```python\nfrom xendit import Disbursement\n\ndisbursement_list = Disbursement.get_by_ext_id(\n    external_id="demo_1475459775872",\n)\nprint(disbursement_list)\n\n```\n\nWill return\n\n```\n[\n    {\n        "user_id": "5ed75086a883856178afc12e",\n        "external_id": "demo_1475459775872",\n        "amount": 17000,\n        "bank_code": "BCA",\n        "account_holder_name": "Bob Jones",\n        "disbursement_description": "Reimbursement for shoes",\n        "status": "PENDING",\n        "id": "5ef1c4f40c2e150017ce3b96",\n    },\n    {\n        "user_id": "5ed75086a883856178afc12e",\n        "external_id": "demo_1475459775872",\n        "amount": 17000,\n        "bank_code": "BCA",\n        "account_holder_name": "Bob Jones",\n        "disbursement_description": "Disbursement from Postman",\n        "status": "PENDING",\n        "id": "5ef1befeecb16100179e1d05",\n    },\n    ...\n]\n```\n#### Get Available Banks\n\n```python\nfrom xendit import Disbursement\n\ndisbursement_banks = Disbursement.get_available_banks()\nprint(disbursement_banks)\n```\n\nWill return\n\n```\n[\n    ...\n    {\n        "name": "Mandiri Taspen Pos (formerly Bank Sinar Harapan Bali)",\n        "code": "MANDIRI_TASPEN",\n        "can_disburse": True,\n        "can_name_validate": True,\n    },\n    {\n        "name": "Bank QNB Indonesia (formerly Bank QNB Kesawan)",\n        "code": "QNB_INDONESIA",\n        "can_disburse": True,\n        "can_name_validate": True,\n    }\n]\n```\n\n### Batch Disbursement Service\n\n#### Create Batch Disbursement\n\n```python\nfrom xendit import BatchDisbursement\n\nbatch_disbursement_items = []\nbatch_disbursement_items.append(\n    BatchDisbursement.helper_create_batch_item(\n        amount=10000,\n        bank_code="BCA",\n        bank_account_name="Adyaksa W",\n        bank_account_number="12345678",\n        description="Sample Batch Disbursement",\n        external_id=f"batch-disbursement-item-12345"\n    )\n)\nbatch_disbursement = BatchDisbursement.create(\n    reference="batch_disbursement-1595326225",\n    disbursements=batch_disbursement_items,\n)\nprint(batch_disbursement)\n```\n\nWill return\n\n```\n{\n    "status": "UPLOADING",\n    "reference": "batch_disbursement-1595326225",\n    "total_uploaded_amount": 10000,\n    "total_uploaded_count": 1,\n    "created": "2020-07-21T10:10:35.782Z",\n    "id": "5f16bf1bfc70de0017b858bf"\n}\n```\n\n#### Get Batch Disbursement Available Banks\n\nYou can use [Get Available Banks](#get-available-banks) to use this feature.\n\n### xenPlatform Service\n\n#### Create Account\n\n```python\nfrom xendit import XenPlatform, XenPlatformAccountType\n\nxenplatform_account = XenPlatform.create_account(\n    account_email="test-xenplatform@pythonxendit.co",\n    type=XenPlatformAccountType.OWNED,\n    business_profile={\'business_name\': \'python-xendit\'},\n)\nprint(xenplatform_account)\n```\n\nWill return\n\n```\n{\n    "account_email": "test-xenplatform@pythonxendit.co",\n    "user_id": "5f2132ed172cd67992c402d6",\n    "created": "2020-07-29T08:27:25.346Z",\n    "status": "SUCCESSFUL",\n    "type": "OWNED"\n}\n```\n\n#### Set Callback URLs\n\n```python\nfrom xendit import XenPlatform, XenPlatformURLType\n\nxenplatform_callback_url = XenPlatform.set_callback_url(\n    type=XenPlatformURLType.INVOICE,\n    url="https://test-url-invoice.com",\n)\nprint(xenplatform_callback_url)\n```\n\nWill return\n\n```\n{\n    "status": "SUCCESSFUL",\n    "user_id": "5e6b30d967627b957de8c123",\n    "url": "https://test-url-invoice.com",\n    "environment": "TEST",\n    "callback_token": "66a6680348e1c33ed2b9053a8eb9291b9e2230ff4f4d3057c9f4ac26405d2123"\n}\n```\n\n#### Transfers\n\n```python\nfrom xendit import XenPlatform\n\nxenplatform_transfers = XenPlatform.transfers(\n    reference="transfer001",\n    amount=10000,\n    source_user_id="54afeb170a2b18519b1b8768",\n    destination_user_id="5cafeb170a2b1851246b8768",\n)\nprint(xenplatform_transfers)\n```\n\nWill return\n\n```\n{\n    "created": "2020-01-01T08:51:44.484Z",\n    "transfer_id": "60b9d810-d9a3-456c-abbf-2786ec7a9651",\n    "reference": "transfer001",\n    "source_user_id": "54afeb170a2b18519b1b8768",\n    "destination_user_id": "5cafeb170a2b1851246b8768",\n    "status": "SUCCESSFUL",\n    "amount": 10000\n}\n```\n\n## Contributing\n\nFor any requests, bugs, or comments, please open an [issue](https://github.com/xendit/xendit-python/issues) or [submit a pull request](https://github.com/xendit/xendit-python/pulls).\n\nTo start developing on this repository, you need to have Poetry installed for package dependency. After that, you can run ```poetry install``` to install the dependency. To enter the environment, run ```poetry shell```\n\n### Tests\n\n#### Running the Test\n\nMake sure the the code passes all tests.\n\nRun the test:\n\n```\npython -m pytest tests/\n```\n\nRun with coverage:\n\n```\npython -m pytest tests/ --cov=xendit/\n```',
    'author': 'Maahir Ur Rahman',
    'author_email': 'maahir@xendit.co',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/xendit/xendit-python',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
