from typing import Optional, Callable, Union
from ..models.models import AdminActionResult, GroupActionResult, ExtendedGroup
from ..types.types import AdminActionTypes, GroupType


class Actions:
    def __init__(self, client):
        self.client = client
        self.admin_action_result = AdminActionResult()
        self.group_action_result = GroupActionResult()

    async def _get_extended_group(
        self,
        group: Union[str, int]
    ) -> Optional[ExtendedGroup]:
        result = None

        def callback(data):
            nonlocal result
            result = data

        self.client.info.request_group(group, callback)
        return result

    def ban_user(
        self,
        group: int,
        user_id: int,
        callback: Optional[Callable[[AdminActionResult], None]] = None
    ):
        self.admin_action_result = AdminActionResult()

        def on_response(data):
            if isinstance(data, dict) and data.get('code') == 200:
                self.admin_action_result.success = True
            else:
                self.admin_action_result.unknown_reason = True

            if callback:
                callback(self.admin_action_result)

        self.client.write_packet(
            self.client.packet.admin_action(group, user_id, AdminActionTypes.BAN),
            adv=True,
            success=on_response
        )

    def kick_user(
        self,
        group: int,
        user_id: int,
        callback: Optional[Callable[[AdminActionResult], None]] = None
    ):
        self.admin_action_result = AdminActionResult()

        def on_response(data):
            if isinstance(data, dict) and data.get('code') == 200:
                self.admin_action_result.success = True
            else:
                self.admin_action_result.unknown_reason = True

            if callback:
                callback(self.admin_action_result)

        self.client.write_packet(
            self.client.packet.admin_action(group, user_id, AdminActionTypes.KICK),
            adv=True,
            success=on_response
        )

    def silence_user(
        self,
        group: int,
        user_id: int,
        callback: Optional[Callable[[AdminActionResult], None]] = None
    ):
        self.admin_action_result = AdminActionResult()

        def on_response(data):
            if isinstance(data, dict) and data.get('code') == 200:
                self.admin_action_result.success = True
            else:
                self.admin_action_result.unknown_reason = True

            if callback:
                callback(self.admin_action_result)

        self.client.write_packet(
            self.client.packet.admin_action(group, user_id, AdminActionTypes.SILENCE),
            adv=True,
            success=on_response
        )

    def reset_user(
        self,
        group: int,
        user_id: int,
        callback: Optional[Callable[[AdminActionResult], None]] = None
    ):
        self.admin_action_result = AdminActionResult()

        def on_response(data):
            if isinstance(data, dict) and data.get('code') == 200:
                self.admin_action_result.success = True
            else:
                self.admin_action_result.unknown_reason = True

            if callback:
                callback(self.admin_action_result)

        self.client.write_packet(
            self.client.packet.admin_action(group, user_id, AdminActionTypes.RESET),
            adv=True,
            success=on_response
        )

    def mod_user(
        self,
        group: int,
        user_id: int,
        callback: Optional[Callable[[AdminActionResult], None]] = None
    ):
        self.admin_action_result = AdminActionResult()

        def on_response(data):
            if isinstance(data, dict) and data.get('code') == 200:
                self.admin_action_result.success = True
            else:
                self.admin_action_result.unknown_reason = True

            if callback:
                callback(self.admin_action_result)

        self.client.write_packet(
            self.client.packet.admin_action(group, user_id, AdminActionTypes.MOD),
            adv=True,
            success=on_response
        )

    def admin_user(
        self,
        group: int,
        user_id: int,
        callback: Optional[Callable[[AdminActionResult], None]] = None
    ):
        self.admin_action_result = AdminActionResult()

        def on_response(data):
            if isinstance(data, dict) and data.get('code') == 200:
                self.admin_action_result.success = True
            else:
                self.admin_action_result.unknown_reason = True

            if callback:
                callback(self.admin_action_result)

        self.client.write_packet(
            self.client.packet.admin_action(group, user_id, AdminActionTypes.ADMIN),
            adv=True,
            success=on_response
        )

    def join_group(
        self,
        group_name: str,
        password: Optional[str] = None,
        callback: Optional[Callable[[GroupActionResult], None]] = None
    ):
        self.group_action_result = GroupActionResult()

        def on_response(data):
            if isinstance(data, dict):
                code = data.get('code', 0)
                if code == 200:
                    self.group_action_result.success = True
                elif code == 403:
                    self.group_action_result.banned = True
                elif code == 404:
                    self.group_action_result.group_not_found = True
                else:
                    self.group_action_result.unknown_error = True

            if callback:
                callback(self.group_action_result)

        self.client.write_packet(
            self.client.packet.group_join(group_name, password),
            adv=True,
            success=on_response
        )

    def leave_group(
        self,
        group_id: int,
        callback: Optional[Callable[[GroupActionResult], None]] = None
    ):
        self.group_action_result = GroupActionResult()

        def on_response(data):
            if isinstance(data, dict) and data.get('code') == 200:
                self.group_action_result.success = True
            else:
                self.group_action_result.unknown_error = True

            if callback:
                callback(self.group_action_result)

        self.client.write_packet(
            self.client.packet.group_leave(group_id),
            adv=True,
            success=on_response
        )
