import socketio
import secrets
import string
from typing import Optional, Callable, Any, Dict, List
from ..types.types import DeviceType, OnlineState
from ..models.models import ExtendedClient, ExtendedUser, ExtendedGroup, ExtendedMessage, Message
from .packets import Packets
from .packet import Packet
from .delegates import Delegates


def generate_token(length: int = 16) -> str:
    chars = string.ascii_letters + string.digits
    return ''.join(secrets.choice(chars) for _ in range(length))


class Client:
    def __init__(self, url: Optional[str] = None):
        self.server = url or 'https://v3.palringo.com:3051'
        self.connection: Optional[socketio.Client] = None
        self.packet = Packets()
        self.on = Delegates()
        self.info = None
        self.actions = None
        self.messaging = None
        self.stages = None
        self._id: Optional[int] = None
        self._email: Optional[str] = None
        self._password: Optional[str] = None
        self._device_type: Optional[DeviceType] = None

    def logout(self):
        self.write_packet(self.packet.logout(), adv=True)
        if self.connection:
            self.connection.disconnect()

    def login(
        self,
        username: str,
        password: str,
        device: DeviceType = DeviceType.WEB,
        state: OnlineState = OnlineState.ONLINE,
        type_: str = 'email'
    ):
        if not device:
            device = DeviceType.WEB
        if not state:
            state = OnlineState.ONLINE

        device_type_array = [
            'unknown', 'bot', 'pc', 'genericmobile', 'mac',
            'iphone', 'ipad', 'android', 'web', 'windowsphone7'
        ]

        token = f"WE{generate_token()}"
        url = f"{self.server}?token={token}&device={device_type_array[device]}"

        self.connection = socketio.Client(
            reconnection=False,
            engineio_logger=False,
            logger=False
        )

        @self.connection.on('welcome')
        def on_welcome(data):
            self.write_packet(
                self.packet.login_packet(username, password, type_, device),
                adv=False,
                success=self._on_login_response
            )

        @self.connection.on('connect')
        def on_connect():
            self.on.trigger('cn')

        @self.connection.on('disconnect')
        def on_disconnect():
            self.on.trigger('dc')

        @self.connection.on('message send')
        def on_message(data):
            if 'body' in data:
                body = data['body']
                if body.get('originator') == self._id:
                    return

                msg = ExtendedMessage(body)
                user_id = msg.originator

                if msg.is_group:
                    self.on.trigger('gm', msg)
                else:
                    self.on.trigger('pm', msg)

        @self.connection.on('group admin')
        def on_group_admin(data):
            if 'body' in data:
                body = data['body']
                action_type = body.get('type')
                if action_type in ['join', 'leave']:
                    self.on.trigger('gu', body)
                else:
                    self.on.trigger('aa', body)

        @self.connection.on('subscriber state update')
        def on_subscriber_update(data):
            if 'body' in data:
                self.on.trigger('gu', data['body'])

        try:
            self.connection.connect(url, transports=['websocket'])
        except Exception as e:
            if self.on._log:
                self.on.trigger('log', f"Connection error: {e}")

        self._email = username
        self._password = password
        self._device_type = device

    def _on_login_response(self, data: Dict):
        if data.get('code') != 200:
            self.on.trigger('lf', data)
            return

        body = data.get('body', {})
        self._id = body.get('_id') or body.get('subscriber', {}).get('id')

        from ..information.information import Information
        from ..actions.actions import Actions
        from ..communication.messaging import Messaging

        client_data = ExtendedClient()
        if 'subscriber' in body:
            client_data.subscriber = body['subscriber']
        if 'extended' in body:
            client_data.extended = body['extended']

        self.info = Information(self, client_data)
        self.actions = Actions(self)
        self.messaging = Messaging(self)

        self.on.trigger('ls', client_data)

        self.write_packet(self.packet.subscribe_to_pm(), adv=False)

        def on_groups(groups: List[ExtendedGroup]):
            if groups:
                group_ids = [g.id for g in groups if g.id]
                if group_ids:
                    self.write_packet(
                        self.packet.subscribe_to_groups(group_ids),
                        adv=False
                    )

        if self.info:
            self.info.request_groups(on_groups)

        self.update_state(OnlineState.ONLINE)

    def update_state(self, state: OnlineState):
        self.write_packet(self.packet.update_user_state(state))

    def write_packet(
        self,
        packet: Packet,
        adv: bool = False,
        debug: bool = False,
        success: Optional[Callable] = None,
        failed: Optional[Callable] = None
    ):
        if not self.connection or not self.connection.connected:
            if self.on._log:
                self.on.trigger('log', 'Not connected to server')
            return

        payload = {}
        if packet.body:
            payload['body'] = packet.body
        if packet.headers:
            payload['headers'] = packet.headers

        try:
            if not adv:
                if success:
                    self.connection.emit(packet.command, payload, callback=success)
                else:
                    self.connection.emit(packet.command, payload)
            else:
                def callback(data):
                    if debug and self.on._log:
                        self.on.trigger('log', f'Packet: {packet.command}, Response: {data}')

                    if isinstance(data, dict):
                        code = data.get('code', 0)
                        body = data.get('body')

                        if code == 200 and success:
                            success(body)
                        elif code != 200 and failed:
                            failed(data)

                self.connection.emit(packet.command, payload, callback=callback)
        except Exception as e:
            if self.on._log:
                self.on.trigger('log', f'Packet error: {e}')
