import hashlib
from typing import Optional, List
from .packet import Packet
from ..types.types import (
    AdminActionTypes,
    Language,
    DeviceType,
    OnlineState,
    Gender,
    RelationshipStatus,
    LookingFor,
)
from ..models.models import ExtendedUser, CGroup, SelectedList


class Packets:
    def logout(self):
        return Packet('security logout')

    def login_packet(
        self,
        username: str,
        password: str,
        type_: str,
        device: DeviceType
    ):
        if type_ == 'email':
            password_hash = hashlib.md5(password.encode()).hexdigest()
            return Packet(
                'security login',
                {
                    'deviceTypeId': device,
                    'type': 'email',
                    'username': username,
                    'password': password_hash,
                    'md5Password': True
                },
                {'version': 2}
            )

    def subscribe_to_pm(self):
        return Packet('message private subscribe', {})

    def subscribe_to_groups(self, id_list: List[int]):
        return Packet('message group subscribe', {'idList': id_list})

    def group_list(self):
        return Packet('group list')

    def group_profile(self, search: any, extended: bool = True):
        return Packet(
            'group profile',
            {
                'id': search if isinstance(search, int) else None,
                'name': search if isinstance(search, str) else None,
                'extended': extended
            },
            {'version': 3}
        )

    def request_group_statistics(self, group_id: int):
        return Packet('group stats', {'id': group_id})

    def create_group(self, group: CGroup):
        return Packet('group create', group.__dict__)

    def update_group_profile(self, group: CGroup):
        return Packet('group profile update', group.__dict__)

    def admin_action(self, group: int, target: int, action: AdminActionTypes):
        return Packet(
            'group admin',
            {
                'groupId': group,
                'id': target,
                'capabilities': action
            }
        )

    def group_member_list(self, id_: int):
        return Packet(
            'group member list',
            {'id': id_, 'subscribe': True},
            {'version': 2}
        )

    def group_join(self, group_name: str, password: Optional[str] = None):
        return Packet(
            'group join',
            {'name': group_name, 'password': password}
        )

    def group_leave(self, group_id: int):
        return Packet('group leave', {'id': group_id})

    def user_profile(self, user_id: int, extended_profile: bool = True):
        return Packet(
            'subscriber profile',
            {'id': user_id, 'extended': extended_profile}
        )

    def request_achievements(self, language: Language = Language.ENGLISH):
        return Packet('achievement list', {'language': language})

    def request_user_achievements(self, user_id: int):
        return Packet('achievement subscriber list', {'id': user_id})

    def update_user_state(self, state: OnlineState):
        return Packet(
            'subscriber settings update',
            {'state': {'state': state}}
        )

    def update_user_profile(self, profile: ExtendedUser):
        return Packet('subscriber profile update', profile.__dict__)

    def request_user_charm_stats(self, user_id: int):
        return Packet('charm subscriber statistics', {'id': user_id})

    def request_user_active_charms(
        self,
        user_id: int,
        offset: int = 0,
        limit: int = 25
    ):
        return Packet(
            'charm subscriber active list',
            {'id': user_id, 'offset': offset, 'limit': limit}
        )

    def request_user_expired_charms(
        self,
        user_id: int,
        offset: int = 0,
        limit: int = 25
    ):
        return Packet(
            'charm subscriber expired list',
            {'id': user_id, 'offset': offset, 'limit': limit}
        )

    def user_add(self, user_id: int, message: str = ''):
        return Packet(
            'subscriber contact add',
            {'id': user_id, 'message': message}
        )

    def user_delete(self, user_id: int):
        return Packet('subscriber contact delete', {'id': user_id})

    def user_block(self, user_id: int):
        return Packet('subscriber contact block', {'id': user_id})

    def request_topics(
        self,
        language: Language = Language.ENGLISH,
        name: str = 'default'
    ):
        return Packet(
            'topic file',
            {'name': name, 'languageId': language}
        )

    def request_discover_groups(
        self,
        language: Language = Language.ENGLISH,
        max_results: int = 25,
        offset: int = 0,
        admin: bool = False,
        rep_level: Optional[int] = None,
        recipe_id: Optional[int] = None
    ):
        return Packet(
            'group discovery list',
            {
                'maxResults': max_results,
                'offset': offset,
                'admin': admin,
                'reputationLevelOverride': rep_level,
                'language': language,
                'recipieId': recipe_id
            }
        )

    def notifications(
        self,
        language: Language = Language.ENGLISH,
        device_type: DeviceType = DeviceType.WEB
    ):
        return Packet(
            'notification list',
            {'language': language, 'deviceType': device_type}
        )

    def clear_notifications(self):
        return Packet('notification list clear')

    def conversation_list(self):
        return Packet('message history conversations')

    def message_history(
        self,
        id_: int,
        timestamp: str,
        is_group: bool = True,
        before: bool = True
    ):
        return Packet(
            'message history',
            {
                'id': id_,
                'timestampEnd': timestamp if before else None,
                'timestampBegin': None if before else timestamp,
                'isGroup': is_group
            }
        )

    def group_message(self, recipient: int, data: bytes, mime_type: str):
        return Packet(
            'message send',
            {
                'recipient': recipient,
                'isGroup': True,
                'mimeType': mime_type,
                'data': data
            }
        )

    def private_message(self, recipient: int, data: bytes, mime_type: str):
        return Packet(
            'message send',
            {
                'recipient': recipient,
                'isGroup': False,
                'mimeType': mime_type,
                'data': data
            }
        )
