# xwolfpython

An unofficial Python API for WOLF instant messaging platform. This is a Python port of the TypeScript [WOLFApi-TS](https://github.com/masamesa/WOLFApi-TS) library.

## Installation

```bash
pip install xwolfpython
```

## Features

- Connect to WOLF messaging platform
- Send and receive messages (group and private)
- Manage group memberships
- Admin actions (ban, kick, silence, etc.)
- User profile management
- Group management
- Charms and achievements
- And much more!

## Quick Start

```python
from xwolfpython import WolfClient
from xwolfpython.types.types import DeviceType, OnlineState

bot = WolfClient()

@bot.On.LoginSuccess.setter
def on_login_success(user):
    print(f"Logged in as: {user.subscriber.get('nickname')}")

@bot.On.GroupMessage.setter
def on_group_message(msg):
    print(f"Group message: {msg.text}")

    # Reply to message
    bot.Messaging.reply(msg, "Hello from Python!")

bot.login("your_email", "your_password", DeviceType.WEB, OnlineState.ONLINE)

# Keep the bot running
import time
try:
    while True:
        time.sleep(1)
except KeyboardInterrupt:
    bot.logout()
```

## Examples

### Sending Messages

```python
# Send group message
bot.Messaging.group_message(group_id, "Hello group!")

# Send private message
bot.Messaging.private_message(user_id, "Hello user!")

# Send image
bot.Messaging.group_message(group_id, "https://example.com/image.jpg", is_image=True)
```

### Admin Actions

```python
# Ban user
bot.Action.ban_user(group_id, user_id, lambda result: print(result.success))

# Kick user
bot.Action.kick_user(group_id, user_id)

# Silence user
bot.Action.silence_user(group_id, user_id)

# Reset user
bot.Action.reset_user(group_id, user_id)
```

### Group Management

```python
# Join group
bot.Action.join_group("group_name")

# Leave group
bot.Action.leave_group(group_id)

# Get group info
bot.Info.request_group(group_id, lambda group: print(group.name))

# Get group members
bot.Info.group_member_list(group_id, lambda members: print(len(members)))
```

### User Information

```python
# Get user profile
bot.Info.request_user(user_id, lambda user: print(user.nickname))

# Update bot status
bot.update_state(OnlineState.BUSY)

# Update bot name
bot.Info.update_name("New Name")
```

## Event Handlers

Available event handlers:

- `LoginSuccess`: Called when login is successful
- `LoginFailed`: Called when login fails
- `Connected`: Called when connected to server
- `Disconnected`: Called when disconnected from server
- `GroupMessage`: Called when a group message is received
- `PrivateMessage`: Called when a private message is received
- `AdminAction`: Called when an admin action occurs
- `UserUpdate`: Called when a user update occurs
- `Log`: Called for logging information

## Requirements

- Python 3.7+
- python-socketio[client] >= 5.0.0
- requests >= 2.25.0

## License

MIT License

## Credits

This library is a Python port of [WOLFApi-TS](https://github.com/masamesa/WOLFApi-TS) by masamesa.

## Disclaimer

This is an unofficial API and is not affiliated with WOLF or Palringo. Use at your own risk.
